;+
; Project     : SOHO - LASCO/EIT
;                   
; Name        : DEFINE_OS
;               
; Purpose     : Widget tool to define a LASCO/EIT Observing Sequence.
;               
; Explanation : This tool allows the user to specify a specific LASCO/EIT
;               Observing Sequence (OS) consisting of a LEB Program (LP),
;		telescope setup, camera setup, image processing steps, etc.
; 		The tool then calculates the estimated time to process the
;		image(s) and the processed image size.  The LASCO/EIT 
;		Scheduling tool can then be called from this tool.
;               
; Use         : Generally this tool will be called from the LASCO/EIT
;		Scheduling tool.  But it can be called from the command
;		line without any params:	IDL> define_os
;    
; Inputs      : None.
;               
; Opt. Inputs : None.
;               
; Outputs     : None.
;               
; Opt. Outputs: None.
;               
; Keywords    : None.
;
; Calls       : OS_INIT, OS_UTIL, OS_TABLES, OS_LP, MASK, OS_STATS
;
; Common      : OS_SHARE is defined in DEFINE_OS.
;		OS_INIT_SHARE is defined in OS_INIT.
;               
; Restrictions: None.
;               
; Side effects: None.
;               
; Category    : Planning, Scheduling.
;               
; Prev. Hist. : None.
;
; Written     : Scott Paswaters, NRL, November 1994.
;               
; Modified    : 
;
; Version     : Version 0.1, December 5, 1994
;-            

;__________________________________________________________________________________________________________
;

PRO DEFINE_OS_EVENT, event

COMMON OS_SHARE, osv

CASE (event.id) OF

   osv.quit : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, osv.base
   END

   osv.doorb : BEGIN	;** mark door open/closed for stats
      IF (osv.door EQ 0) THEN osv.door = 1 ELSE osv.door = 0
   END

   osv.statsb : BEGIN	;** calculate statistics
      OS_STATS, ptime, psize, obs_mode_id
   END

   osv.scienceb : BEGIN	;** call science tool
      OS_CALL_SCIENCE
   END

   osv.bits_bytes : BEGIN	;** convert size units
      WIDGET_CONTROL, osv.size_text, GET_VALUE=os_size
      from = osv.size_units
      to = event.index   ;** 0:bits, 1:bytes, 2:% buffer
      osv.size_units = to
      new_value = SCHEDULE_CONVERT_UNITS(DOUBLE(os_size), from, to)
      WIDGET_CONTROL, osv.size_text, SET_VALUE=STRING(new_value,FORMAT='(d10.1)')
   END

   ELSE : BEGIN
   END

ENDCASE

END

;__________________________________________________________________________________________________________
;

PRO DEFINE_OS

COMMON OS_SHARE
COMMON OS_INIT_SHARE, op_types, tele_types, table_types, fw_types, pw_types, exit_types, proc_tab_types, $
   ip_arr, plan_lp, lprow

    IF XRegistered("DEFINE_OS") THEN RETURN

    OS_INIT		        ;** initialize some variables
    OS_UTIL			;** compile utilities ahead of time
    OS_TABLES			;** compile table upload procedures ahead of time
    OS_LP			;** compile observation procedures ahead of time

    IF ((SIZE(sched_list))(1) EQ 0) THEN sched_list = ''	;** not defined yet use default
    utable_types =["  Image Processing Table  ", $
                  " Camera Parameters Table  ", $
                  " Exposure Durations Table ", $
                  "   Occulter Mask Table    ", $
                  "      ROI Mask Table      ", $
                  "     Bad Column Table     ", $
                  "     Transient Table      ", $
                  "   FP WL Exposure Table   "]
    table_pro = ["TABLE_PROCESSING", $
                 "TABLE_CAMERA", $
                 "TABLE_EXPOSURE", $
                 "MASK", $
                 "MASK", $
                 "TABLE_BADCOL", $
                 "TABLE_TRANSIENT",$
                 "TABLE_FP_WL_EXP"]

    nop = N_ELEMENTS(op_types)
    op_buttons = LONARR(nop)
    op_pro = ["LP_NORMAL_IMAGE",	$
              "LP_DARK_IMAGE",		$
              "LP_CAL_LAMP",		$
              "LP_SEQ_PW_FW",		$
              "LP_CONT_IMAGE",		$
              "LP_TAKE_SEQ",		$
              "LP_CONCURRENT",		$
              "LP_SUM_IMG",		$
              "LP_MOVE_M1",		$
              "LP_M1_MEASURE",		$
              "LP_WOBBLE_IN",		$
              "LP_WOBBLE_OUT",		$
              "LP_CHK_CORR",		$
              "LP_FP_SCAN_LINE",	$
              "LP_FP_CAMCOORD",		$
              "LP_GRND_PERIPH",		$
              "LP_PERIPH_LOAD",		$
              "LP_UP_DFLT_PRM",		$
              "LP_GRND_MECH",		$
              "LP_TRANS_DET",		$
              "LP_DUMP_MEMORY",		$
              "LP_STATUS_REQ",		$
              "LP_CMD_TABLE",		$
              "LP_NULL"]
    op_sens = BYTARR(N_ELEMENTS(op_pro))
    good = [0,1,2,3,7,10,13,15,22]
    op_sens(good) = 1
    door = 0	;** closed


    ;********************************************************************
    ;** SET UP WIDGETS **************************************************

    sched_base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT Observing Sequence Definition Tool', /FRAME)

      col0 = WIDGET_BASE(sched_base, /COLUMN, /FRAME)
      arow = WIDGET_BASE(col0, /ROW)
      statsb = WIDGET_BUTTON(arow, VALUE='                               CALCULATE STATISTICS                                ')
      arow2 = WIDGET_BASE(arow, /ROW, /NONEXCLUSIVE)
      doorb = WIDGET_BUTTON(arow2,  VALUE='DOOR CLOSED')
      toprow = WIDGET_BASE(col0, /ROW)
        temp = WIDGET_LABEL(toprow, VALUE=' Time to Process:')
        new_value = 0D
        time_text = WIDGET_LABEL(toprow, VALUE=STRING(new_value,FORMAT='(i5)')+' secs')

        temp = WIDGET_LABEL(toprow, VALUE='    Processed Size:')
        size_text = WIDGET_LABEL(toprow, VALUE=STRING(new_value,FORMAT='(d10.1)'))
        bits_bytes = CW_BSELECTOR2(toprow, ["bits ","bytes","% buffer"], SET_VALUE=0)

        temp = WIDGET_LABEL(toprow, VALUE='    Compression Factor:')
        comp_text = WIDGET_LABEL(toprow, VALUE=STRING(new_value,FORMAT='(d5.1)')+' %')

      scienceb = WIDGET_BUTTON(col0, VALUE='CALL SCIENCE TOOL')

      info_text = WIDGET_LABEL(col0, VALUE='                                                                                        ')

      temp = WIDGET_LABEL(sched_base, VALUE='TABLE DEFINITIONS')
      nextrow = WIDGET_BASE(sched_base, /ROW, /FRAME)
        nc = 3
        colm = LONARR(nc)
        FOR i=0, nc-1 DO colm(i) = WIDGET_BASE(nextrow, /COLUMN)
        nt = N_ELEMENTS(utable_types)
        FOR i=0, nt-1 DO BEGIN
          temp = WIDGET_BUTTON(colm(i / CEIL(FLOAT(nt)/nc)), VALUE=utable_types(i), EVENT_PRO=table_pro(i))
          IF (STRTRIM(utable_types(i),2) EQ "Occulter Mask Table") THEN occ_id = temp
          IF (STRTRIM(utable_types(i),2) EQ "ROI Mask Table") THEN roi_id = temp
        ENDFOR

      temp = WIDGET_LABEL(sched_base, VALUE='OBSERVING SEQUENCES')
      lprow = WIDGET_BASE(sched_base, /ROW, /FRAME)
        nc = 6
        colm = LONARR(nc)
        FOR i=0, nc-1 DO colm(i) = WIDGET_BASE(lprow, /COLUMN)
        nt = N_ELEMENTS(op_types)
        FOR i=0, nt-1 DO $
          op_buttons(i) = WIDGET_BUTTON(colm(i / CEIL(FLOAT(nt)/nc)), VALUE=op_types(i), EVENT_FUNC=op_pro(i))

      quit = WIDGET_BUTTON(sched_base, VALUE=" Quit ")

    ;********************************************************************
    ;** REALIZE THE WIDGETS *********************************************

    WIDGET_CONTROL, sched_base, /REALIZE
    WIDGET_CONTROL, occ_id, SET_UVALUE='OCCULTER'
    WIDGET_CONTROL, roi_id, SET_UVALUE='ROI'
    FOR i=0, nt-1 DO $
       WIDGET_CONTROL, op_buttons(i), SENSITIVE=op_sens(i)

    osv = CREATE_STRUCT( 'base', sched_base, 	$
                            'op_buttons', op_buttons,	$
                            'op_pro', op_pro,		$
                            'quit', quit,		$
                            'door', door,		$
                            'doorb', doorb,		$
                            'statsb', statsb,		$
                            'scienceb', scienceb,	$
                            'time_text', time_text,	$
                            'size_text', size_text,	$
                            'size_units', 0,		$
                            'bits_bytes', bits_bytes,	$
                            'comp_text', comp_text,	$
                            'info_text', info_text )

    XMANAGER, 'DEFINE_OS', sched_base

END
