;+
; Project     : SOHO - LASCO/EIT
;                   
; Name        : MASK
;               
; Purpose     : Widget tool to define a LASCO/EIT Mask or ROI Table.
;               
; Explanation : This tool allows the user to display a LASCO/EIT
;               image and interactively select the blocks to be 
;		masked (ie. not transmitted).  
;               
; Use         : IDL> MASK
;    
; Inputs      : None.
;               
; Opt. Inputs : caller		A structure containing the id of the caller.
;               
; Outputs     : None.
;               
; Opt. Outputs: None.
;               
; Keywords    : None.
;
; Calls       : CW_BSELECTOR2, POPUP_HELP, READFITS, FIND_DATA_RUNS, RWMASK
;
; Common      : MASK_SHARE	defined in this routine.
;		OS_ALL_SHARE 	defined in this routine or OS_INIT.PRO.
;               
; Restrictions: None.
;               
; Side effects: None.
;               
; Category    : Planning, Scheduling.
;               
; Prev. Hist. : None.
;
; Written     : Scott Paswaters, NRL, November 1994.
;               
; Modified    : 
;
; Version     : Version 0.1, December 5, 1994
;-            

;__________________________________________________________________________________________________________
;

PRO MASK_EVENT, event

COMMON MASK_SHARE, maskv
COMMON OS_ALL_SHARE, ccd, ip, ipd, ex, exd, occ_blocks, roi_blocks, fpwl, fpwld



    CASE (event.id) OF

	maskv.prnt : BEGIN
           selected = WHERE(maskv.blocks EQ 1)
           num = N_ELEMENTS(selected)
           PRINT, 'Number of blocks selected = ', num
           PRINT, 'Blocks = [', FORMAT='($,A)'
           FOR i=0, num-1 DO PRINT, selected(i),',', FORMAT='($,I5,A)'
           PRINT, ']'
           out = FIND_DATA_RUNS(INDGEN(1024),REFORM(maskv.blocks),0,0,1)
           str = 'Blocks = ('
           FOR i=0, (N_ELEMENTS(out)/2)-1 DO BEGIN
              str = str + STRTRIM(STRING(out(i,0)),2) + '-' + STRTRIM(STRING(out(i,1)),2) + ','
           ENDFOR
           str = STRMID(str,0,STRLEN(str)-1) + ')'
           PRINT, str
           obeblocks = BLOCKS2OBE(maskv.blocks)
           FOR i=0, 64-1,8 DO BEGIN
              FOR j=0, 7 DO BEGIN
                 b = obeblocks(i+j) AND 'ffff'x
                 b = STRTRIM(STRING(b, FORMAT='(z4)'),2)
                 s = '0x0000'
                 STRPUT, s, b, STRLEN(s) - STRLEN(b)
                 PRINT, s+', ', FORMAT='($,a)'
              ENDFOR
              PRINT
           ENDFOR
	END

        maskv.showgrid : BEGIN
           WSET, maskv.draw_win
           CASE (maskv.grid) OF
              0 : BEGIN		;SHOW GRID
                 WIDGET_CONTROL, maskv.showgrid, SET_VALUE=maskv.grid_types(1)        ;** change labels
                 DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap]
                 maskv.grid = 1
              END
              1 : BEGIN		;HIDE GRID
                 WIDGET_CONTROL, maskv.showgrid, SET_VALUE=maskv.grid_types(0)        ;** change labels
                 DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
                 maskv.grid = 0
              END
              ELSE : BEGIN
              END
           ENDCASE
        END
 
        maskv.loadimage : BEGIN
           newfile = PICKFILE(/READ, FILTER='*.fits', $
                              /FIX_FILTER, /NOCONFIRM, /MUST_EXIST)
           IF (newfile NE '') THEN BEGIN
           WIDGET_CONTROL, /HOUR
           WSET, maskv.draw_win
           ERASE, COLOR=maskv.gcol
           img = READFITS(newfile)
           img = CONGRID(img, maskv.ms, maskv.ms)
           std = STDEV(img, mn)
           img = BYTSCL(img, 0, mn+4*std, TOP=255-10)		;** leave 10 colors unallocated
           img = img + 10
           WSET, maskv.img_map
           TV, img
           WSET, maskv.mask_map
           temp = INTARR(maskv.ms,maskv.ms) + maskv.gcol
           TV, temp
           WSET, maskv.new_map
           TV, img
           WSET, maskv.new_gmap
           TV, img
           FOR i=0,31 DO PLOTS, [i*maskv.bs,i*maskv.bs], [0,maskv.ms], /DEVICE, COLOR=maskv.gcol
           FOR i=0,31 DO PLOTS, [0,maskv.ms], [i*maskv.bs,i*maskv.bs], /DEVICE, COLOR=maskv.gcol

           selected = WHERE(maskv.blocks EQ 1)
           IF (selected(0) NE -1) THEN BEGIN
             FOR i=0,N_ELEMENTS(selected)-1 DO BEGIN
               yc = selected(i)/32*maskv.bs
	       xc = (selected(i) - selected(i)/32*32)*maskv.bs
               WSET, maskv.new_map  & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
               WSET, maskv.new_gmap & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
             ENDFOR
           ENDIF
           ;LOADCT, 0, /SILENT
           ;TVLCT, 142, 229, 238, maskv.gcol
           WSET, maskv.draw_win 
           IF (maskv.grid) THEN $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap] $
           ELSE $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
           ENDIF
        END

	maskv.invert : BEGIN
           WIDGET_CONTROL, /HOUR
           WSET, maskv.new_map  & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
           WSET, maskv.new_gmap & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
           FOR i=0,31 DO PLOTS, [i*maskv.bs,i*maskv.bs], [0,maskv.ms], /DEVICE, COLOR=maskv.gcol
           FOR i=0,31 DO PLOTS, [0,maskv.ms], [i*maskv.bs,i*maskv.bs], /DEVICE, COLOR=maskv.gcol
           notselected = WHERE(maskv.blocks NE 1)
           newselected = notselected
           maskv.blocks(*) = 0
           IF (newselected(0) NE -1) THEN BEGIN
             maskv.blocks(newselected) = 1
             FOR i=0,N_ELEMENTS(newselected)-1 DO BEGIN
               yc = newselected(i)/32*maskv.bs
	       xc = (newselected(i) - newselected(i)/32*32)*maskv.bs
               WSET, maskv.new_map  & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
               WSET, maskv.new_gmap & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
             ENDFOR
           ENDIF
           WSET, maskv.draw_win 
           IF (maskv.grid) THEN $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap] $
           ELSE $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
           IF (maskv.type EQ 'OCCULTER') THEN occ_blocks(maskv.tele,*) = maskv.blocks ELSE $
           IF (maskv.type EQ 'ROI') THEN roi_blocks(maskv.tele,*) = maskv.blocks
	END

	maskv.clear : BEGIN
           maskv.blocks(*) = 0
           WSET, maskv.new_map  & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
           WSET, maskv.new_gmap & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
           FOR i=0,31 DO PLOTS, [i*maskv.bs,i*maskv.bs], [0,maskv.ms], /DEVICE, COLOR=maskv.gcol
           FOR i=0,31 DO PLOTS, [0,maskv.ms], [i*maskv.bs,i*maskv.bs], /DEVICE, COLOR=maskv.gcol
           WSET, maskv.draw_win 
           IF (maskv.grid) THEN $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap] $
           ELSE $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
           IF (maskv.type EQ 'OCCULTER') THEN occ_blocks(maskv.tele,*) = maskv.blocks ELSE $
           IF (maskv.type EQ 'ROI') THEN roi_blocks(maskv.tele,*) = maskv.blocks
	END

	maskv.draw_w : BEGIN	;** event in draw window
           IF (event.press NE 0) THEN maskv.press = event.press
           IF (event.release NE 0) THEN maskv.press = 0B
           IF (maskv.press NE 0) THEN BEGIN	;** button select
              x = 0 > event.x < (maskv.ms-1)
              y = 0 > event.y < (maskv.ms-1)
              bf = 1024./maskv.ms			;** block factor
              block = FIX(bf*x/32) + FIX(bf*y/32)*32
              yc = block/32*maskv.bs
	      xc = (block - block/32*32)*maskv.bs
              CASE (maskv.press) OF
                 1B: BEGIN	;** Left mouse button; select block
                    maskv.blocks(block) = 1		
                    WSET, maskv.new_map  & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
                    WSET, maskv.new_gmap & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
                    IF (maskv.type EQ 'OCCULTER') THEN occ_blocks(maskv.tele,*) = maskv.blocks ELSE $
                    IF (maskv.type EQ 'ROI') THEN roi_blocks(maskv.tele,*) = maskv.blocks
                 END
                 2B: BEGIN 	;** Middle mouse button; de-select block
                    maskv.blocks(block) = 0 
                    WSET, maskv.new_map  & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.img_map]
                    WSET, maskv.new_gmap & DEVICE, $
                          COPY = [xc+1, yc+1, maskv.bs-1, maskv.bs-1, xc+1, yc+1, maskv.img_map]
                    IF (maskv.type EQ 'OCCULTER') THEN occ_blocks(maskv.tele,*) = maskv.blocks ELSE $
                    IF (maskv.type EQ 'ROI') THEN roi_blocks(maskv.tele,*) = maskv.blocks
                 END
                 4B: BEGIN 	;** Right mouse button; adjust color table
                    nc  = ROUND((FLOAT(y)/maskv.ms)*!D.TABLE_SIZE)
                    cen = ROUND((FLOAT(x)/maskv.ms)*!D.TABLE_SIZE)
                    c0 = (cen - nc/2) > 0 < (!D.TABLE_SIZE-1)
                    c1 = (cen + nc/2) > 0 < (!D.TABLE_SIZE-1)
                    p = INTARR(!D.TABLE_SIZE)
                    IF (cen GT 0) THEN p(0:cen-1) = 0
                    IF (cen LT !D.TABLE_SIZE-1) THEN p(cen:!D.TABLE_SIZE-1) = 255
                    IF (c0 NE c1) THEN p(c0:c1-1) = BYTSCL(INDGEN(c1-c0)*!D.TABLE_SIZE, TOP=255-10)
                    TVLCT, p,p,p, maskv.gcol+1
                 END
              ENDCASE
              WSET, maskv.draw_win
              IF (maskv.grid) THEN $
                DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap] $
              ELSE $
                DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
           ENDIF
	END

	maskv.quit : BEGIN	;** exit program
	   WIDGET_CONTROL, /DESTROY, maskv.base
	END

	maskv.helpb : BEGIN	;** help text
           help_str = STRARR(10)
           help_str(0) = 'Select blocks to be masked with the left mouse button.'
           help_str(1) = 'De-select blocks with the middle mouse button.'
           help_str(2) = 'Use the right mouse button to adjust the color table.'
           POPUP_HELP, help_str, TITLE="MASK TOOL HELP"
	END

	maskv.telepd : BEGIN	;** select telescope table
           maskv.tele = event.index
           IF (maskv.type EQ 'OCCULTER') THEN maskv.blocks = occ_blocks(maskv.tele,*) ELSE $
           IF (maskv.type EQ 'ROI') THEN maskv.blocks = roi_blocks(maskv.tele,*)
           WSET, maskv.new_map  & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
           WSET, maskv.new_gmap & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
           FOR i=0,31 DO PLOTS, [i*maskv.bs,i*maskv.bs], [0,maskv.ms], /DEVICE, COLOR=maskv.gcol
           FOR i=0,31 DO PLOTS, [0,maskv.ms], [i*maskv.bs,i*maskv.bs], /DEVICE, COLOR=maskv.gcol
           selected = WHERE(maskv.blocks EQ 1)
           IF (selected(0) NE -1) THEN BEGIN
             FOR i=0,N_ELEMENTS(selected)-1 DO BEGIN
               yc = selected(i)/32*maskv.bs
               xc = (selected(i) - selected(i)/32*32)*maskv.bs
               WSET, maskv.new_map  & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
               WSET, maskv.new_gmap & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
             ENDFOR
           ENDIF
           WSET, maskv.draw_win
           IF (maskv.grid) THEN $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap] $
           ELSE $
             DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
	END

        ELSE : BEGIN
        END

   ENDCASE

END

;__________________________________________________________________________________________________________
;

PRO MASK, caller

COMMON MASK_SHARE
COMMON OS_ALL_SHARE, ccd, ip, ipd, ex, exd, occ_blocks, roi_blocks, fpwl, fpwld


    IF XRegistered("MASK") THEN BEGIN
       WIDGET_CONTROL, caller.id, GET_UVALUE=type
       maskv.type = type
       title = type+' TABLE'
       WIDGET_CONTROL, maskv.base, TLB_SET_TITLE=title
       IF (maskv.type EQ 'OCCULTER') THEN maskv.blocks = occ_blocks(maskv.tele,*) ELSE $
       IF (maskv.type EQ 'ROI') THEN maskv.blocks = roi_blocks(maskv.tele,*)
       WSET, maskv.new_map  & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
       WSET, maskv.new_gmap & DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.img_map]
       FOR i=0,31 DO PLOTS, [i*maskv.bs,i*maskv.bs], [0,maskv.ms], /DEVICE, COLOR=maskv.gcol
       FOR i=0,31 DO PLOTS, [0,maskv.ms], [i*maskv.bs,i*maskv.bs], /DEVICE, COLOR=maskv.gcol
       selected = WHERE(maskv.blocks EQ 1)
       IF (selected(0) NE -1) THEN BEGIN
         FOR i=0,N_ELEMENTS(selected)-1 DO BEGIN
           yc = selected(i)/32*maskv.bs
           xc = (selected(i) - selected(i)/32*32)*maskv.bs
           WSET, maskv.new_map  & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
           WSET, maskv.new_gmap & DEVICE, COPY = [xc, yc, maskv.bs, maskv.bs, xc, yc, maskv.mask_map]
         ENDFOR
       ENDIF
       WSET, maskv.draw_win
       IF (maskv.grid) THEN $
         DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_gmap] $
       ELSE $
         DEVICE, COPY = [0, 0, maskv.ms, maskv.ms, 0, 0, maskv.new_map]
       RETURN
    ENDIF

    ms = 640
    bs = ms/32			;** block size
    blocks = BYTARR(32*32)
    press = 0B
    title = 'LASCO/EIT MASK TOOL'
    tele_type = [ "  C1 Table  ", $
                  "  C2 Table  ", $
                  "  C3 Table  ", $
                  "  EIT Table "]

    IF ((SIZE(maskv))(1) EQ 0) THEN BEGIN          ;** not defined yet use default
       tele = 0
       grid = 1
       type = ''			;** occulter, or mask; for scheduling program
    ENDIF ELSE BEGIN
       tele = maskv.tele
       grid = maskv.grid
       type = maskv.type
    ENDELSE


    ;********************************************************************
    ;** SET UP WIDGETS **************************************************

    IF (KEYWORD_SET(caller)) THEN BEGIN
       WIDGET_CONTROL, caller.id, GET_UVALUE=type
       IF (type EQ 'OCCULTER') THEN BEGIN
          blocks = occ_blocks(tele,*) 
          title = 'OCCULTER MASK TABLE'
       ENDIF ELSE IF (type EQ 'ROI') THEN BEGIN
          blocks = roi_blocks(tele,*)
          title = 'ROI MASK TABLE'
       ENDIF
       base = WIDGET_BASE(/COLUMN, TITLE=title, /FRAME, GROUP_LEADER=caller.id)
    ENDIF ELSE $
       base = WIDGET_BASE(/COLUMN, TITLE=title, /FRAME)

    row = WIDGET_BASE(base, /ROW)
      telepd = CW_BSELECTOR2(row, tele_type, SET_VALUE=tele)
      helpb = WIDGET_BUTTON(row, VALUE='  HELP  ')
    draw_w = WIDGET_DRAW(base, XSIZE=ms, YSIZE=ms, /BUTTON_EVENTS, /MOTION_EVENTS)

    row = WIDGET_BASE(base, /ROW)
      grid_types = ['   Show Grid   ','   Hide Grid   ']
      showgrid = WIDGET_BUTTON(row, VALUE=grid_types(1))
      loadimage  = WIDGET_BUTTON(row, VALUE='  Load Image   ')
      rwmask  = WIDGET_BUTTON(row, VALUE='Load/Save Mask ', EVENT_PRO='RWMASK')
    row = WIDGET_BASE(base, /ROW)
      invert = WIDGET_BUTTON(row, VALUE=' Invert Blocks ')
      clear = WIDGET_BUTTON(row, VALUE=' Clear Blocks  ')
      prnt = WIDGET_BUTTON(row, VALUE=' Print Blocks  ')
    quit = WIDGET_BUTTON(base, VALUE=" Dismiss ")

    ;********************************************************************
    ;** REALIZE THE WIDGETS *********************************************

    WIDGET_CONTROL, base, /REALIZE
    WIDGET_CONTROL, draw_w, GET_VALUE=draw_win
    WIDGET_CONTROL, /HOUR

    gcol = 9	;** color index for grid
    ;img = READFITS('/home/argus/scott/images/occulter/c2.fits')
    IF ((SIZE(img))(0) NE 2) THEN img = BYTARR(ms,ms) $
    ELSE BEGIN
       img = CONGRID(img, ms, ms)
       std = STDEV(img, mn)
       img = BYTSCL(img, 0, mn+4*std, TOP=255-10)		;** leave 10 colors unallocated
    ENDELSE
    img = img + 10

    img_map = 29
    WINDOW, img_map, XSIZE=ms, YSIZE=ms, /PIXMAP
    TV, img
    mask_map = 30
    WINDOW, mask_map, XSIZE=ms, YSIZE=ms, /PIXMAP
    temp = INTARR(ms,ms) + gcol
    TV, temp
    new_map = 31
    WINDOW, new_map, XSIZE=ms, YSIZE=ms, /PIXMAP
    TV, img
    new_gmap = 28
    WINDOW, new_gmap, XSIZE=ms, YSIZE=ms, /PIXMAP
    TV, img
    FOR i=0,31 DO PLOTS, [i*bs,i*bs], [0,ms], /DEVICE, COLOR=gcol
    FOR i=0,31 DO PLOTS, [0,ms], [i*bs,i*bs], /DEVICE, COLOR=gcol

    ;** if blocks were set at some previous time reset them
    selected = WHERE(blocks EQ 1)
    IF (selected(0) NE -1) THEN BEGIN
      FOR i=0,N_ELEMENTS(selected)-1 DO BEGIN
        yc = selected(i)/32*bs
        xc = (selected(i) - selected(i)/32*32)*bs
        WSET, new_map  & DEVICE, COPY = [xc, yc, bs, bs, xc, yc, mask_map]
        WSET, new_gmap & DEVICE, COPY = [xc, yc, bs, bs, xc, yc, mask_map]
      ENDFOR
    ENDIF

    LOADCT, 0, /SILENT

 TVLCT, 255,0,0, 1      ;** load red into entry 1
 TVLCT, 0,255,0, 2      ;** load green into entry 2
 TVLCT, 142,229,238, 3  ;** load blue into entry 3
 TVLCT, 255,255,0,4     ;** load yellow into entry 4
 TVLCT, 200,0,0, 5      ;** load dull red into entry 5
 TVLCT, 0,200,0, 6      ;** load dull green into entry 6
 TVLCT, 0,206,237, 7    ;** load dull blue into entry 7
 TVLCT, 200,200,0,8     ;** load dull yellow into entry 8
 TVLCT, 142,229,238, 9  ;** load light blue into entry 9 - used by mask.pro


    WSET, draw_win 
    IF (grid) THEN $
      DEVICE, COPY = [0, 0, ms, ms, 0, 0, new_gmap] $
    ELSE $
      DEVICE, COPY = [0, 0, ms, ms, 0, 0, new_map]

    maskv = CREATE_STRUCT( 'base', base,	 	$
                           'quit', quit, 		$
                           'telepd', telepd, 		$
                           'helpb', helpb, 		$
                           'draw_win', draw_win, 	$
                           'draw_w', draw_w, 		$
                           'bs', bs, 			$
                           'ms', ms, 			$
                           'blocks', blocks,		$
                           'img_map', img_map, 		$
                           'mask_map', mask_map, 	$
                           'new_map', new_map, 		$
                           'new_gmap', new_gmap, 	$
                           'grid', grid, 		$
                           'gcol', gcol, 		$
                           'press', press, 		$
                           'showgrid', showgrid, 	$
                           'grid_types', grid_types, 	$
                           'clear', clear, 		$
                           'invert', invert, 		$
                           'prnt', prnt, 		$
                           'type', type, 		$
                           'tele', tele, 		$
                           'loadimage', loadimage)

   XMANAGER, 'MASK', base

END
