;+
; Project     : SOHO - LASCO/EIT
;
; Name        : OS_TABLES
;	      : TABLE_CAMERA		Implemented.
;	      : TABLE_PROCESSING	Implemented.
;	      : TABLE_EXPOSURE		Implemented.
;	      : TABLE_OCCULTER		Implemented.
;	      : TABLE_MASK		Implemented.
;	      : TABLE_BADCOL
;	      : TABLE_TRANSIENT
;	      : TABLE_FP_WL_EXP		Implemented.
;
; Purpose     : Widgets to define LEB Tables.
;
; Explanation : These routines create a widget interface for a specific
;               LASCO/EIT LEB Table definition.
;
; Use         : TABLE_CAMERA, caller
;
; Inputs      : caller  structure containing id of caller.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Calls       : CW_BSELECTOR2, P_CHECK, POPUP_HELP, MASK
;
; Called From : DEFINE_OS
;
; Common      : OS_INIT_SHARE 		Defined in OS_INIT.
;               OS_ALL_SHARE  		Defined in OS_INIT.
;               The following are defined in these routines.
;		OS_TABLE_CAMERA_SHARE, ccdv
;		OS_TABLE_PROCESSING_SHARE, ipv
;		OS_TABLE_EXPOSURE_SHARE, exv
;		OS_TABLE_OCCULTER_SHARE, base, dismiss
;		OS_TABLE_MASK_SHARE, base, dismiss
;		OS_TABLE_BADCOL_SHARE, base, dismiss
;		OS_TABLE_TRANSIENT_SHARE, base, dismiss
;		OS_TABLE_FP_WL_EXP_SHARE, fpwlv
;
; Restrictions: None.
;
; Side effects: None.
;
; Category    : Planning, Scheduling.
;
; Prev. Hist. : None.
;
; Written     : Scott Paswaters, NRL, November 1994.
;
; Modified    :
;
; Version     : Version 0.1, December 5, 1994
;-
 
;__________________________________________________________________________________________________________
;

;**
;** Routine to initialize table upload routines used by schedule.pro
;**
;** PRO TABLE_CAMERA
;** PRO TABLE_PROCESSING
;** PRO TABLE_EXPOSURE
;** PRO TABLE_OCCULTER
;** PRO TABLE_MASK
;** PRO TABLE_BADCOL
;** PRO TABLE_TRANSIENT
;** PRO TABLE_FP_WL_EXP
;** PRO OS_TABLES
;**
;**

;________________________________________________________________________________________________________
;

PRO TABLE_CAMERA_EVENT, event

COMMON OS_ALL_SHARE
COMMON OS_INIT_SHARE
COMMON OS_TABLE_CAMERA_SHARE, ccdv

CASE (event.id) OF

   ccdv.dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, ccdv.base
      GOTO, done
   END

   ccdv.helpb : BEGIN	;** display help
      help_str = STRARR(10)
      help_str(0) = 'These tables define the camera state for each telescope.'
      help_str(1) = 'The first 20 columns on the CCD are underscan.'
      help_str(2) = 'In the x2: field you can enter +value where value is the number'
      help_str(3) = 'of columns you wish to have starting with x1. Ex. +512'
      POPUP_HELP, help_str, TITLE="CAMERA TABLE HELP"
   END

   ccdv.telepd : BEGIN
      ccdv.tele = event.index
      WIDGET_CONTROL, ccdv.x1_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).x1),2)
      WIDGET_CONTROL, ccdv.x2_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).x2),2)
      WIDGET_CONTROL, ccdv.y1_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).y1),2)
      WIDGET_CONTROL, ccdv.y2_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).y2),2)
      WIDGET_CONTROL, ccdv.xsum_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).xsum),2)
      WIDGET_CONTROL, ccdv.ysum_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).ysum),2)
      WIDGET_CONTROL, ccdv.clearpd, SET_VALUE=ccd(ccdv.tele,ccdv.table).clr_spd
      WIDGET_CONTROL, ccdv.clear_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).nclrs),2)
      WIDGET_CONTROL, ccdv.portpd, SET_VALUE=ccd(ccdv.tele,ccdv.table).port
      WIDGET_CONTROL, ccdv.leb_sum_text, SET_VALUE=STRTRIM(ccd(ccdv.tele,ccdv.table).leb_sum, 2)
      WIDGET_CONTROL, ccdv.lpulse_text, SET_VALUE=STRTRIM(ccd(ccdv.tele,ccdv.table).lpulse, 2)
      WIDGET_CONTROL, ccdv.htrpd, SET_VALUE=ccd(ccdv.tele,ccdv.table).hihtr
      WIDGET_CONTROL, ccdv.lowhtr_text, SET_VALUE=STRTRIM(ccd(ccdv.tele,ccdv.table).lowhtr, 2)
      WIDGET_CONTROL, ccdv.sidepd, SET_VALUE=ccd(ccdv.tele,ccdv.table).side
      WIDGET_CONTROL, ccdv.cc_opts_id, SET_VALUE=ccd(ccdv.tele,ccdv.table).fp_candc
      IF (ccdv.tele NE 0) THEN ses = 0 ELSE ses = 1
      WIDGET_CONTROL, ccdv.cc_opts_id, SENSITIVE=ses
      PLOT_CCD, ccdv.draw_win, ccdv.tele, $
         ccd(ccdv.tele,ccdv.table).x1, ccd(ccdv.tele,ccdv.table).x2, $
         ccd(ccdv.tele,ccdv.table).y1, ccd(ccdv.tele,ccdv.table).y2
   END

   ccdv.tablepd : BEGIN
      ccdv.table = event.index
      WIDGET_CONTROL, ccdv.x1_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).x1),2)
      WIDGET_CONTROL, ccdv.x2_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).x2),2)
      WIDGET_CONTROL, ccdv.y1_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).y1),2)
      WIDGET_CONTROL, ccdv.y2_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).y2),2)
      WIDGET_CONTROL, ccdv.xsum_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).xsum),2)
      WIDGET_CONTROL, ccdv.ysum_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).ysum),2)
      WIDGET_CONTROL, ccdv.clearpd, SET_VALUE=ccd(ccdv.tele,ccdv.table).clr_spd
      WIDGET_CONTROL, ccdv.clear_text, SET_VALUE=STRTRIM(STRING(ccd(ccdv.tele,ccdv.table).nclrs),2)
      WIDGET_CONTROL, ccdv.portpd, SET_VALUE=ccd(ccdv.tele,ccdv.table).port
      WIDGET_CONTROL, ccdv.leb_sum_text, SET_VALUE=STRTRIM(ccd(ccdv.tele,ccdv.table).leb_sum, 2)
      WIDGET_CONTROL, ccdv.lpulse_text, SET_VALUE=STRTRIM(ccd(ccdv.tele,ccdv.table).lpulse, 2)
      WIDGET_CONTROL, ccdv.htrpd, SET_VALUE=ccd(ccdv.tele,ccdv.table).hihtr
      WIDGET_CONTROL, ccdv.lowhtr_text, SET_VALUE=STRTRIM(ccd(ccdv.tele,ccdv.table).lowhtr, 2)
      WIDGET_CONTROL, ccdv.sidepd, SET_VALUE=ccd(ccdv.tele,ccdv.table).side
      WIDGET_CONTROL, ccdv.cc_opts_id, SET_VALUE=ccd(ccdv.tele,ccdv.table).fp_candc
      PLOT_CCD, ccdv.draw_win, ccdv.tele, $
         ccd(ccdv.tele,ccdv.table).x1, ccd(ccdv.tele,ccdv.table).x2, $
         ccd(ccdv.tele,ccdv.table).y1, ccd(ccdv.tele,ccdv.table).y2
   END

   ccdv.x1_text : BEGIN
      WIDGET_CONTROL, ccdv.x1_text, GET_VALUE=x1 & x1 = x1(0)
      temp = P_CHECK(x1,'x1',new) & WIDGET_CONTROL, ccdv.x1_text, SET_VALUE=new
      WIDGET_CONTROL, ccdv.x2_text, /INPUT_FOCUS
      WIDGET_CONTROL, ccdv.x2_text, GET_VALUE=x2
      WIDGET_CONTROL, ccdv.x2_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(x2),2))]
      PLOT_CCD, ccdv.draw_win, ccdv.tele, $
         x1, ccd(ccdv.tele,ccdv.table).x2, $
         ccd(ccdv.tele,ccdv.table).y1, ccd(ccdv.tele,ccdv.table).y2
   END

   ccdv.x2_text : BEGIN
      WIDGET_CONTROL, ccdv.x2_text, GET_VALUE=x2 & x2 = x2(0)
      temp = P_CHECK(x2,'x2',new) & WIDGET_CONTROL, ccdv.x2_text, SET_VALUE=new
      WIDGET_CONTROL, ccdv.y1_text, /INPUT_FOCUS
      WIDGET_CONTROL, ccdv.y1_text, GET_VALUE=y1
      WIDGET_CONTROL, ccdv.y1_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(y1),2))]
      PLOT_CCD, ccdv.draw_win, ccdv.tele, $
         ccd(ccdv.tele,ccdv.table).x1, x2, $
         ccd(ccdv.tele,ccdv.table).y1, ccd(ccdv.tele,ccdv.table).y2
   END

   ccdv.y1_text : BEGIN
      WIDGET_CONTROL, ccdv.y1_text, GET_VALUE=y1 & y1 = y1(0)
      temp = P_CHECK(y1,'y1',new) & WIDGET_CONTROL, ccdv.y1_text, SET_VALUE=new
      WIDGET_CONTROL, ccdv.y2_text, /INPUT_FOCUS
      WIDGET_CONTROL, ccdv.y2_text, GET_VALUE=y2
      WIDGET_CONTROL, ccdv.y2_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(y2),2))]
      PLOT_CCD, ccdv.draw_win, ccdv.tele, $
         ccd(ccdv.tele,ccdv.table).x1, ccd(ccdv.tele,ccdv.table).x2, $
         y1, ccd(ccdv.tele,ccdv.table).y2
   END

   ccdv.y2_text : BEGIN
      WIDGET_CONTROL, ccdv.y2_text, GET_VALUE=y2 & y2 = y2(0)
      temp = P_CHECK(y2,'y2',new) & WIDGET_CONTROL, ccdv.y2_text, SET_VALUE=new
      WIDGET_CONTROL, ccdv.xsum_text, /INPUT_FOCUS
      WIDGET_CONTROL, ccdv.xsum_text, GET_VALUE=xsum
      WIDGET_CONTROL, ccdv.xsum_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(xsum),2))]
      PLOT_CCD, ccdv.draw_win, ccdv.tele, $
         ccd(ccdv.tele,ccdv.table).x1, ccd(ccdv.tele,ccdv.table).x2, $
         ccd(ccdv.tele,ccdv.table).y1, y2
   END

   ccdv.xsum_text : BEGIN
      WIDGET_CONTROL, ccdv.xsum_text, GET_VALUE=xsum & xsum = xsum(0)
      temp = P_CHECK(xsum,'xsum',new) & WIDGET_CONTROL, ccdv.xsum_text, SET_VALUE=new
      WIDGET_CONTROL, ccdv.ysum_text, /INPUT_FOCUS
      WIDGET_CONTROL, ccdv.ysum_text, GET_VALUE=ysum
      WIDGET_CONTROL, ccdv.ysum_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(ysum),2))]
   END

   ccdv.ysum_text : BEGIN
      WIDGET_CONTROL, ccdv.ysum_text, GET_VALUE=ysum & ysum = ysum(0)
      temp = P_CHECK(ysum,'ysum',new) & WIDGET_CONTROL, ccdv.ysum_text, SET_VALUE=new
      WIDGET_CONTROL, ccdv.x1_text, /INPUT_FOCUS
      WIDGET_CONTROL, ccdv.x1_text, GET_VALUE=x1
      WIDGET_CONTROL, ccdv.x1_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(x1),2))]
   END

   ccdv.clear_text : BEGIN
      WIDGET_CONTROL, ccdv.clear_text, GET_VALUE=nclrs & nclrs = nclrs(0)
      temp = P_CHECK(nclrs,'nclrs',new) & WIDGET_CONTROL, ccdv.clear_text, SET_VALUE=new
   END

   ccdv.lpulse_text : BEGIN
      WIDGET_CONTROL, ccdv.lpulse_text, GET_VALUE=lpulse & lpulse = lpulse(0)
      temp = P_CHECK(lpulse,'lpulse',new) & WIDGET_CONTROL, ccdv.lpulse_text, SET_VALUE=new
   END

   ccdv.leb_sum_text : BEGIN
      WIDGET_CONTROL, ccdv.leb_sum_text, GET_VALUE=leb_sum & leb_sum = leb_sum(0)
      temp = P_CHECK(leb_sum,'leb_sum',new) & WIDGET_CONTROL, ccdv.leb_sum_text, SET_VALUE=new
   END

   ccdv.lowhtr_text : BEGIN
      WIDGET_CONTROL, ccdv.lowhtr_text, GET_VALUE=lowhtr & lowhtr = lowhtr(0)
      temp = P_CHECK(lowhtr,'lowhtr',new) & WIDGET_CONTROL, ccdv.lowhtr_text, SET_VALUE=new
   END

   ELSE : BEGIN
   END

ENDCASE

WIDGET_CONTROL, ccdv.x1_text, GET_VALUE=x1
WIDGET_CONTROL, ccdv.x2_text, GET_VALUE=x2
WIDGET_CONTROL, ccdv.y1_text, GET_VALUE=y1
WIDGET_CONTROL, ccdv.y2_text, GET_VALUE=y2
WIDGET_CONTROL, ccdv.xsum_text, GET_VALUE=xsum
WIDGET_CONTROL, ccdv.ysum_text, GET_VALUE=ysum
WIDGET_CONTROL, ccdv.clearpd, GET_VALUE=clr_spd
WIDGET_CONTROL, ccdv.clear_text, GET_VALUE=nclrs
WIDGET_CONTROL, ccdv.portpd, GET_VALUE=port
WIDGET_CONTROL, ccdv.leb_sum_text, GET_VALUE=leb_sum
WIDGET_CONTROL, ccdv.lpulse_text, GET_VALUE=lpulse
WIDGET_CONTROL, ccdv.htrpd, GET_VALUE=hihtr
WIDGET_CONTROL, ccdv.lowhtr_text, GET_VALUE=lowhtr
WIDGET_CONTROL, ccdv.sidepd, GET_VALUE=side
WIDGET_CONTROL, ccdv.cc_opts_id, GET_VALUE=cc_opts
ccd(ccdv.tele,ccdv.table).x1 = FIX(x1(0))
ccd(ccdv.tele,ccdv.table).x2 = FIX(x2(0))
ccd(ccdv.tele,ccdv.table).y1 = FIX(y1(0))
ccd(ccdv.tele,ccdv.table).y2 = FIX(y2(0))
ccd(ccdv.tele,ccdv.table).xsum = FIX(xsum(0))
ccd(ccdv.tele,ccdv.table).ysum = FIX(ysum(0))
ccd(ccdv.tele,ccdv.table).clr_spd = BYTE(clr_spd(0))
ccd(ccdv.tele,ccdv.table).nclrs = FIX(nclrs(0))
ccd(ccdv.tele,ccdv.table).port = FIX(port(0))
ccd(ccdv.tele,ccdv.table).lpulse = FIX(lpulse(0))
ccd(ccdv.tele,ccdv.table).leb_sum = FIX(leb_sum(0))
ccd(ccdv.tele,ccdv.table).hihtr = BYTE(hihtr(0))
ccd(ccdv.tele,ccdv.table).lowhtr = FIX(lowhtr(0))
ccd(ccdv.tele,ccdv.table).side = BYTE(side(0))
fpopts = 0
FOR i=0, N_ELEMENTS(cc_opts)-1 DO fpopts = fpopts + cc_opts(i) * 2^i
ccd(ccdv.tele,ccdv.table).fp_candc = fpopts

done:
END

;________________________________________________________________________________________________________
;

PRO TABLE_CAMERA, caller

COMMON OS_ALL_SHARE
COMMON OS_TABLE_CAMERA_SHARE
COMMON OS_INIT_SHARE

    IF XRegistered("TABLE_CAMERA") THEN RETURN

   IF ((SIZE(ccdv))(1) EQ 0) THEN BEGIN		 ;** not defined yet use default
      tele = 0		
      table = 0
   ENDIF ELSE BEGIN
      tele = ccdv.tele		
      table = ccdv.table
   ENDELSE

   cc_opts = ['Long  - Begin OP', $
              'Short - After Image', $
              'Long  - After Image', $
              'Status + Mux', $
              'Short - Wobble', $
              'Short - Wobble']

    ;********************************************************************
    ;** SET UP WIDGETS **************************************************

    base = WIDGET_BASE(/ROW, TITLE='LASCO/EIT TABLE_CAMERA TOOL', /FRAME, GROUP_LEADER=caller.id)

      base1 = WIDGET_BASE(base, /COLUMN)

      row0 = WIDGET_BASE(base1, /ROW)
        telepd =  CW_BSELECTOR2(row0, tele_types, SET_VALUE=tele)
        tablepd = CW_BSELECTOR2(row0, table_types, SET_VALUE=table)
        helpb = WIDGET_BUTTON(row0, VALUE='     Help      ')

      row = WIDGET_BASE(base1, /ROW)

       ccdcol = WIDGET_BASE(row, /COLUMN, /FRAME)

         camrow = WIDGET_BASE(ccdcol, /ROW, /FRAME)
           temp = WIDGET_LABEL(camrow, VALUE='          CCD Setup')

         camrow0 = WIDGET_BASE(ccdcol, /ROW)
           temp = WIDGET_LABEL(camrow0, VALUE='  x1:')
           x1_text = WIDGET_TEXT(camrow0, /EDITABLE, YSIZE=1, XSIZE=5, $
              VALUE=STRTRIM(STRING(ccd(tele,table).x1),2))
           temp = WIDGET_LABEL(camrow0, VALUE='  x2:')
           x2_text = WIDGET_TEXT(camrow0, /EDITABLE, YSIZE=1, XSIZE=5, $
              VALUE=STRTRIM(STRING(ccd(tele,table).x2),2))

         camrow1 = WIDGET_BASE(ccdcol, /ROW)
           temp = WIDGET_LABEL(camrow1, VALUE='  y1:')
           y1_text = WIDGET_TEXT(camrow1, /EDITABLE, YSIZE=1, XSIZE=5, $
              VALUE=STRTRIM(STRING(ccd(tele,table).y1),2))
           temp = WIDGET_LABEL(camrow1, VALUE='  y2:')
           y2_text = WIDGET_TEXT(camrow1, /EDITABLE, YSIZE=1, XSIZE=5, $
              VALUE=STRTRIM(STRING(ccd(tele,table).y2),2))

         camrow2 = WIDGET_BASE(ccdcol, /ROW)
           temp = WIDGET_LABEL(camrow2, VALUE='xsum:')
           xsum_text = WIDGET_TEXT(camrow2, /EDITABLE, YSIZE=1, XSIZE=5, $
              VALUE=STRTRIM(STRING(ccd(tele,table).xsum),2))
           temp = WIDGET_LABEL(camrow2, VALUE='ysum:')
           ysum_text = WIDGET_TEXT(camrow2, /EDITABLE, YSIZE=1, XSIZE=5, $
              VALUE=STRTRIM(STRING(ccd(tele,table).ysum),2))

       col = WIDGET_BASE(row, /COLUMN, /FRAME)
         camrow3 = WIDGET_BASE(col, /ROW)
           portpd =CW_BSELECTOR2(camrow3, ['Port A','Port B','Port C','Port D'], $
			SET_VALUE=ccd(tele,table).port)
           clearpd =CW_BSELECTOR2(camrow3, ['Slow Clears','Fast Clears'], SET_VALUE=ccd(tele,table).clr_spd)
           clear_text = WIDGET_TEXT(camrow3, /EDITABLE, YSIZE=1, XSIZE=2, $
                        VALUE=STRTRIM(STRING(ccd(tele,table).nclrs),2))

         row2 = WIDGET_BASE(col, /ROW)
           temp = WIDGET_LABEL(row2, VALUE='Long Pulse:')
           lpulse_text = WIDGET_TEXT(row2, /EDITABLE, YSIZE=1, XSIZE=2, $
                                  VALUE=STRTRIM(STRING(ccd(tele,table).lpulse),2))
           temp = WIDGET_LABEL(row2, VALUE='LEB Sum:')
           leb_sum_text = WIDGET_TEXT(row2, /EDITABLE, YSIZE=1, XSIZE=2, $
                                  VALUE=STRTRIM(STRING(ccd(tele,table).leb_sum),2))

         row2 = WIDGET_BASE(col, /ROW)
           htrpd =CW_BSELECTOR2(row2, ['High Htr OFF','High Htr ON'], SET_VALUE=ccd(tele,table).hihtr)
           temp = WIDGET_LABEL(row2, VALUE='Low Htr:')
           lowhtr_text = WIDGET_TEXT(row2, /EDITABLE, YSIZE=1, XSIZE=3, $
                                  VALUE=STRTRIM(STRING(ccd(tele,table).lowhtr),2))
         sidepd =CW_BSELECTOR2(col, ['Frontsided','Backsided'], SET_VALUE=ccd(tele,table).side)

         col = WIDGET_BASE(base, /COLUMN, /FRAME)
           row2 = WIDGET_BASE(col, /ROW, /FRAME)
             temp = WIDGET_LABEL(row2, VALUE='    FP C&C Options')
           cc_opts_id = CW_BGROUP(col, cc_opts, /NONEXCLUSIVE)

     dismiss = WIDGET_BUTTON(base1, VALUE=" Dismiss ")

     draw_w = WIDGET_DRAW(base, /FRAME, XSIZE=256+5, YSIZE=256)

    ;********************************************************************
    ;** REALIZE THE WIDGETS *********************************************

    WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)
    WIDGET_CONTROL, x1_text, /INPUT_FOCUS
    WIDGET_CONTROL, x1_text, GET_VALUE=x1
    WIDGET_CONTROL, x1_text, SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(x1),2))]
    WIDGET_CONTROL, draw_w, GET_VALUE=draw_win
    IF (tele NE 0) THEN WIDGET_CONTROL, cc_opts_id, SENSITIVE=0

    PLOT_CCD, draw_win, tele, $
       ccd(tele,table).x1, ccd(tele,table).x2, ccd(tele,table).y1, ccd(tele,table).y2

    ccdv = CREATE_STRUCT('base', base,			$
                         'dismiss', dismiss,		$
                         'draw_win', draw_win,		$
                         'x1_text', x1_text,		$
                         'x2_text', x2_text,		$
                         'y1_text', y1_text,		$
                         'y2_text', y2_text,		$
                         'xsum_text', xsum_text,	$
                         'ysum_text', ysum_text,	$
                         'telepd', telepd,		$
                         'tablepd', tablepd,		$
                         'helpb', helpb,		$
                         'clearpd', clearpd,		$
                         'clear_text', clear_text,	$
                         'lpulse_text', lpulse_text,	$
                         'leb_sum_text', leb_sum_text,	$
                         'cc_opts_id', cc_opts_id,	$
                         'portpd', portpd,		$
                         'htrpd', htrpd,		$
                         'sidepd', sidepd,		$
                         'lowhtr_text', lowhtr_text,	$
                         'table', table,	$
                         'tele', tele)

    XMANAGER, 'TABLE_CAMERA', base

END

;________________________________________________________________________________________________________
;

PRO TABLE_PROCESSING_EVENT, event

COMMON OS_ALL_SHARE
COMMON OS_INIT_SHARE
COMMON OS_TABLE_PROCESSING_SHARE, ipv

    CASE (event.id) OF

	ipv.dismiss : BEGIN	;** exit program
	   WIDGET_CONTROL, /DESTROY, ipv.base
	END

	ipv.defaultb : BEGIN	;** load defaults
           ip = ipd
           good = WHERE(ip(ipv.iptable).steps GE 0)
           IF (good(0) LT 0) THEN WIDGET_CONTROL, ipv.proclist2, SET_VALUE = "" $
           ELSE WIDGET_CONTROL, ipv.proclist2, SET_VALUE = ip_arr(ip(ipv.iptable).steps(good)-1).ip_description
           len = N_ELEMENTS(good) & IF (len GT 5) THEN WIDGET_CONTROL, ipv.proclist2, SET_LIST_TOP = len-5
	END


	ipv.proclist : BEGIN
           good = WHERE(ip(ipv.iptable).steps GE 0)
           IF (good(0) LT 0) THEN $
              ip(ipv.iptable).steps(0) = event.index+1 $;** first selection
           ELSE BEGIN
              steps = ip(ipv.iptable).steps(WHERE(ip(ipv.iptable).steps GE 0))
              ip(ipv.iptable).steps = [steps, event.index+1]  ;** additional selection
           ENDELSE
           good = WHERE(ip(ipv.iptable).steps GE 0)
           IF (good(0) LT 0) THEN WIDGET_CONTROL, ipv.proclist2, SET_VALUE = "" $
           ELSE WIDGET_CONTROL, ipv.proclist2, SET_VALUE = ip_arr(ip(ipv.iptable).steps(good)-1).ip_description
           len = N_ELEMENTS(good) & IF (len GT 5) THEN WIDGET_CONTROL, ipv.proclist2, SET_LIST_TOP = len-5
	END

	ipv.proclist2 : BEGIN
           ip(ipv.iptable).steps(event.index) = -1
           good = WHERE(ip(ipv.iptable).steps GE 0)
           steps = ip(ipv.iptable).steps
           IF (good(0) GE 0) THEN BEGIN
              ip(ipv.iptable).steps = -1
              ip(ipv.iptable).steps(0:N_ELEMENTS(good)-1) = steps(good)
           ENDIF
           good = WHERE(ip(ipv.iptable).steps GE 0)
           IF (good(0) LT 0) THEN WIDGET_CONTROL, ipv.proclist2, SET_VALUE = "" $
           ELSE WIDGET_CONTROL, ipv.proclist2, SET_VALUE = ip_arr(ip(ipv.iptable).steps(good)-1).ip_description
           len = N_ELEMENTS(good) & IF (len GT 5) THEN WIDGET_CONTROL, ipv.proclist2, SET_LIST_TOP = len-5
	END

        ipv.ip_table_pd : BEGIN		;** new table number
           ipv.iptable = event.index
           ;** list current contents for this table
           good = WHERE(ip(ipv.iptable).steps GE 0)
           IF (good(0) LT 0) THEN WIDGET_CONTROL, ipv.proclist2, SET_VALUE = "" $
           ELSE WIDGET_CONTROL, ipv.proclist2, SET_VALUE = ip_arr(ip(ipv.iptable).steps(good)-1).ip_description
           len = N_ELEMENTS(good) & IF (len GT 5) THEN WIDGET_CONTROL, ipv.proclist2, SET_LIST_TOP = len-5
        END

        ELSE : BEGIN
        END

   ENDCASE

END

;________________________________________________________________________________________________________
;

PRO TABLE_PROCESSING, caller

COMMON OS_TABLE_PROCESSING_SHARE
COMMON OS_INIT_SHARE
COMMON OS_ALL_SHARE

   IF XRegistered("TABLE_PROCESSING") THEN RETURN

   IF ((SIZE(ipv))(1) EQ 0) THEN BEGIN		 ;** not defined yet use default
      iptable = 0
   ENDIF ELSE BEGIN
      iptable = ipv.iptable
   ENDELSE

    ;********************************************************************
    ;** SET UP WIDGETS **************************************************

    base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT TABLE_PROCESSING TOOL', /FRAME, $
             GROUP_LEADER=caller.id)

      row1 = WIDGET_BASE(base, /ROW)
        ip_table_pd = CW_BSELECTOR2(row1, proc_tab_types, SET_VALUE=iptable)
        defaultb = WIDGET_BUTTON(row1, VALUE='Load Defaults')

      row0 = WIDGET_BASE(base, /ROW, /FRAME)

        col0 = WIDGET_BASE(row0, /COLUMN)

          lbl =  WIDGET_LABEL(col0, VALUE = "Image Processing Steps")
          proclist = WIDGET_LIST(col0, VALUE = ip_arr.ip_description, YSIZE = 9, XSIZE=30)

        col1 = WIDGET_BASE(row0, /COLUMN)

          lbl =  WIDGET_LABEL(col1, VALUE = "Selected Processing Steps")
          proclist2 = WIDGET_LIST(col1, YSIZE = 9, XSIZE=30)

      dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")

    ;********************************************************************
    ;** REALIZE THE WIDGETS *********************************************

    WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)

    good = WHERE(ip(iptable).steps GE 0)
    IF (good(0) LT 0) THEN WIDGET_CONTROL, proclist2, SET_VALUE = "" $
    ELSE WIDGET_CONTROL, proclist2, SET_VALUE = ip_arr(ip(iptable).steps(good)-1).ip_description
    len = N_ELEMENTS(good) & IF (len GT 5) THEN WIDGET_CONTROL, ipv.proclist2, SET_LIST_TOP = len-5

    ipv = CREATE_STRUCT( 'base', base,			$
                         'dismiss', dismiss,		$
                         'proclist', proclist,		$
                         'proclist2', proclist2,	$
                         'iptable', iptable,		$
                         'defaultb', defaultb,		$
                         'ip_table_pd', ip_table_pd)

    XMANAGER, 'TABLE_PROCESSING', base

END

;________________________________________________________________________________________________________
;

PRO TABLE_EXPOSURE_EVENT, event
   COMMON OS_TABLE_EXPOSURE_SHARE, exv
   COMMON OS_ALL_SHARE
   COMMON OS_INIT_SHARE

CASE (event.id) OF

   exv.dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, exv.base
   END

   exv.helpb : BEGIN	;** display help
      help_str = STRARR(10)
      help_str(0) = 'These tables define exposure times in units of 1/32 seconds.'
      help_str(1) = 'The default values contain the optimum exposure time for each configuration.'
      help_str(2) = 'If CCD pixel summing has been selected from the camera parameters table'
      help_str(3) = 'then the default exposure times are adjusted automatically in the LEB.'
      POPUP_HELP, help_str, TITLE="EXPOSURE TABLE HELP"
   END

   exv.reset : BEGIN	;** load defaults
      ;** account for any summing for this tele,table configuration
      FOR fw = 0, 4 DO BEGIN
         FOR pw = 0, 4 DO BEGIN
            ex(exv.tele,exv.table,fw,pw) = exd(exv.tele,exv.table,fw,pw)
            WIDGET_CONTROL, exv.textw(fw,pw), SET_VALUE=STRTRIM(STRING(ex(exv.tele,exv.table,fw,pw)),2)
         ENDFOR
      ENDFOR
   END

   exv.telepd : BEGIN
      exv.tele = event.index
      FOR fw = 0, 4 DO BEGIN
         FOR pw = 0, 4 DO BEGIN
            WIDGET_CONTROL, exv.labelw(fw,pw), SET_VALUE=STRTRIM(STRMID(fw_types(fw,exv.tele),0,10),2)+', '+ $
                  STRMID(pw_types(pw,exv.tele),0,9)
            WIDGET_CONTROL, exv.textw(fw,pw), SET_VALUE=STRTRIM(STRING(ex(exv.tele,exv.table,fw,pw)),2)
         ENDFOR
      ENDFOR
   END

   exv.tablepd : BEGIN
      exv.table = event.index
      FOR fw = 0, 4 DO BEGIN
         FOR pw = 0, 4 DO BEGIN
            WIDGET_CONTROL, exv.textw(fw,pw), SET_VALUE=STRTRIM(STRING(ex(exv.tele,exv.table,fw,pw)),2)
         ENDFOR
      ENDFOR
   END

   ELSE : BEGIN 	;** must be changing an exposure value
      WIDGET_CONTROL, event.id, GET_UVALUE=uval
      fw = uval(0)
      pw = uval(1)
      exv.fw = fw
      exv.pw = pw
      WIDGET_CONTROL, exv.textw(fw,pw), GET_VALUE=val & val = val(0)
      temp = P_CHECK(val,'exp',new) & WIDGET_CONTROL, exv.textw(fw,pw), SET_VALUE=new
      ex(exv.tele,exv.table,fw,pw) = FIX(new)
      pw = (pw + 1) MOD 5
      IF (pw EQ 0) THEN fw = (fw + 1) MOD 5
      WIDGET_CONTROL, exv.textw(fw,pw), /INPUT_FOCUS
      WIDGET_CONTROL, exv.textw(fw,pw), GET_VALUE=val & val = val(0)
      WIDGET_CONTROL, exv.textw(fw,pw), SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(val),2))]
   END

ENDCASE

END

;________________________________________________________________________________________________________
;

PRO TABLE_EXPOSURE, caller
   COMMON OS_TABLE_EXPOSURE_SHARE
   COMMON OS_ALL_SHARE
   COMMON OS_INIT_SHARE

   IF XRegistered("TABLE_EXPOSURE") THEN RETURN

   IF ((SIZE(exv))(1) EQ 0) THEN BEGIN		 ;** not defined yet use default
      tele = 0		
      table = 0
   ENDIF ELSE BEGIN
      tele = exv.tele		
      table = exv.table
   ENDELSE

   base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT EXPOSURE TABLE TOOL', /FRAME, $
             GROUP_LEADER=caller.id)
   row = WIDGET_BASE(base, /ROW)
      telepd =  CW_BSELECTOR2(row, tele_types, SET_VALUE=tele)
      tablepd = CW_BSELECTOR2(row, table_types, SET_VALUE=table)
      reset = WIDGET_BUTTON(row, VALUE='    Load Defaults    ')
      helpb = WIDGET_BUTTON(row, VALUE='        Help         ')

   row = WIDGET_BASE(base, /ROW)
   col = LONARR(5)
   col(0) = WIDGET_BASE(row, /COLUMN, /FRAME, SPACE=0)
   col(1) = WIDGET_BASE(row, /COLUMN, /FRAME, SPACE=0)
   col(2) = WIDGET_BASE(row, /COLUMN, /FRAME, SPACE=0)
   col(3) = WIDGET_BASE(row, /COLUMN, /FRAME, SPACE=0)
   col(4) = WIDGET_BASE(row, /COLUMN, /FRAME, SPACE=0)
   labelw = LONARR(5,5)
   textw = LONARR(5,5)
   FOR fw = 0, 4 DO BEGIN
      FOR pw = 0, 4 DO BEGIN
         row = WIDGET_BASE(col(fw), /ROW, YPAD=0)
         label = WIDGET_LABEL(row, VALUE= STRTRIM(STRMID(fw_types(fw,tele),0,10),2)+', '+ $
                  STRMID(pw_types(pw,tele),0,9))
         text = WIDGET_TEXT(row, /EDITABLE, YSIZE=1, XSIZE=5, $
                  VALUE=STRTRIM(STRING(ex(tele,table,fw,pw)),2), UVALUE=[fw,pw])
         labelw(fw,pw) = label
         textw(fw,pw) = text
      ENDFOR
   ENDFOR

   dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")

   WIDGET_CONTROL, /REAL, base

   exv = CREATE_STRUCT( 'base', base,			$
                        'dismiss', dismiss,		$
                        'labelw', labelw,		$
                        'textw', textw,			$
                        'tablepd', tablepd,		$
                        'telepd', telepd,		$
                        'reset', reset,			$
                        'helpb', helpb,			$
 			'fw', fw,			$
 			'pw', pw,			$
 			'tele', tele,			$
			'table', table )

   XMANAGER, 'TABLE_EXPOSURE', base
END

;________________________________________________________________________________________________________
;

PRO TABLE_OCCULTER_EVENT, event
   COMMON OS_TABLE_OCCULTER_SHARE, base, dismiss
CASE (event.id) OF
   dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, base
   END
   ELSE : BEGIN
   END
ENDCASE
END

PRO TABLE_OCCULTER, caller
   COMMON OS_TABLE_OCCULTER_SHARE
   IF XRegistered("TABLE_OCCULTER") THEN RETURN
   base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT OCCULTER TABLE TOOL', /FRAME, $
             GROUP_LEADER=caller.id)
      temp = WIDGET_LABEL(base, VALUE='          Not Implemented Yet          ')
      dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")
   WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)
   XMANAGER, 'TABLE_OCCULTER', base
END

;________________________________________________________________________________________________________
;

PRO TABLE_MASK_EVENT, event
   COMMON OS_TABLE_MASK_SHARE, base, dismiss
CASE (event.id) OF
   dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, base
   END
   ELSE : BEGIN
   END
ENDCASE
END

PRO TABLE_MASK, caller
   COMMON OS_TABLE_MASK_SHARE
   IF XRegistered("TABLE_MASK") THEN RETURN
   base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT MASK TABLE TOOL', /FRAME, $
             GROUP_LEADER=caller.id)
      temp = WIDGET_LABEL(base, VALUE='          Not Implemented Yet          ')
      dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")
   WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)
   XMANAGER, 'TABLE_MASK', base
END

;________________________________________________________________________________________________________
;

PRO TABLE_BADCOL_EVENT, event
   COMMON OS_TABLE_BADCOL_SHARE, base, dismiss
CASE (event.id) OF
   dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, base
   END
   ELSE : BEGIN
   END
ENDCASE
END

PRO TABLE_BADCOL, caller
   COMMON OS_TABLE_BADCOL_SHARE
   IF XRegistered("TABLE_BADCOL") THEN RETURN
   base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT BADCOL TABLE TOOL', /FRAME, $
             GROUP_LEADER=caller.id)
      temp = WIDGET_LABEL(base, VALUE='          Not Implemented Yet          ')
      dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")
   WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)
   XMANAGER, 'TABLE_BADCOL', base
END

;________________________________________________________________________________________________________
;

PRO TABLE_TRANSIENT_EVENT, event
   COMMON OS_TABLE_TRANSIENT_SHARE, base, dismiss
CASE (event.id) OF
   dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, base
   END
   ELSE : BEGIN
   END
ENDCASE
END

PRO TABLE_TRANSIENT, caller
   COMMON OS_TABLE_TRANSIENT_SHARE
   IF XRegistered("TABLE_TRANSIENT") THEN RETURN
   base = WIDGET_BASE(/COLUMN, TITLE='LASCO/EIT TRANSIENT TABLE TOOL', /FRAME, $
             GROUP_LEADER=caller.id)
      temp = WIDGET_LABEL(base, VALUE='          Not Implemented Yet          ')
      dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")
   WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)
   XMANAGER, 'TABLE_TRANSIENT', base
END

;________________________________________________________________________________________________________
;

PRO TABLE_FP_WL_EXP_EVENT, event

COMMON OS_ALL_SHARE
COMMON OS_INIT_SHARE
COMMON OS_TABLE_FP_WL_EXP_SHARE, fpwlv

CASE (event.id) OF

   fpwlv.dismiss : BEGIN	;** exit program
      WIDGET_CONTROL, /DESTROY, fpwlv.base
   END

   fpwlv.wlstart : BEGIN
      WIDGET_CONTROL, fpwlv.wlstart, GET_VALUE=val & val = val(0)
      temp = P_CHECK(val,'wl',new) & WIDGET_CONTROL, fpwlv.wlstart, SET_VALUE=new
   END
   fpwlv.wldelta : BEGIN
      WIDGET_CONTROL, fpwlv.wldelta, GET_VALUE=val & val = val(0)
      temp = P_CHECK(val,'wl',new) & WIDGET_CONTROL, fpwlv.wldelta, SET_VALUE=new
   END

   fpwlv.modifywl : BEGIN	;** modify with start & delta
      fw = fpwlv.fw
      table = fpwlv.table
      WIDGET_CONTROL, fpwlv.wlstart, GET_VALUE=val & val = val(0)
      start = DOUBLE(val)
      WIDGET_CONTROL, fpwlv.wldelta, GET_VALUE=val & val = val(0)
      delta = DOUBLE(val)
      FOR i=0, 9 DO BEGIN
         fpwl(fpwlv.table,fpwlv.fw,i).wl = start+delta*i
         WIDGET_CONTROL, fpwlv.fpwlid(0,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).wl,FORMAT='(f9.4)'),2)
      ENDFOR
   END

   fpwlv.defaultb : BEGIN	;** load defaults
      fpwl = fpwld
      FOR i=0, 9 DO BEGIN
         WIDGET_CONTROL, fpwlv.fpwlid(0,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).wl,FORMAT='(f9.4)'),2)
         WIDGET_CONTROL, fpwlv.fpwlid(1,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).exp),2)
         WIDGET_CONTROL, fpwlv.fpwlid(2,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).order),2)
      ENDFOR
   END

   fpwlv.helpb : BEGIN	;** display help
      help_str = STRARR(10)
      help_str(0) = 'This tool defines the Fabry Perot Wavelength Exposure Tables.'
      help_str(1) = 'There are three tables and five filter positions per table.'
      help_str(2) = 'There can be up to 10 Wavelength Exposure pairs for every table.'
      help_str(3) = 'Exposure time units are 1/32 seconds.'
      POPUP_HELP, help_str, TITLE="FP WL EXP TABLE HELP"
   END

   fpwlv.fwpd : BEGIN
      fpwlv.fw = event.index
      WIDGET_CONTROL, fpwlv.wlstart, SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,0).wl,FORMAT='(f9.4)'),2)
      FOR i=0, 9 DO BEGIN
         WIDGET_CONTROL, fpwlv.fpwlid(0,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).wl,FORMAT='(f9.4)'),2)
         WIDGET_CONTROL, fpwlv.fpwlid(1,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).exp),2)
         WIDGET_CONTROL, fpwlv.fpwlid(2,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).order),2)
      ENDFOR
   END

   fpwlv.tablepd : BEGIN
      fpwlv.table = event.index
      WIDGET_CONTROL, fpwlv.wlstart, SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,0).wl,FORMAT='(f9.4)'),2)
      FOR i=0, 9 DO BEGIN
         WIDGET_CONTROL, fpwlv.fpwlid(0,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).wl,FORMAT='(f9.4)'),2)
         WIDGET_CONTROL, fpwlv.fpwlid(1,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).exp),2)
         WIDGET_CONTROL, fpwlv.fpwlid(2,i), SET_VALUE=STRTRIM(STRING(fpwl(fpwlv.table,fpwlv.fw,i).order),2)
      ENDFOR
   END

   ELSE : BEGIN		;** must be entering a new wl or exp or order
      WIDGET_CONTROL, event.id, GET_UVALUE=uval
      WIDGET_CONTROL, event.id, GET_VALUE=val & val = val(0)
      ind1 = uval(0)	;** 0:wl, 1:exp, 2:order
      ind2 = uval(1)	;** entry number in table 0-9
      CASE (ind1) OF
         0 : BEGIN
               temp = P_CHECK(val,'wl',new) & WIDGET_CONTROL, event.id, SET_VALUE=new
               fpwl(fpwlv.table,fpwlv.fw,ind2).wl = DOUBLE(new)
             END
         1 : BEGIN
               temp = P_CHECK(val,'wlexp',new) & WIDGET_CONTROL, event.id, SET_VALUE=new
               fpwl(fpwlv.table,fpwlv.fw,ind2).exp = FIX(new)
             END
         2 : BEGIN
               temp = P_CHECK(val,'order',new) & WIDGET_CONTROL, event.id, SET_VALUE=new
               fpwl(fpwlv.table,fpwlv.fw,ind2).order = FIX(new)
             END
      ENDCASE
      ind = WHERE(fpwlv.fpwlid EQ event.id)	;** put focus on next var
      ind = (ind + 1) MOD 30
      WIDGET_CONTROL, fpwlv.fpwlid(ind), /INPUT_FOCUS
      WIDGET_CONTROL, fpwlv.fpwlid(ind), GET_VALUE=val & val = val(0)
      WIDGET_CONTROL, fpwlv.fpwlid(ind), SET_TEXT_SELECT=[STRLEN(STRTRIM(STRING(val),2))]
   END

ENDCASE

END

;________________________________________________________________________________________________________
;

PRO TABLE_FP_WL_EXP, caller

COMMON OS_ALL_SHARE
COMMON OS_TABLE_FP_WL_EXP_SHARE
COMMON OS_INIT_SHARE

   IF XRegistered("TABLE_FP_WL_EXP") THEN RETURN

   IF ((SIZE(fpwlv))(1) EQ 0) THEN BEGIN		 ;** not defined yet use default
      fw = 0		
      table = 0
   ENDIF ELSE BEGIN
      fw = fpwlv.fw		
      table = fpwlv.table
   ENDELSE

    ;********************************************************************
    ;** SET UP WIDGETS **************************************************

    base = WIDGET_BASE(/COLUMN, TITLE='FP WL EXPOSURE TABLE', /FRAME, GROUP_LEADER=caller.id)

      row = WIDGET_BASE(base, /ROW)
        tablepd = CW_BSELECTOR2(row, table_types, SET_VALUE=table)
        fwpd =  CW_BSELECTOR2(row, fw_types(*,0), SET_VALUE=fw)
        helpb = WIDGET_BUTTON(row, VALUE='Help')
        defaultb = WIDGET_BUTTON(row, VALUE='Load Defaults')
      row = WIDGET_BASE(base, /ROW)
        temp = WIDGET_LABEL(row, VALUE='Start WL:')
        wlstart = WIDGET_TEXT(row, VALUE=STRTRIM(STRING(fpwl(table,fw,0).wl, FORMAT='(f9.4)'),2), XSIZE=9, $
                  /EDITABLE)
        temp = WIDGET_LABEL(row, VALUE='Delta:')
        wldelta = WIDGET_TEXT(row, VALUE=STRTRIM(STRING(0.076, FORMAT='(f9.4)'),2), XSIZE=9, $
                  /EDITABLE)
        modifywl = WIDGET_BUTTON(row, VALUE='Apply Start & Delta')
      row = WIDGET_BASE(base, /ROW)
        col = LONARR(2)
        col(0) = WIDGET_BASE(row, /COLUMN, SPACE=0, XPAD=20, YPAD=0)
        col(1) = WIDGET_BASE(row, /COLUMN, SPACE=0, XPAD=20, YPAD=0)
        fpwlid = LONARR(3,10)		;** text widget ids, 3 vars (wl,exp,order); 10 entries
        i=0
        FOR c=0,1 DO BEGIN
          FOR r=0,4 DO BEGIN
             line = WIDGET_BASE(col(c), /ROW, YPAD=0)
             temp = WIDGET_LABEL(line, VALUE='     FP WL '+STRTRIM(STRING(i),2)+':')
             fpwlid(0,i) = WIDGET_TEXT(line, VALUE=STRTRIM(STRING(fpwl(table,fw,i).wl, FORMAT='(f9.4)'),2), XSIZE=9, $
                  UVALUE=[0,i], /EDITABLE)
             line = WIDGET_BASE(col(c), /ROW, YPAD=0)
             temp = WIDGET_LABEL(line, VALUE='  Exposure '+STRTRIM(STRING(i),2)+':')
             fpwlid(1,i) = WIDGET_TEXT(line, VALUE=STRTRIM(STRING(fpwl(table,fw,i).exp),2), XSIZE=9, $
                  UVALUE=[1,i], /EDITABLE)
             line = WIDGET_BASE(col(c), /ROW, YPAD=0)
             temp = WIDGET_LABEL(line, VALUE='     Order '+STRTRIM(STRING(i),2)+':')
             fpwlid(2,i) = WIDGET_TEXT(line, VALUE=STRTRIM(STRING(fpwl(table,fw,i).order),2), XSIZE=9, $
                  UVALUE=[2,i], /EDITABLE)
             spacer = WIDGET_LABEL(col(c), VALUE='    ')
             i=i+1
          ENDFOR
        ENDFOR

      dismiss = WIDGET_BUTTON(base, VALUE=" Dismiss ")

    ;********************************************************************
    ;** REALIZE THE WIDGETS *********************************************

    WIDGET_CONTROL, /REAL, base
    WIDGET_CONTROL, base, TLB_SET_XOFFSET=X0(base), TLB_SET_YOFFSET=Y0(base)

    fpwlv = CREATE_STRUCT('base', base,			$
                         'dismiss', dismiss,		$
                         'fwpd', fwpd,			$
                         'tablepd', tablepd,		$
                         'helpb', helpb,		$
                         'defaultb', defaultb,		$
                         'fpwlid', fpwlid,		$
                         'table', table,		$
                         'modifywl', modifywl,		$
                         'wlstart', wlstart,		$
                         'wldelta', wldelta,		$
                         'fw', fw)

    XMANAGER, 'TABLE_FP_WL_EXP', base

END

;________________________________________________________________________________________________________
;

PRO OS_TABLES
END
