;+
; Project     : SOHO - LASCO/EIT
;                   
; Name        : SCHEDULE
;               
; Purpose     : Widget tool to schedule a LASCO/EIT Observing Sequence.
;               
; Explanation : This is the top level tool for defining the LASCO/EIT
;		observing schedule.  From this tool you can define and
;		schedule OP's and OS's.  A graphical representation of
;		the schedule is displayed including the expected
;		LEB telemetry buffer usage.
;               
; Use         : SCHEDULE, ops
;    
; Inputs      : None.
;               
; Opt. Inputs : ops	Start time of display in TAI.
;               
; Outputs     : None.
;               
; Opt. Outputs: None.
;               
; Keywords    : None.
;
; Calls       : GET_UTC, UTC2TAI, NINT, TAI2UTC, CW_BSELECTOR2, INSTRUCT, 
;		FAKE_OP, FAKE_OS, SCHEDULE_PLOT, SCHEDULE_EVENT
;
; Common      : None.
;               
; Restrictions: None.
;               
; Side effects: None.
;               
; Category    : Planning, Scheduling.
;               
; Prev. Hist. : Widget layout adapted from SOHO/CDS Planning Tool.
;
; Written     : Scott Paswaters, NRL, November 1994.
;               
; Modified    : 
;
; Version     : Version 0.1, December 5, 1994
;-            

;__________________________________________________________________________________________________________
;
PRO SCHEDULE, ops


 ;
 ; The variable STARTDIS is used throughout the code.  It is the first time point 
 ; displayed in the draw window and is in TAI form.  Likewise, ENDDIS
 ; is the last displayed time point.
 ;

 read_ipt_flag = ''
 IF N_ELEMENTS(ops) EQ 0 THEN BEGIN	; no time entered, use default
      GET_UTC, stime			; get system time
      stime.mjd = stime.mjd + 1		; add a day
      stime.time = 0			; only want date
      startdis = UTC2TAI(stime)		; seconds since 1 jan 1958 (TAI)
 ENDIF ELSE BEGIN
   pos = STRPOS(ops, '.IPT')
   IF (pos GT 0) THEN BEGIN	;** read in input IPT
    read_ipt_flag = ops
    SCHEDULE_READ_IPT, ops, /SILENT, startdis=startdis
   ENDIF ELSE $
      startdis = UTC2TAI(ops)     ; Hopefully, the user entered a valid time string so get in TAI 
 ENDELSE

 baseop = startdis 			; Remember input operational day (TAI),
				        ;    STARTDIS will be changing.

 duration = 86400d   			; Default display span in seconds
 enddis   = startdis + duration 	; The last displayed time point (TAI)




 ;
 ; Initialize some VALUEs
 ;

 study_start = startdis		; STUDY_START: VALUE to go in start text widget
 study_stop  = enddis		; STUDY_STOP : VALUE to go in stop text widget


 mode = 0 			; Initialize to display mode
				;    mode = 0 is display mode
				;    mode = 1 is edit mode

 ;
 ;
 ; Create the widget interface
 ;
 ;

 base = WIDGET_BASE(TITLE='LASCO/EIT Scheduling Tool   /   ' + $
			  'Operational Day ' + TAI2UTC(baseop, /ECS), $
		    SPACE=10, XPAD=20, YPAD=20, /ROW)


  ; 
  ; The COLUMN of button, text, and list widgets to the left of the DRAW window
  ;

  col1 = WIDGET_BASE(base, SPACE=25, /COLUMN)


  ; 
  ; The row of buttons at the top of the left side COLUMN
  ;

  controls =  WIDGET_BASE(col1, /ROW)
     quit = WIDGET_BUTTON(controls, VALUE='QUIT', UVALUE='QUIT')
;     help = WIDGET_BUTTON(controls, VALUE='Help', UVALUE='HELP')
     gen  = WIDGET_BUTTON(controls, VALUE='Verify Schedule', UVALUE='VERIFY')

     ; 
     ; To add more time spans, add the VALUEs below and provide an appropiate
     ; case statement in the event handler (PLAN_EV)
     ;

     XPDMENU, ['"Time Span" {', $
			'"1 hour span  " 1HR', $
			'"3 hour span  " 3HRS', $
			'"6 hour span  " 6HRS', $
			'"12 hour span " 12HRS',$
                        '"1 day span   " 1DAY', $
                        '"1.5 day span " 1_5DAY', $
                        '"2 day span   " 2DAY', $
                        '"3 day span   " 3DAY', $
                        '"5 day span   " 5DAY', $
                        '"7 day span   " 7DAY', $
		'}' ], controls


  ; 
  ; Below the control button row, these are buttons to call outside programs.
  ;

  refer = WIDGET_BASE(col1, SPACE=5, /FRAME, /COLUMN)
   prog = WIDGET_BASE(refer, SPACE=5, /COLUMN, XPAD=10, YPAD=5)
     temp = WIDGET_BUTTON(prog, VALUE='Define an Observing Program  (OP)', EVENT_PRO='PLAN_OP')
     temp = WIDGET_BUTTON(prog, VALUE='Define an Observing Sequence (OS)', UVALUE='DEFINE_OS')

  refer = WIDGET_BASE(col1, SPACE=5, /FRAME, /COLUMN)
   prog = WIDGET_BASE(refer, SPACE=5, /COLUMN, XPAD=10, YPAD=5)
     temp = WIDGET_BUTTON(prog, VALUE='     Store Current Schedule      ', UVALUE='STORE')
     temp = WIDGET_BUTTON(prog, VALUE='Generate Plans File for Commanding', UVALUE='OUTPUT_PLAN')
     temp = WIDGET_BUTTON(prog, VALUE='        Generate IAP File        ', UVALUE='OUTPUT_IAP')
     temp = WIDGET_BUTTON(prog, VALUE='          Read EAP File          ', UVALUE='INPUT_KAP')
     temp = WIDGET_BUTTON(prog, VALUE='   Generate Buffer Usage File    ', UVALUE='OUTPUT_BUF')
     ;temp = WIDGET_BUTTON(prog, VALUE="       Delete Multiple OS's      ", UVALUE='DELETE_OSs')
     temp = WIDGET_BUTTON(prog, VALUE="          Read IPT File          ", UVALUE='INPUT_IPT')


  ;
  ; Below the outside call buttons, there are 2 text window for entering
  ; start and stop times and a list widget that displays study names. 
  ;

  edit = WIDGET_BASE(col1, SPACE=5, /COLUMN, /FRAME)

     ; In order to line up text widgets, pair the label in a COLUMN
     ; and the text windows in a COLUMN.

     row = WIDGET_BASE(edit, /ROW)
       get_os_op = CW_BSELECTOR2(row, ["SELECT OS","SELECT OP"], SET_VALUE=0, UVALUE="GET_OS_OP")
       num_text = WIDGET_LABEL(row, VALUE='  OS_NUM:     0  ')

     times_input = WIDGET_BASE(edit, /ROW)
      lab_times = WIDGET_BASE(times_input, /COLUMN, SPACE=20)
       lab = WIDGET_LABEL(lab_times, VALUE='START')
       lab = WIDGET_LABEL(lab_times, VALUE='COUNT')
       lab = WIDGET_LABEL(lab_times, VALUE='DELTA')
       lab = WIDGET_LABEL(lab_times, VALUE='STOP')

      tex_times = WIDGET_BASE(times_input,/COLUMN)
       start_text = WIDGET_TEXT(tex_times, XSIZE=22, /EDITABLE, $
				 UVALUE='TEXTSTART')
       count_text = WIDGET_TEXT(tex_times, XSIZE=5, /EDITABLE, UVALUE='COUNT')
       delta_text = WIDGET_TEXT(tex_times, XSIZE=11, /EDITABLE, UVALUE='DELTA')
       stop_text = WIDGET_TEXT(tex_times, XSIZE=22, /EDITABLE, $
				 UVALUE='TEXTSTOP')
      lab = WIDGET_LABEL(times_input, VALUE='')

      os_label = WIDGET_BASE(edit, /ROW)
       col = WIDGET_BASE(os_label, /COLUMN, SPACE=20)
         lab = WIDGET_LABEL(col, VALUE='SIZE')
         lab = WIDGET_LABEL(col, VALUE='DURATION')
       col = WIDGET_BASE(os_label, /COLUMN)
         new_value = 0D
         os_size_text = WIDGET_TEXT(col, XSIZE=11, /EDITABLE, VALUE=STRING(new_value,FORMAT='(d10.1)'), $
                        UVALUE='OS_SIZE')
         os_dur_text = WIDGET_TEXT(col, XSIZE=11, /EDITABLE, VALUE=STRING(new_value,FORMAT='(d6.1)'), $
                        UVALUE='OS_DUR')
       col = WIDGET_BASE(os_label, /COLUMN)
         bits_bytes = CW_BSELECTOR2(col, ["bits ","bytes","% buffer"], SET_VALUE=0, UVALUE="BITS_BYTES")
         secs_mins  = CW_BSELECTOR2(col, ["secs ","mins "], SET_VALUE=0, UVALUE="SECS_MINS")
      

     ;
     ; Near the bottom of the left side are some editing control buttons.
     ;

     edit_study  = WIDGET_BASE(edit, /COL); XPAD=10, SPACE=10)
      row  = WIDGET_BASE(edit_study, /ROW); XPAD=10, SPACE=10)
        tmp = WIDGET_BUTTON(row, VALUE='ADD',  UVALUE='ADD')
	tmp = WIDGET_BUTTON(row, VALUE='DELETE', UVALUE='DELETE')
	tmp = WIDGET_BUTTON(row, VALUE="DELETE MULTIPLE OS's", UVALUE='DELETE_OSs')
      row  = WIDGET_BASE(edit_study, /ROW); XPAD=10, SPACE=10)
	tmp = WIDGET_BUTTON(row, VALUE='QUICK SUMMARY', UVALUE='SUMMARY')
	tmp = WIDGET_BUTTON(row, VALUE='DETAILS', UVALUE='DETAILS')

  ;
  ; On the right-hand side of the interface is the display window.
  ;

  plan = WIDGET_BASE(base, /COLUMN)

     ;
     ; The draw widget shows the studies from STARTDIS to ENDDIS
     ;

     draw = WIDGET_DRAW(plan, XSIZE=700, YSIZE=512*2, /BUTTON_EVENT, /SCROLL, Y_SCROLL_SIZE=512, $
			      X_SCROLL_SIZE=700, RETAIN=2, /FRAME, UVALUE='DRAW') 


     ;
     ; Below the draw window are the time buttons.
     ;   In display mode, they modify the STARTDIS VALUE by the specified time
     ;   In edit mode, they modity the times in the start and stop text window.
     ;

     jump = WIDGET_BASE(plan, /ROW, /FRAME)

      jump1 = WIDGET_BASE(jump, /ROW)
;   	bck6 = WIDGET_BUTTON(jump1, VALUE=' <year',  UVALUE='BACKYR')
   	bck5 = WIDGET_BUTTON(jump1, VALUE=' <mon',   UVALUE='BACKMON')
   	bck4 = WIDGET_BUTTON(jump1, VALUE=' <day  ', UVALUE='BACKDAY')
   	bck3 = WIDGET_BUTTON(jump1, VALUE=' <hour ', UVALUE='BACKHR')
   	bck2 = WIDGET_BUTTON(jump1, VALUE=' <min ',  UVALUE='BACKMIN')
   	bck1 = WIDGET_BUTTON(jump1, VALUE=' <sec ',  UVALUE='BACKSEC')

      movetype = WIDGET_BASE(jump, /FRAME, /COLUMN)
	;
	; These buttons allow the user to select the current mode,
	; either display mode or edit mode.  The resulting action 
	; of many widgets is determined by this current mode.
	;

        arrow = WIDGET_BASE(movetype, /EXCLUSIVE, /COLUMN)
	    ; DO NOT use the /NO_RELEASE keyword, button setting control
	    ; is performed in the event handler.
            arr_span = WIDGET_BUTTON(arrow, VALUE='Display', UVALUE='ARROW_DIS')
            arr_edit = WIDGET_BUTTON(arrow, VALUE='Edit',  UVALUE='ARROW_EDIT')

      jump2 = WIDGET_BASE(jump, /ROW)
   	for1 = WIDGET_BUTTON(jump2, VALUE='  sec> ',  UVALUE='FORSEC')
   	for2 = WIDGET_BUTTON(jump2, VALUE='  min> ',  UVALUE='FORMIN')
   	for3 = WIDGET_BUTTON(jump2, VALUE='  hour> ', UVALUE='FORHR')
   	for4 = WIDGET_BUTTON(jump2, VALUE='  day>  ', UVALUE='FORDAY')
   	for5 = WIDGET_BUTTON(jump2, VALUE='  mon>',   UVALUE='FORMON')
;   	for6 = WIDGET_BUTTON(jump2, VALUE='  year>',  UVALUE='FORYR')



     rrow = WIDGET_BASE(plan, /ROW)

     mark = WIDGET_BASE(rrow, /COL, /FRAME)
      lab = WIDGET_LABEL(mark, VALUE='MARK OS INSTANCE FOR:')
      row  = WIDGET_BASE(mark, /ROW, XPAD=10, SPACE=10)
       col  = WIDGET_BASE(row, /COL)
        tmp = WIDGET_BUTTON(col, VALUE='(c) CAMERA TABLE UPLOAD',  UVALUE='CAMTABLE')
        tmp = WIDGET_BUTTON(col, VALUE='(l) WAVELEN/EXP TAB UPLD',  UVALUE='WLEXPTABLE')
       col  = WIDGET_BASE(row, /COL)
        tmp = WIDGET_BUTTON(col, VALUE='(*)  FP CHECK & CORRECT',  UVALUE='CANDC')
        tmp = WIDGET_BUTTON(col, VALUE='(fp) FP MOVE TO WAVELENGTH',  UVALUE='MOVEFP')

  ; below this is buffer statistics
  buffstats = WIDGET_BASE(rrow, SPACE=5, /COLUMN, /FRAME)
     lab = WIDGET_LABEL(buffstats, VALUE='Telemetry Statistics')
     date_str = UTC2STR(TAI2UTC(startdis), /ECS, /TRUNCATE) + ' to ' + $
                UTC2STR(TAI2UTC(enddis), /ECS, /TRUNCATE)
     usage_dates = WIDGET_LABEL(buffstats, VALUE=date_str)
     lab = WIDGET_LABEL(buffstats, VALUE='Total Usage    C1      C2      C3     EIT')
     usage_stats = WIDGET_LABEL(buffstats, VALUE='   0.0%      0.0%   0.0%   0.0%   0.0%   ')

  
 WIDGET_CONTROL, base, /REALIZE		; bring widget to life
 WIDGET_CONTROL, /HOUR

 ;
 ; Set some initial widget conditions
 ;

 WIDGET_CONTROL, start_text, SET_VALUE=TAI2UTC(startdis, /ECS)	; write current
 WIDGET_CONTROL, stop_text,  SET_VALUE=TAI2UTC(enddis,   /ECS)	;    time range

 WIDGET_CONTROL, start_text, /INPUT_FOCUS	; input focus at start time

 WIDGET_CONTROL, arr_span, /SET_BUTTON		; initially in display mode
 WIDGET_CONTROL, edit_study, SENSITIVE=0	; display mode has no editting
 WIDGET_CONTROL, os_label, SENSITIVE=0		; start out in OP mode
; WIDGET_CONTROL, count_text, SENSITIVE=0	; start out in OP mode
; WIDGET_CONTROL, delta_text, SENSITIVE=0	; start out in OP mode


 WIDGET_CONTROL, draw, GET_VALUE=win_index	; find out window's index no.
 WIDGET_CONTROL, draw, SET_DRAW_VIEW=[0,0]	; set viewport to bottom left
 WSET, win_index                                ; set out to display window

 ;** set up color table
 LOADCT, 0
 TVLCT, 255,0,0, 1	;** load red into entry 1
 TVLCT, 0,255,0, 2	;** load green into entry 2
 TVLCT, 142,229,238, 3	;** load blue into entry 3
 TVLCT, 255,255,0,4	;** load yellow into entry 4
 TVLCT, 200,0,0, 5	;** load dull red into entry 5
 TVLCT, 0,200,0, 6	;** load dull green into entry 6
 TVLCT, 0,206,237, 7	;** load dull blue into entry 7
 TVLCT, 200,200,0,8	;** load dull yellow into entry 8
 TVLCT, 142,229,238, 9  ;** load light blue into entry 9 - used by mask.pro

 ;
 ; Draw all instruments' plans for the input operational day
 ;
 
 FAKE_OP, startdis
 IF (read_ipt_flag EQ '') THEN FAKE_OS, startdis
 READ_KAP, startdis, enddis, /DEFAULT
 charsize=1.2
 pct_start = 50


 ;
 ; Create a structure that will be carried around as the UVALUE of the main base (BASE)
 ; 

 schedv = { $
	base:base,  	    	    $	; widget ID for the base widget
	arr_span:arr_span,  	    $	; widget ID for the DISPLAY MODE button
	arr_edit:arr_edit,  	    $	; widget ID for the PLAN MODE button
	win_index:win_index,  	    $	; window number for the draw window
	x:!x,  	    		    $	; !X system var for schedule plot window
	y:!y,  	    		    $	; !Y system var for schedule plot window
	p:!p,  	    		    $	; !P system var for schedule plot window
	baseop:baseop,		    $   ; the user entered ops day (TAI)
	startdis:startdis, 	    $	; Time of first display data pt. (TAI)
	enddis:enddis,   	    $	; Last display point time (TAI)
	duration:duration,	    $	; How many seconds to display 
	charsize:charsize,	    $	; charsize
	num_text:num_text,     	    $	; widget ID for the OS_NUM or OP_NUM label
	start_text:start_text,      $	; widget ID for the start time text
	stop_text:stop_text,  	    $	; widget ID for the stop time text
	last_text:stop_text,  	    $	; widget ID for last time
	count_text:count_text, 	    $	; widget ID for the count text
	delta_text:delta_text, 	    $	; widget ID for the delta time text
	edit_study:edit_study,      $	; widget ID for the parent of the SCHEDULE & READY buttons
	mode:mode,  		    $	; the current mode (display or edit)
	study_start:study_start,    $	; start time of a study in plan mode
	study_stop:study_stop,      $	; end time of a study in plan mode
	os_label:os_label,          $   ; widget ID for the parent of the OS_DURATION & OS_SIZE labels
	os_size_text:os_size_text,  $   ; widget ID for the OS_SIZE label
	os_dur_text:os_dur_text,    $   ; widget ID for the OS_DURATION label
	os_size_units:0,            $   ; for converting os_size bits to desired units (default bits)
	os_dur_factor:1.0D,         $   ; factor for converting os_dur secs to desired units (default secs)
	bits_bytes:bits_bytes,      $   ; widget ID for the bits_bytes button
	secs_mins:secs_mins,        $   ; widget ID for the secs_mins button
	pct_start:pct_start,        $   ; percentage of buffer at start of schedule
	usage_dates:usage_dates,    $   ; widget ID for the buffer usage date label
	usage_stats:usage_stats     $   ; widget ID for the buffer usage stats label
       }

 SCHEDULE_PLOT, schedv

 ;
 ; Load the structure into the main base's UVALUE for one way trip to handler
 ;

 WIDGET_CONTROL, base, SET_UVALUE=schedv

 ;
 ; Call the XMANAGER to take control until there is an event
 ;

 XMANAGER, 'SCHEDULE', base, EVENT_HANDLER='SCHEDULE_EVENT'

RETURN
END
