;+
; Project     : SOHO - LASCO/EIT
;                   
; Name        : SCHEDULE_PLOT
;               
; Purpose     : Plot schedule in LASCO/EIT Scheduling tool's plot window.
;               
; Explanation : This routine plots the current: DSN contact, TM Submode,
;		OP's, OS's, and LEB Buffer in the Scheduling tool's
;		output window.
;               
; Use         : SCHEDULE_PLOT, startdis, enddis, charsize
;    
; Inputs      : startdis        Start of display time in TAI.
;               enddis        	End of display time in TAI.
;               
; Opt. Inputs : charsize	Size of characters to write.
;               
; Outputs     : None.
;               
; Opt. Outputs: None.
;               
; Keywords    : None.
;
; Calls       : CDS time conversion routines.
;
; Called From : SCHEDULE, SCHEDULE_EVENT
;
; Common      : OP_SCHEDULED	Defined in FAKE_OP.PRO
;		OP_DEFINED	Defined in FAKE_OP.PRO
;		OS_SCHEDULED 	Defined in FAKE_OS.PRO
;		OS_DEFINED 	Defined in OS_INIT.PRO
;		OBS_PROG_DEF	Defined in FAKE_OP.PRO
;		OS_PLOT_SHARE	Defined in SCHEDULE_PLOT.PRO
;               
; Restrictions: None.
;               
; Side effects: None.
;               
; Category    : Planning, Scheduling.
;               
; Prev. Hist. : Built on SOHO/CDS Planning Tool.
;
; Written     : Scott Paswaters, NRL, November 1994.
;               
; Modified    : 
;
; Version     : Version 0.1, December 5, 1994
;-            

;__________________________________________________________________________________________________________
;
PRO SCHEDULE_PLOT, schedv

COMMON OP_SCHEDULED
COMMON OP_DEFINED
COMMON OS_SCHEDULED
COMMON OS_DEFINED
COMMON OBS_PROG_DEF

COMMON OS_PLOT_SHARE, os_yarr

COMMON KAP_INIT
COMMON KAP_INPUT

COMMON SCHEDULE_BUFFER, save_buffer

startdis = schedv.startdis
enddis = schedv.enddis
charsize = schedv.charsize

WSET, schedv.win_index

;
;  Get the character size.
;
   datastart = TAI2UTC(startdis)                ; form .DAY and .TIME
   ; START_HOUR  Start of display plot, in # hours into the STARTDIS day
   ; STOP_HOUR   End of display plot in # hours since STARTDIS (maybe >24)
 
   t1 = datastart.time / 3600000.       ; start display at this hour
   t2  = t1 + ((enddis - startdis) / 3600d) ; end display hour
 
   datastart.time = 0                           ; only want day part
   daystart = UTC2TAI(datastart)                ; start of STARTDIS in TAI

	IF N_ELEMENTS(charsize) EQ 0 THEN BEGIN
		IF !P.CHARSIZE GT 0 THEN charsize = !P.CHARSIZE ELSE	$
			charsize = 1
	ENDIF

   ; draw window will contain two plots
   !P.MULTI = [0,0,2,0,0]

;--------  FIRST PLOT ---------------
;
;  Make the plot frame, but don't plot the X axis.
;
   !Y.MARGIN = [2, 2]	;** bottom and top margin in chars
        rows = 100	;** yrange [0,100]
	IF N_ELEMENTS(xtitle) EQ 0 THEN xtitle=TAI2UTC(startdis, /ECS) ELSE xtitle = xtitle
	PLOT,[t1,t2],[0,rows-1],/NODATA,XSTYLE=5,YSTYLE=1,YTICKS=1,	$
		YTICKNAME=REPLICATE(' ',2),XMARGIN=[15,4],CHARSIZE=charsize, $
		XTICK_GET=XTICKS,PSYM=0,LINESTYLE=0, TITLE=xtitlE
;
;  Plot the X axis with time format labels.
;
	HOURS = FIX(XTICKS) MOD 24
	W = WHERE(HOURS LT 0, COUNT)
	IF COUNT GT 0 THEN HOURS(W) = HOURS(W) + 24
	MINUTES = 60*(XTICKS - FIX(XTICKS))
	W = WHERE(MINUTES LT 0, COUNT)
	IF COUNT GT 0 THEN BEGIN
		HOURS(W) = HOURS(W) - 1
		MINUTES(W) = MINUTES(W) + 60
	ENDIF
	W = WHERE(MINUTES GE 60, COUNT)
	IF COUNT GT 0 THEN BEGIN
		HOURS(W) = HOURS(W) + 1
		MINUTES(W) = MINUTES(W) - 60
	ENDIF
	HOURS = STRMID(STRING(100+HOURS,FORMAT='(I3)'),1,2)
	MINUTES = STRMID(STRING(100+FIX(MINUTES+0.5),FORMAT='(I3)'),1,2)
	AXIS,XAXIS=0,XSTYLE=1,XTICKNAME=HOURS+':'+MINUTES,CHARSIZE=charsize
	AXIS,XAXIS=1,XSTYLE=1,XTICKNAME=REPLICATE(' ',30)


	dy = !D.Y_CH_SIZE * charsize / (3 * !Y.S(1) * !D.Y_SIZE)

;
;  Draw the line for DSN & Plot.
;
	OPLOT,[t1,t2],[95.0,95.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,97.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'DSN  '
        id = WHERE(kap_resource_names EQ "DSN_CONTACT") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        col_ind = [255, 170]
        IF (ind(0) GE 0) THEN BEGIN
           dsn_contact = kap_resource_arr(ind)
           ind = SORT([dsn_contact.startime])	;** sort by start time
           dsn_contact = dsn_contact(ind)
           FOR i = 0,N_ELEMENTS(dsn_contact)-1 DO BEGIN
              x1 = (dsn_contact(i).startime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x2 = (dsn_contact(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = 95
              y2 = y1 + 1.5		;** 1.2 is thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2], COLOR=col_ind(i MOD 2)
              ENDIF
           ENDFOR
        ENDIF

;
;  Draw the lines for TM MODE & Plot.
;
	OPLOT,[t1,t2],[87.0,87.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,91.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'TM Mode     '
	XYOUTS,t1,88.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'Idle'
	OPLOT,[t1,t2],[89.0,89.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,90.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'LR'
	OPLOT,[t1,t2],[91.0,91.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,92.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'MR'
	OPLOT,[t1,t2],[93.0,93.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,94.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'HR'

        mins = ((enddis - startdis) / 60D)+1	;** minutes in plotting range
        tmmode = INTARR(mins)		; tm mode array 1 element per minute of plotting window
        id = WHERE(kap_resource_names EQ "TLM_MODE") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           tlm_mode = kap_resource_arr(ind)
           ind = SORT([tlm_mode.startime])	;** sort by start time
           tlm_mode = tlm_mode(ind)
           FOR i = 0,N_ELEMENTS(tlm_mode)-1 DO BEGIN
              x1 = (tlm_mode(i).startime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              IF (i EQ N_ELEMENTS(tlm_mode)-1) THEN $
                 x2 = !X.CRANGE(1) $				;** don't know next mode, cont this 1 to end
              ELSE $
                 x2 = (tlm_mode(i+1).startime - daystart) / 3600D
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = (tlm_mode(i).type)*2 + 87	;** 87 is where TM plotting region starts + 2 units high
              y2 = y1 + 1.5		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF

              ;** save array of modes during this plotting range (1 element per minute)
              tm0 = (x1*60D) - (!X.CRANGE(0) * 60D)
              tm1 = (x2*60D) - (!X.CRANGE(0) * 60D)
              IF (tm1 GT tm0) THEN tmmode(tm0:tm1-1) = tlm_mode(i).type
           ENDFOR
        ENDIF

;
;  Draw the lines for Tape Dump & Plot.
;
	OPLOT,[t1,t2],[75.0,75.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,81.0-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'Tape Dump '
        id = WHERE(kap_resource_names EQ "TLM_TAPE_DUMP") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           tape_dump = kap_resource_arr(ind)
           ind = SORT([tape_dump.startime])	;** sort by start time
           tape_dump = tape_dump(ind)
           FOR i = 0,N_ELEMENTS(tape_dump)-1 DO BEGIN
              x1 = (tape_dump(i).startime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x2 = (tape_dump(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = 81
              y2 = y1 + 1.2		;** 1.2 is thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF

;
;  Draw the lines for MDI_M and MDI_H & Plot.
;
	OPLOT,[t1,t2],[65.0,65.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,70.0-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'MDI     '
	XYOUTS,t1,67.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'M '
	OPLOT,[t1,t2],[70.0,70.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,72.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'H '
        id = WHERE(kap_resource_names EQ "TLM_MDI") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           tlm_mdi = kap_resource_arr(ind)
           ind = SORT([tlm_mdi.startime])	;** sort by start time
           tlm_mdi = tlm_mdi(ind)
           FOR i = 0,N_ELEMENTS(tlm_mdi)-1 DO BEGIN
              x1 = (tlm_mdi(i).startime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x2 = (tlm_mdi(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = 65
              IF (tlm_mdi(i).type EQ 1) THEN y1 = y1+5
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF

;
;  Draw the lines for TLM_HR_IDLE & Plot.
;
	OPLOT,[t1,t2],[60.0,60.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,62.5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'TLM HR Idle '
        id = WHERE(kap_resource_names EQ "TLM_HR_IDLE") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           tlm_hr_idle = kap_resource_arr(ind)
           ind = SORT([tlm_hr_idle.startime])	;** sort by start time
           tlm_hr_idle = tlm_hr_idle(ind)
           FOR i = 0,N_ELEMENTS(tlm_hr_idle)-1 DO BEGIN
              x1 = (tlm_hr_idle(i).startime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x2 = (tlm_hr_idle(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = 60
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF

;
;  Draw the lines for CLOCK_ADJUST & Plot.
;
	OPLOT,[t1,t2],[55.0,55.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,57.5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'Clock Adjust '
        id = WHERE(kap_resource_names EQ "CLOCK_ADJUST") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           clock_adjust = kap_resource_arr(ind)
           ind = SORT([clock_adjust.startime])	;** sort by start time
           clock_adjust = clock_adjust(ind)
           USERSYM, [0,1,0,-1], [-1,0,1,0], /FILL
           FOR i = 0,N_ELEMENTS(clock_adjust)-1 DO BEGIN
              x1 = (clock_adjust(i).startime - daystart) / 3600	  ;** tai -> hrs since start STARTDIS day
              y1 = 57.5
              IF ((x1 GE !X.CRANGE(0)) AND (x1 LE !X.CRANGE(1))) THEN OPLOT, [x1], [y1], PSYM=8, SYMSIZE=2
           ENDFOR
        ENDIF

;
;  Draw the lines for SPACECRAFT_MANEUVER & Plot.
;
	OPLOT,[t1,t2],[50.0,50.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,52.5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'S/C Maneuver '
        id = WHERE(kap_resource_names EQ "SPACECRAFT_MANEUVER") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           sc_maneuver = kap_resource_arr(ind)
           ind = SORT([sc_maneuver.startime])	;** sort by start time
           sc_maneuver = sc_maneuver(ind)
           FOR i = 0,N_ELEMENTS(sc_maneuver)-1 DO BEGIN
              x1 = (sc_maneuver(i).startime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x2 = (sc_maneuver(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = (x2 < !X.CRANGE(1)) 
              y1 = 50.0
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF

;
;  Draw the lines for RESERVED_COMMANDING & Plot.
;
	OPLOT,[t1,t2],[35.0,35.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,42.5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'Rsrvd Cmd      '
	XYOUTS,t1,37.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'INST '
	OPLOT,[t1,t2],[40.0,40.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,42.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'PLM '
	OPLOT,[t1,t2],[45.0,45.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,47.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'SVM '
        id = WHERE(kap_resource_names EQ "RESERVED") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           reserved_cmd = kap_resource_arr(ind)
           ind = SORT([reserved_cmd.startime])	;** sort by start time
           reserved_cmd = reserved_cmd(ind)
           FOR i = 0,N_ELEMENTS(reserved_cmd)-1 DO BEGIN
              x1 = (reserved_cmd(i).startime - daystart) / 3600D;** tai -> hrs since start STARTDIS day
              x2 = (reserved_cmd(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = 40.0
              IF (reserved_cmd(i).type EQ 1) THEN y1 = y1+5
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF
        IF (DATATYPE(kap_nrt_reserved_arr) EQ 'STC') THEN BEGIN
           nrt_reserved = kap_nrt_reserved_arr(ind)
           ind = SORT([nrt_reserved.startime])	;** sort by start time
           nrt_reserved = nrt_reserved(ind)
           FOR i = 0,N_ELEMENTS(nrt_reserved)-1 DO BEGIN
              x1 = (nrt_reserved(i).startime - daystart) / 3600D;** tai -> hrs since start STARTDIS day
              x2 = (nrt_reserved(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = 35.0
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF
;
;  Draw the lines for THROUGHPUT_MODE & Plot.
;
        yb = 25. & yt = 35.
	OPLOT,[t1,t2],[yb,yb],PSYM=0,LINESTYLE=0
	XYOUTS,t1,yb+5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'NRT Commanding '
	XYOUTS,t1,yb+2.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'no RCR '
	OPLOT,[t1,t2],[yb+5,yb+5],PSYM=0,LINESTYLE=1
	XYOUTS,t1,yb+7.5-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'w/RCR '
        id = WHERE(kap_resource_names EQ "THROUGHPUT") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           throughput = kap_resource_arr(ind)
           ind = SORT([throughput.startime])	;** sort by start time
           throughput = throughput(ind)
           FOR i = 0,N_ELEMENTS(throughput)-1 DO BEGIN
              x1 = (throughput(i).startime - daystart) / 3600D;** tai -> hrs since start STARTDIS day
              x2 = (throughput(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = yb
              IF (throughput(i).type EQ 1) THEN y1 = y1+5
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GE x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF
           ENDFOR
        ENDIF
;
;  Draw the lines for IIE_MASTER & Plot.
;
        yb = 20. & yt = 25.
	OPLOT,[t1,t2],[yb,yb],PSYM=0,LINESTYLE=0
	XYOUTS,t1,yb+2.5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'IIE Master  '
        IF (DATATYPE(kap_iie_arr) EQ 'STC') THEN BEGIN
           iie = kap_iie_arr(ind)
           ind = SORT([iie.startime])	;** sort by start time
           iie = iie(ind)
           FOR i = 0,N_ELEMENTS(iie)-1 DO BEGIN
              x1 = (iie(i).startime - daystart) / 3600D;** tai -> hrs since start STARTDIS day
              x2 = (iie(i).endtime - daystart) / 3600D	;** tai -> hrs since start STARTDIS day
              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              y1 = yb
              y2 = y1 + 5.0		;** add thickness of line
              IF (x2 GT x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
	         XYOUTS,x1+(x2-x1)/2,yb+2.5-dy, iie(i).instrume, FONT=-1,ALIGN=1,CHARSIZE=charsize,COLOR=0
              ENDIF
           ENDFOR
        ENDIF


;--------  END FIRST PLOT ---------------


;--------  SECOND PLOT ---------------
;
;  Make the plot frame, but don't plot the X axis.
;
   !Y.MARGIN = [4, 0]	;** bottom and top margin in chars
        rows = 100	;** yrange [0,100]
	PLOT,[t1,t2],[0,rows-1],/NODATA,XSTYLE=5,YSTYLE=1,YTICKS=1,	$
		YTICKNAME=REPLICATE(' ',2),XMARGIN=[15,4],CHARSIZE=charsize, $
		XTICK_GET=XTICKS,PSYM=0,LINESTYLE=0
;
;  Plot the X axis with time format labels.
;
	AXIS,XAXIS=0,XSTYLE=1,XTICKNAME=HOURS+':'+MINUTES,CHARSIZE=charsize, $
		XTITLE=xtitle
	AXIS,XAXIS=1,XSTYLE=1,XTICKNAME=REPLICATE(' ',30)


	dy = !D.Y_CH_SIZE * charsize / (3 * !Y.S(1) * !D.Y_SIZE)

;  mark day boundaries with gray background
   cols = [0,80]
   y1 = 15.0
   y2 = 75.0
   done = 0
   x1 = 24.0
   x2 = 48.0 < t2
   i = 1
   IF (x1 GE t2) THEN done = 1
   WHILE (NOT(done)) DO BEGIN
      ind = (i MOD 2)
      POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2], COLOR=cols(ind)
      x1 = x2
      x2 = (x2 + 24.0) < t2
      i = i + 1
      IF (x1 GE t2) THEN done = 1
   ENDWHILE

;
;  Draw the line for TM SUBMODE & Plot.
;
	OPLOT,[t1,t2],[87.0,87.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,91.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'TM SUBMODE  '
	XYOUTS,t1,88.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'1'
	OPLOT,[t1,t2],[89.0,89.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,90.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'2'
	OPLOT,[t1,t2],[91.0,91.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,92.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'3'
	OPLOT,[t1,t2],[93.0,93.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,94.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'4'
	OPLOT,[t1,t2],[95.0,95.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,96.0-dy,FONT=-1,ALIGN=1,CHARSIZE=charsize,'5'

        prev_hr = 24
        mins = mins + (prev_hr*60)	;** add n hr for calculating preceeding hours buffer usage
        tmsubmode = INTARR(mins)+1	; tm submode array 1 element per minute of plotting window
					;** initialize it to LR SCI
        id = WHERE(kap_resource_names EQ "TLM_SUBMODE") & id = id(0)
        ind = WHERE(kap_resource_arr.id EQ id)
        IF (ind(0) GE 0) THEN BEGIN
           tlm_submode = kap_resource_arr(ind)
           ind = SORT([tlm_submode.startime])	;** sort by start time
           tlm_submode = tlm_submode(ind)
           tmsubmode(*) = tlm_submode(N_ELEMENTS(tlm_submode)-1).type
           FOR i = 0,N_ELEMENTS(tlm_submode)-1 DO BEGIN
              x1 = (tlm_submode(i).startime - daystart) / 3600D	 ;** tai -> hrs since start STARTDIS day
              IF (i EQ N_ELEMENTS(tlm_submode)-1) THEN $
                 x2 = !X.CRANGE(1) $				;** don't know next mode, cont this 1 to end
              ELSE $
                 x2 = (tlm_submode(i+1).startime - daystart) / 3600D
              x1 = x1 > (!X.CRANGE(0) )
              x2 = x2 < (!X.CRANGE(1) )
              y1 = (tlm_submode(i).type-1)*2 + 87;** 87 is where TM plotting region starts + 2 units high
              y2 = y1 + 1.5		;** add thickness of line
              IF (x2 GT x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2]
              ENDIF

              ;** save array of modes during this plotting range (1 element per minute)
              ;** tai -> mins since start of plotting range ( and previous two hours ) 
              tm0 = (tlm_submode(i).startime - (daystart+(t1*60D*60D)-(prev_hr*60D*60D))) / 60D
              IF ( (tm0 GE 0) AND (tm0 LT N_ELEMENTS(tmsubmode)) ) THEN BEGIN
                 IF (i EQ N_ELEMENTS(tlm_submode)-1) THEN $
                    tm1 = N_ELEMENTS(tmsubmode) $		;** don't know next mode, cont this 1 to end
                 ELSE $
                    tm1 = (tlm_submode(i+1).startime - (daystart+(t1*60D*60D)-(prev_hr*60D*60D))) / 60D
                 tm1 = tm1 < N_ELEMENTS(tmsubmode)
                 IF (tm1 GT tm0) THEN tmsubmode(tm0:tm1-1) = tlm_submode(i).type
              ENDIF
           ENDFOR
        ENDIF

;
;  Draw the lines for lasco-OP's & Plot
;
	OPLOT,[t1,t2],[75.0,75.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,81.0-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'LASCO/EIT (OP) '
 
  yy1 = 75.0
  yy2 = 87.0
  FOR i=0, N_ELEMENTS(op_arr)-1 DO BEGIN
     ; times are in TAI format, convert to hours since start STARTDIS day
     x1 = (op_arr(i).op_start - daystart) / 3600D
     x2 = (op_arr(i).op_stop  - daystart) / 3600D
     x1 = x1 > !X.CRANGE(0)
     x2 = x2 < !X.CRANGE(1)
     loc = WHERE(defined_op_arr.num EQ op_arr(i).num)
     IF (loc(0) GE 0) THEN BEGIN
        id = defined_op_arr(loc(0)).id
        ;label = op_id_arr(id).op_id_name
        label = STRTRIM(defined_op_arr(loc(0)).sci_obj, 2)
        IF X2 GT X1 THEN BEGIN
           OPLOT, [X1,X1], [yy1,yy2], PSYM=0, LINESTYLE=0
           OPLOT, [X2,X2], [yy1,yy2], PSYM=0, LINESTYLE=0
           WRITE_IN_BOX,X1,X2,yy1,yy2,LABEL
        ENDIF
     ENDIF ELSE PRINT, '%SCHEDULE_PLOT: unknown op_num for defined_op_arr.'
  ENDFOR



;
;  Draw the lines for the OS's
;
        IF (DATATYPE(os_arr) EQ 'INT') THEN GOTO, CONT
        ;os_arr2 = os_arr(SORT([os_arr.os_num]))	;** sort on os_num
        ;aind = INDGEN(N_ELEMENTS(os_arr))
        ;aind = REVERSE(aind)
        ;os_arr2 = os_arr(aind)
        os_arr2 = os_arr
        ;uniq_os_arr2 = os_arr2(UNIQ([os_arr2.os_num]))
        uniq_os_arr2 = os_arr2(UNIQ_NOSORT([os_arr2.os_num]))
        num_os = N_ELEMENTS(uniq_os_arr2)
        yy1 = 15.0
        yy2 = 75.0
        height = (yy2 - yy1)/num_os
        os_ypos = FLTARR(num_os+1)
        os_ypos(N_ELEMENTS(os_ypos)-1) = yy2
        os_y = {os_y, pos:0.0, os_num:0L}
        os_yarr = REPLICATE(os_y, num_os)
        j = 0
	FOR i = yy1, yy2, height DO BEGIN
           os_ypos(j) = i
           OPLOT, [t1,t2], [i,i], PSYM=0, LINESTYLE=0
	   IF (ROUND(i) LT yy2) THEN BEGIN
              XYOUTS, t1, i+height/2-dy, FONT=-1, ALIGN=1, CHARSIZE=charsize, $
                             'OS_'+STRTRIM(STRING(FIX(uniq_os_arr2(j).os_num)),2)+' '
              os_yarr(j).pos = i
              os_yarr(j).os_num = uniq_os_arr2(j).os_num
           ENDIF
           j=j+1
        ENDFOR
;
;
;  Plot the OS's
;
        col_ind = [ [1,5], [2,6], [3,7], [4,8], [255,170] ] ;** c1, c2, c3, eit, unk
        XYOUTS, 0.02, 0.012, "C1", COLOR=col_ind(0,0), /NORMAL
        XYOUTS, 0.05, 0.012, "C2", COLOR=col_ind(0,1), /NORMAL
        XYOUTS, 0.08, 0.012, "C3", COLOR=col_ind(0,2), /NORMAL
        XYOUTS, 0.11, 0.012, "EIT", COLOR=col_ind(0,3), /NORMAL
	FOR i = 0, num_os-1 DO BEGIN	;** for every unique os_num
           good = WHERE(os_arr.os_num EQ uniq_os_arr2(i).os_num)

           ;** each telescope is a different color
           tele = 4	;** default color is white
           os_ind = WHERE(defined_os_arr.os_num EQ uniq_os_arr2(i).os_num)
           IF (os_ind(0) GE 0) THEN tele = defined_os_arr(os_ind).tele

           ;** sort on start time to plot
           IF (N_ELEMENTS(good) GT 1) THEN good = good(SORT([os_arr(good).os_start]))	
           FOR j=0, N_ELEMENTS(good)-1 DO BEGIN	;** for every occurance of this os_num
              os = os_arr(good(j))
              y1 = os_ypos(i)
              y2 = os_ypos(i+1)
              ; times are in TAI format, convert to hours since start STARTDIS day
              x1 = (os.os_start - daystart) / 3600D
              x2 = (os.os_stop  - daystart) / 3600D

              x1 = x1 > !X.CRANGE(0)
              x2 = x2 < !X.CRANGE(1)
              IF (x2 GT x1) THEN BEGIN
		 ;** make sure line is at least 1 pixel wide
                 x2 = x2 > (convert_coord([(convert_coord([x1,0], /TO_DEVICE))(0)+1,0], /TO_DATA, /DEVICE))(0)
                 POLYFILL, [x1,x2,x2,x1], [y1,y1,y2,y2], COLOR=col_ind((j MOD 2),tele)

                 IF (DATATYPE(load_camtable) EQ 'STC') THEN BEGIN
                    ;** mark cam table updates with a 'c'
                    ind = WHERE((os.os_num EQ load_camtable.os_num) AND (os.os_start EQ load_camtable.os_start))
                    IF (ind(0) NE -1) THEN XYOUTS, x1, y1+(y2-y1)/2, 'c', /DATA, alignment=1.0
                 ENDIF

                 IF (DATATYPE(load_wlexptable) EQ 'STC') THEN BEGIN
                    ;** mark wavelength/exposure table updates with a 'lambda'
                    ind = WHERE((os.os_num EQ load_wlexptable.os_num) AND (os.os_start EQ load_wlexptable.os_start))
                    IF (ind(0) NE -1) THEN XYOUTS, x1, y1, '!4k!3', /DATA
                 ENDIF

                 IF (DATATYPE(do_candc) EQ 'STC') THEN BEGIN
                    ;** mark check and corrects with a '*'
                    ind = WHERE((os.os_num EQ do_candc.os_num) AND (os.os_start EQ do_candc.os_start))
                    IF (ind(0) NE -1) THEN XYOUTS, x1, y1, '*', /DATA, alignment=1.0
                 ENDIF

                 IF (DATATYPE(move_fp) EQ 'STC') THEN BEGIN
                    ;** mark move FP
                    ind = WHERE((os.os_num EQ move_fp.os_num) AND (os.os_start EQ move_fp.os_start))
                    IF (ind(0) NE -1) THEN BEGIN
                       ;IF (move_fp(ind(0)).move EQ 1) THEN mark='off' ELSE mark='on'
                       mark = 'fp'
                       XYOUTS, x1, y1, mark, /DATA, COLOR=4
                    ENDIF
                 ENDIF

              ENDIF
           ENDFOR
        ENDFOR
;
;  Draw the lines for lasco-buffer
;
CONT:
        ytop = 12
	OPLOT,[t1,t2],[15.0,15.0],PSYM=0,LINESTYLE=0
	XYOUTS,t1,7.5-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'LASCO-buffer   '
	OPLOT,[t1,t2],[12.0,12.0],PSYM=0,LINESTYLE=1
	XYOUTS,t1,12.0-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'100'
	OPLOT,[t1,t2],[9.6,9.6],PSYM=0,LINESTYLE=1
	XYOUTS,t1,9.6-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'80'
	OPLOT,[t1,t2],[2.4,2.4],PSYM=0,LINESTYLE=1
	XYOUTS,t1,2.4-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'20'
	XYOUTS,t1,0.0-dy, FONT=-1,ALIGN=1,CHARSIZE=charsize,'0'
;
;  Plot the lasco-buffer
;
        usage = LONARR(4)			;** telemetry usage by each telescope
        ;rate = [5274L, 26726L, 5274L, 5274L]	;** 5 kbs low rate 20 kbs high rate telemetry downlink speed
        rate = [15833L, 26747L, 5277L, 5277L, 7916L]
        bufsize = 1069256D			;** size of circular buffer in words (A & B are the same)
        bufsize = bufsize*2D*8D			;** convert to bits (2 bytes/word, 8 bits/byte)
        arrx = (FINDGEN(mins) + (!X.CRANGE(0)-prev_hr)*60D) / 60D
        arry = FLTARR(4,mins)			;** only contains os_size at os_stop for every scheduled image
        IF (DATATYPE(os_arr) NE 'INT') THEN BEGIN
        os_arr2 = os_arr(SORT([os_arr.os_stop]))	;** sort on stop time
	FOR i = 0, N_ELEMENTS(os_arr2)-1 DO BEGIN	;** for every os scheduled
           tstop = (os_arr2(i).os_stop - daystart) / 3600D     ;** tai -> hrs since start STARTDIS day
           tmins = CEIL((os_arr2(i).os_stop - os_arr2(i).os_start)/60) > 1
           IF ((tstop GE (!X.CRANGE(0)-prev_hr)) AND (tstop LT !X.CRANGE(1))) THEN BEGIN
              tstopind = (tstop - (!X.CRANGE(0)-prev_hr))
              os_ind = WHERE(defined_os_arr.os_num EQ os_arr2(i).os_num)
              tele = defined_os_arr(os_ind).tele
              s2 = tstopind*60D
              s1 = s2 - tmins + 1
              tsize = FLTARR(tmins)+ (os_arr2(i).os_size/tmins)
              IF ( (s1 GE s2) OR (s1 LT 0) ) THEN BEGIN
                 s1 = s2
                 tsize = os_arr2(i).os_size
              ENDIF
              ;arry(tele,tstopind*60D) = arry(tele,tstopind*60D) + os_arr2(i).os_size
              arry(tele,s1:s2) = arry(tele,s1:s2) + tsize

              ;** assume 1st os starts with buffer at a certain percentage
              IF (i EQ 0) THEN arry(tele,s1) = arry(tele,s1) + bufsize*(FLOAT(schedv.pct_start)/100)
           ENDIF
        ENDFOR
        ENDIF
        buffer = FLTARR(mins)		;** contains amt of data in buffer 1 entry every minute
        tarry = TOTAL(arry,1)
        buffer(*,0) = tarry(0)
	FOR i = 1, N_ELEMENTS(tarry)-1 DO BEGIN	;** for every minute in plot range
           ;** amt in buffer the previous minute - amt downlinked last minute + any new data
           downlinked = rate(tmsubmode(i)-1) * 60	;** kbs * 60 sec = kbm
           buffer(i) = ((buffer(i-1) - downlinked) > 0) + tarry(i)
        ENDFOR
        b = WHERE(buffer NE 0)
        IF (b(0) GT 0) THEN $	 ;** show buffer as being % full before 1st os starts
          buffer(0:b(0)-1) = bufsize*(FLOAT(schedv.pct_start)/100)
        save_buffer = buffer((prev_hr*60):*)
        buffer = buffer / bufsize	;** normalize to 1.0 = 100% full
        buffer = buffer * ytop		;** scale to plot window, ytop is where the 1.0 line is
        buffer = (buffer < 15.)		;** 15 is the top of the buffer window
        arrx = arrx((prev_hr*60):*)
        buffer = buffer((prev_hr*60):*)
        OPLOT, arrx, buffer
        pxval = [arrx(0), arrx, arrx(n_elements(arrx)-1)]
        POLYFILL, pxval, [0, buffer, 0]
        ;** flag places where the buffer is empty with red
        OPLOT, arrx, buffer*(-1), COLOR=1

        ;** Update buffer statistics in schedule widget
        ;**
        date_str = UTC2STR(TAI2UTC(startdis), /ECS, /TRUNCATE) + ' to ' + $
                   UTC2STR(TAI2UTC(enddis), /ECS, /TRUNCATE)
        WIDGET_CONTROL, schedv.usage_dates, SET_VALUE=date_str

;        empty = WHERE(buffer LE 0)
;        IF (empty(0) LT 0) THEN pct_used = '100.0' ELSE $
;           pct_used = STRING( 100.0-(FLOAT(N_ELEMENTS(empty))/N_ELEMENTS(buffer) * 100), FORMAT='(f5.1)' )

        ;** calculate total bits available for SCI downlink over plotting range
        bits_avail = DOUBLE(tmsubmode((prev_hr*60):*))	;** only include plot range (not overflow two hrs)
        FOR i=1, N_ELEMENTS(rate) DO BEGIN
           ind = WHERE(tmsubmode((prev_hr*60):*) EQ i)
           IF (ind(0) GE 0) THEN bits_avail(ind) = rate(i-1) * 60	;** bits/min -> bits/sec
        ENDFOR
        total_bits_avail = TOTAL(bits_avail)

        usage = arry(*,(prev_hr*60):*)
        pct_total = STRING(((DOUBLE(TOTAL(usage)) / total_bits_avail) * 100), FORMAT='(f5.1)' )
        pct_each = STRING(((DOUBLE(TOTAL(usage,2)) / total_bits_avail) * 100), FORMAT='(f5.1)' )

        ;** usage is in bits used per telescope over entire plot range - convert to % of buffer used
        stats_str = '  '+pct_total+'%   '+ $
                    pct_each(0)+'%  '+ $
                    pct_each(1)+'%  '+ $
                    pct_each(2)+'%  '+ $
                    pct_each(3)+'%'
        WIDGET_CONTROL, schedv.usage_stats, SET_VALUE=stats_str


        schedv.x = !x
        schedv.y = !y
        schedv.p = !p


   !P.MULTI = 0

END
