;+
; Project     : SOHO - LASCO/EIT
;
; Name        : SCHEDULE_EVENT
;
; Purpose     : The event handler for the LASCO/EIT SCHEDULE routine.
;
; Use         : SCHEDULE_EVENT, event
;
; Inputs      : event	The event from the SCHEDULE tool.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : None.
;
; Calls       : GENERATE_IPT, SCHEDULE_PLOT, VERIFY_TIMES, UTC2TAI, TAI2UTC, UTC2STR,
;		STR2UTC, INSTRUCT, MOVESCHEDULE, PLAN_OP, DEFINE_OS, SCHEDULE_CALL_OS, 
;		SCHEDULE_MODTIME
;
; Common      : OP_SCHEDULED	Defined in FAKE_OP.
;		OP_DEFINED	Defined in FAKE_OP.
;		OS_DEFINED	Defined in FAKE_OS.
;		OS_SCHEDULED	Defined in FAKE_OS.
;		OS_PLOT_SHARE	Defined in SCHEDULE_PLOT.
;
; Restrictions: The saveset files 'os_arr.dat' and 'op_arr.dat' must exist in 
;		the current directory and have write permission by the user if
;		the user wishes to save changes to the schedule.
;
; Side effects: None.
;
; Category    : Planning, Scheduling.
;
; Prev. Hist. : Program adapted from SOHO/CDS Planning Tool.
;
; Written     : Scott Paswaters, NRL, November 1994.
;
; Modified    :
;
; Version     : Version 0.1, December 5, 1994
;-
 
;__________________________________________________________________________________________________________
;

PRO MARKFP, schedv, ONLINE=online

COMMON OS_SCHEDULED
COMMON OS_DEFINED

   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
   num_text = STRTRIM(num_text, 2)
   WIDGET_CONTROL, schedv.start_text,   GET_VALUE=start_value
   start_value = UTC2TAI(start_value(0))	; ASCII to TAI

   IF (STRMID(num_text,0,2) EQ 'OS') THEN BEGIN
      os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))

      ;** do some checking to see if this is a valid os to mark
      ;** allow only c1
      ;** do not allow for linescan
      ind = WHERE(defined_os_arr.os_num EQ os_num)
      IF (ind(0) EQ -1) THEN BEGIN
         PRINT, '%SCHEDULE_EVENT: OS_NUM: ', os_num, ' not found in defined_os_arr.'
         RETURN
      ENDIF
      ind = ind(0)
      IF (defined_os_arr(ind).tele NE 0) THEN BEGIN
         PRINT, '%SCHEDULE_EVENT: Move FP only applies to C1 observations'
         RETURN
      ENDIF
      IF (defined_os_arr(ind).lp EQ 14) THEN BEGIN
         PRINT, '%SCHEDULE_EVENT: Move FP is not needed for Line Scans'
         RETURN
      ENDIF

      ;** IF this instance is already marked pass the wl & order to routine
      IF (DATATYPE(move_fp) EQ 'STC') THEN BEGIN	;** something scheduled
         ind2 = WHERE( (move_fp.os_num EQ os_num) AND (move_fp.os_start EQ start_value) )
         IF (ind2(0) NE -1) THEN BEGIN			;** already marked
            wl = move_fp(ind2(0)).wl
            order = move_fp(ind2(0)).order
         ENDIF
      ENDIF

      fw =  defined_os_arr(ind).fw
      INPUT_WL_ORDER, fw, wl, order, ONLINE=online
      IF (wl EQ -1) THEN RETURN		;** cancelled
      IF (wl EQ 0) THEN BEGIN		;** remove mark
         PRINT, 'Removing mark for move to wavelength.'
         ind = WHERE((move_fp.os_num NE os_num) OR (move_fp.os_start NE start_value))
         IF (ind(0) EQ -1) THEN $
            move_fp = -1 $
         ELSE $
            move_fp = move_fp(ind)
         SCHEDULE_PLOT, schedv
         RETURN
      ENDIF

      fp_stc = {fp_stc, os_num:0L, os_start:0.0D, wl:0.0D, order:0}
      fp_stc.os_num = os_num
      fp_stc.os_start = start_value
      fp_stc.wl = wl
      fp_stc.order = order
      IF (DATATYPE(move_fp) NE 'STC') THEN BEGIN	;** first entry
         move_fp = fp_stc
      ENDIF ELSE BEGIN 
         ind = WHERE( (move_fp.os_num EQ os_num) AND (move_fp.os_start EQ start_value) )
         IF (ind(0) EQ -1) THEN BEGIN
            PRINT, 'Marking OS instance to move FP'
            move_fp = [move_fp,fp_stc]
         ENDIF ELSE BEGIN
            PRINT, 'This OS instance is already marked to move FP. Changing mark.'
            move_fp(ind(0)).wl = fp_stc.wl
            move_fp(ind(0)).order = fp_stc.order
         ENDELSE
      ENDELSE
      SCHEDULE_PLOT, schedv
   ENDIF

END

;--------------------------------------------------------------------------------------------------

PRO SCHEDULE_EVENT, event

COMMON OP_SCHEDULED
COMMON OP_DEFINED

COMMON OS_DEFINED
COMMON OS_SCHEDULED
COMMON OS_PLOT_SHARE

COMMON KAP_INIT
COMMON KAP_INPUT


 WIDGET_CONTROL, event.id,  GET_UVALUE=input	; get the UVALUE of the event 
						;     generated by the user
 WIDGET_CONTROL, event.top, GET_UVALUE=schedv 	; get structure from UVALUE

 CASE (input) OF 

   'QUIT'   : BEGIN
		;
		; Destroy the planning tool interface and all grouped widgets
  		; 

		WIDGET_CONTROL, event.top, /DESTROY
		RETURN
	      END

   'HELP'   : 

   'VERIFY' : BEGIN
		WIDGET_CONTROL, /HOUR
                OS_SUMMARY_TEXT
                OS_SUMMARY_1LINE
	      END

   'STORE' : BEGIN
		WIDGET_CONTROL, /HOUR
	        SAVE, FILENAME='os_arr.dat', os_arr, defined_os_arr, load_camtable, load_wlexptable, do_candc, move_fp
	        SAVE, FILENAME='op_arr.dat', op_arr, op_struct, defined_op_arr
	     END

   'OUTPUT_PLAN' : BEGIN
		WIDGET_CONTROL, /HOUR
                GENERATE_IPT, schedv.startdis, schedv.enddis, ipt_name
                pos = STRPOS(ipt_name, 'IPT')
                fname = STRMID(ipt_name, 0, pos) + 'summary'
                OS_SUMMARY_1LINE, fname
                fname = STRMID(ipt_name, 0, pos) + 'gif'
                WSET, schedv.win_index
                img = TVRD(0,0,700,512+20)
                TVLCT, r, g, b, /GET
                WRITE_GIF, fname, img, r, g, b
                done = 0
                WHILE NOT(done) DO BEGIN
                   fail = GENERATE_COMMANDS(ipt_name, /SILENT)
                   IF (fail EQ 0) THEN done = 1
                ENDWHILE
	      END

   'OUTPUT_IPT' : BEGIN
		WIDGET_CONTROL, /HOUR
                GENERATE_IPT, schedv.startdis, schedv.enddis, ipt_name
	      END

   'OUTPUT_CMD_TABLE' : BEGIN
		WIDGET_CONTROL, /HOUR
                GENERATE_IPT, schedv.startdis, schedv.enddis, ipt_name
                done = 0
                WHILE NOT(done) DO BEGIN
                   fail = GENERATE_COMMANDS(ipt_name, /CMD_TABLE, /SILENT)
                   IF (fail EQ 0) THEN done = 1
                ENDWHILE
	      END

   'OUTPUT_IAP' : BEGIN
		WIDGET_CONTROL, /HOUR
                GENERATE_IAP, schedv.startdis, schedv.enddis
	      END

   'INPUT_IPT' : BEGIN
                fname = PICKFILE(FILTER='*.IPT')
                IF (fname NE '') THEN BEGIN
		   WIDGET_CONTROL, /HOUR
                   SCHEDULE_READ_IPT, fname, /SILENT, /PROMPT_SHIFT
                   SCHEDULE_PLOT, schedv
                ENDIF
	      END

   'INPUT_KAP' : BEGIN
                READ_KAP, schedv.startdis, schedv.enddis
                SCHEDULE_PLOT, schedv
	      END

   'OUTPUT_BUF' : BEGIN
                GENERATE_BUF, schedv.startdis, schedv.enddis
	      END

   'GET_OS_OP' : BEGIN		;** allow user to select previously defined OS or OP
                 type = event.index	;** 0:OS, 1:OP
                 CASE (type) OF
                    0 : BEGIN		;** select OS
                          ;os_num = SCHEDULE_GET_OS()
                          READ, 'Enter os: ', os_num
                          ;os_num = 1705
                          os_num = FIX(os_num)
                          IF (os_num eq 0) THEN RETURN

                          ind = WHERE(defined_os_arr.os_num EQ os_num)
                          IF (ind(0) EQ -1) THEN BEGIN
                             PRINT, '%SCHEDULE_EVENT: OS_NUM: ', os_num, ' not found in defined_os_arr.'
                             RETURN
                          ENDIF
                          os = defined_os_arr(ind(0))
                          GET_OS_DB_STATS, os, os_size, os_duration

                          WIDGET_CONTROL, schedv.os_label, /SENSITIVE
                          WIDGET_CONTROL, schedv.delta_text, /SENSITIVE
                          WIDGET_CONTROL, schedv.count_text, /SENSITIVE
                          WIDGET_CONTROL, schedv.num_text, SET_VALUE='OS_NUM: '+STRTRIM(STRING(os.os_num),2)
                          WIDGET_CONTROL, schedv.bits_bytes, GET_VALUE=to
                          new_value = SCHEDULE_CONVERT_UNITS(os_size, 0, to)
                          WIDGET_CONTROL, schedv.os_size_text, SET_VALUE=STRING(new_value,FORMAT='(d10.1)')
                          new_value = FLOAT(os_duration * schedv.os_dur_factor)	;** units secs of mins
                          WIDGET_CONTROL, schedv.os_dur_text, SET_VALUE=STRING(new_value,FORMAT='(d6.1)')
	                  WIDGET_CONTROL, schedv.delta_text, SET_VALUE=''
	                  WIDGET_CONTROL, schedv.count_text, SET_VALUE=''
                       END

                    1 : BEGIN		;** select OP
                          ;op_num = SCHEDULE_GET_OP()
                       END

                 ENDCASE

	      END

   'DELETE_OSs' : BEGIN		;** delete selected OS's between two times
                IF (DATATYPE(os_arr) EQ 'INT') THEN RETURN

                WIDGET_CONTROL, schedv.start_text, GET_VALUE=start_value
  	        WIDGET_CONTROL, schedv.stop_text,  GET_VALUE=stop_value	
   	        start_value = UTC2TAI(start_value(0))	; ASCII to TAI
   	        stop_value  = UTC2TAI(stop_value(0))	; ASCII to TAI

                oss = os_arr.os_num
                oss = oss(SORT([oss]))
                oss = oss(UNIQ([oss]))
                oss = STRTRIM(REVERSE(oss), 2)
                selected = BYTARR(N_ELEMENTS(oss))
                result = PICKFILES2(FILES=oss, XSIZE=100, TITLE="Choose OS's to Delete")
                IF ( result(0) NE '' ) THEN BEGIN
                   FOR i=0, N_ELEMENTS(result) - 1 DO BEGIN
                      selected_os = LONG(result(i))
                      ;inds = WHERE(os_arr.os_num NE selected_os)
                      ;IF (inds(0) EQ -1) THEN os_arr = -1 ELSE os_arr = os_arr(inds)

                     keep_os = WHERE( (os_arr.os_num NE selected_os) OR ((os_arr.os_start LT start_value) $
                                   OR (os_arr.os_start GT (stop_value+1))) )
                     IF (keep_os(0) LT 0) THEN BEGIN
                        PRINT, 'deleting all OS'
                        os_arr = -1
                     ENDIF ELSE IF (N_ELEMENTS(keep_os) EQ N_ELEMENTS(os_arr)) THEN BEGIN
                        PRINT, 'no matching OS to delete'
                     ENDIF ELSE BEGIN
                        ;** remove the matching OS's from the os_arr and re-plot
                        os_arr = os_arr(keep_os)
                     ENDELSE

                      IF (DATATYPE(load_camtable) EQ 'STC') THEN BEGIN
                         inds = WHERE( (load_camtable.os_num NE selected_os) OR $
                                   ((load_camtable.os_start LT start_value) OR $
                                    (load_camtable.os_start GT (stop_value+1))) )
                         IF (inds(0) EQ -1) THEN load_camtable = -1 ELSE load_camtable = load_camtable(inds)
                      ENDIF
                      IF (DATATYPE(load_wlexptable) EQ 'STC') THEN BEGIN
                         inds = WHERE( (load_wlexptable.os_num NE selected_os) OR $
                                   ((load_wlexptable.os_start LT start_value) OR $
                                    (load_wlexptable.os_start GT (stop_value+1))) )
                         IF (inds(0) EQ -1) THEN load_wlexptable = -1 ELSE load_wlexptable = load_wlexptable(inds)
                      ENDIF
                      IF (DATATYPE(do_candc) EQ 'STC') THEN BEGIN
                         inds = WHERE( (do_candc.os_num NE selected_os) OR $
                                   ((do_candc.os_start LT start_value) OR $
                                    (do_candc.os_start GT (stop_value+1))) )
                         IF (inds(0) EQ -1) THEN do_candc = -1 ELSE do_candc = do_candc(inds)
                      ENDIF
                      IF (DATATYPE(move_fp) EQ 'STC') THEN BEGIN
                         inds = WHERE( (move_fp.os_num NE selected_os) OR $
                                   ((move_fp.os_start LT start_value) OR $
                                    (move_fp.os_start GT (stop_value+1))) )
                         IF (inds(0) EQ -1) THEN move_fp = -1 ELSE move_fp = move_fp(inds)
                      ENDIF

                   ENDFOR
                   SCHEDULE_PLOT, schedv
                ENDIF

	      END

   'BITS_BYTES' : BEGIN		;** change units on OS_SIZE label
                     WIDGET_CONTROL, schedv.os_size_text, GET_VALUE=os_size
                     from = schedv.os_size_units
                     to = event.index	;** 0:bits, 1:bytes, 2:% buffer
                     schedv.os_size_units = to
                     new_value = SCHEDULE_CONVERT_UNITS(DOUBLE(os_size), from, to)
                     WIDGET_CONTROL, schedv.os_size_text, SET_VALUE=STRING(new_value,FORMAT='(d10.1)')
                  END

   'SECS_MINS' : BEGIN		;** change units on OS_DUR label
                    units = event.index	;** 0:secs, 1:mins
                    IF (units EQ 0) THEN new_os_dur_factor = 1.0D		;** secs
                    IF (units EQ 1) THEN new_os_dur_factor = 1.0D/60.0D		;** mins
                    IF (new_os_dur_factor NE schedv.os_dur_factor) THEN BEGIN
                       WIDGET_CONTROL, schedv.os_dur_text,   GET_VALUE=os_dur
                       os_dur = FLOAT(os_dur)
                       IF (schedv.os_dur_factor EQ 1.0D) THEN factor = 1.D/60.0D   $;** convert secs to minss
                       ELSE factor = 60.0D 				     	    ;** convert mins to secs
                       new_value = FLOAT(os_dur * factor)
                       WIDGET_CONTROL, schedv.os_dur_text, SET_VALUE=STRING(new_value,FORMAT='(d6.1)')
                       schedv.os_dur_factor = new_os_dur_factor
                    ENDIF
                 END

   'DEFINE_OP' : PLAN_OP

   'DEFINE_OS' : BEGIN
                    WIDGET_CONTROL, /HOUR
                    ;WIDGET_CONTROL, schedv.base, ICONIFY=1
                    DEFINE_OS
                 END

   ;
   ;
   ; These buttons determine the plan's time span (shown in the DRAW window).
   ; By adding additional spans to the XPDMENU in widget base identified by 
   ; CONTROLS (in PLAN), and by adding an appropriate CASE statement below,
   ; more time spans can be added
   ;
   ;

   'TIMESPAN' : begin
		   schedv.duration = schedv.span_sec(event.index)
		   MOVESCHEDULE, schedv=schedv
		end


   ;
   ;
   ; Set the mode condition:
   ;	ARROW_DIS  = 0 =  Display mode: Allows modIFication of the plan's 
   ;					start time and END time (via DURATION)
   ;	ARROW_EDIT = 1 =  Edit mode:    Allows editting of the plan at the
   ;					individual study level. 
   ;
   ;

   'ARROW_DIS'  : BEGIN			; pressed the "DISPLAY" button

	schedv.mode = 0					; in display mode now

	;
	; Because turning this button off may be confusing, because
	; neither display nor edit would be turned on, make all events
	; to this button appear to be turning on.
	;

	WIDGET_CONTROL, schedv.edit_study,  SENSITIVE=0

	WIDGET_CONTROL, schedv.arr_span, SET_BUTTON = 1	; make sure buttons 
	WIDGET_CONTROL, schedv.arr_edit, SET_BUTTON = 0	;      look correct

	; 
	; Ignore button turn offs, now that button look correct.
        ; 

	IF event.select EQ 0 THEN RETURN	; Allowed to return without
						; saving CDS to UVALUE.

        ; 
        ; Put the input focus in the start time text widget window
        ;

        WIDGET_CONTROL, schedv.start_text, /INPUT_FOCUS

	END

   'ARROW_EDIT' : BEGIN			; pressed the "EDIT" button

	IF schedv.mode ne 1 THEN BEGIN			; was NOT in edit mode

	   schedv.mode = 1					; in edit mode now

	ENDIF ELSE BEGIN
			
	   ; Was already in edit mode (user must be switching input focus 
	   ;    between the start and the stop widgets).
	   ; Make sure that the time in the text widgets are saved, 
	   ;    This saving means that the user need NOT press RETURN
	   ;	while in the time text windows.
	   ;   

 	   WIDGET_CONTROL, schedv.start_text, GET_VALUE=start_value	
	   WIDGET_CONTROL, schedv.stop_text,  GET_VALUE=stop_value	


	   schedv.study_start = UTC2TAI(start_value(0))	; ASCII to TAI
	   schedv.study_stop  = UTC2TAI(stop_value(0))	; ASCII to TAI

	ENDELSE

	;
	; Because turning this button off may be confusing, because
	; neither display nor edit would be turned on, make all events
	; to this button appear to be turning on.
	;

	WIDGET_CONTROL, schedv.arr_edit, SET_BUTTON = 1	; make sure buttons
	WIDGET_CONTROL, schedv.arr_span, SET_BUTTON = 0	;      look correct


	; 
	; Put the input focus in the text widget window NOT used last time.
	; Highlight the hours part of the time.
	;
	
	IF schedv.last_text EQ schedv.stop_text THEN BEGIN	   ; use start text
   	   WIDGET_CONTROL, schedv.start_text, /INPUT_FOCUS, SET_TEXT_SELECT=[11,2]
	   schedv.last_text = schedv.start_text
	ENDIF ELSE BEGIN				   ; use stop text
   	   WIDGET_CONTROL, schedv.stop_text, /INPUT_FOCUS, SET_TEXT_SELECT=[11,2]
	   schedv.last_text = schedv.stop_text 
	ENDELSE

	; sensitize the action widgets
	WIDGET_CONTROL, schedv.edit_study,  SENSITIVE=1

      END                                


   ;
   ;
   ; The following CASE statements handle the events from the time unit
   ; button, below the DRAW window.
   ;
   ;
   ; When in DISPLAY mode,  THEN the following buttons move the 
   ;    displayed plan the chosen distance in the selected direction.
   ;    Note: month is 30 days, year is 365 days.
   ;
   ; When in EDIT MODE THEN the following buttons change either the 
   ;	start time or the stop time, whichever text widget has any 
   ;	highlights in it.
   ; There is no reason to allow the user, in edit mode, to use
   ;    months or years.
   ;
   ;    

   'BACKYR'  : IF schedv.mode EQ 0 THEN BEGIN		; display mode

		  ; go backwards 365 days * 86400 secs

		  MOVESCHEDULE, sec=-31536000d, schedv=schedv

	       ENDIF 

   'BACKMON' : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; go backwards 30 days * 86400 secs
	
		  MOVESCHEDULE, sec=-2592000d, schedv=schedv

	       ENDIF

   'BACKDAY' : IF schedv.mode EQ 0 THEN BEGIN		; display mode

		  ; go backwards 1 day * 86400 secs

		  MOVESCHEDULE, sec=-86400d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window backward 1 day

 	     	  SCHEDULE_MODTIME, sec=-86400d, schedv=schedv, locate=[8,2]

	       ENDELSE

   'BACKHR'  : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; move display backwards 1 hour (in secs)

		  MOVESCHEDULE, sec=-3600d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window backward 1 hour

 	     	  SCHEDULE_MODTIME, sec=-3600d, schedv=schedv, locate=[11,2]

	       ENDELSE

   'BACKMIN' : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; go backwards 1 min (in secs)  
   
		  MOVESCHEDULE, sec=-60d, schedv=schedv
 		
	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window backward 1 monute

 	     	  SCHEDULE_MODTIME, sec=-60d, schedv=schedv, locate=[14,2]

	       ENDELSE

   'BACKSEC' : IF schedv.mode EQ 0 THEN BEGIN		; display mode

		  ; go backwards 1 second

		  MOVESCHEDULE, sec=-1d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window backward 1 second

 	     	  SCHEDULE_MODTIME, sec=-1d, schedv=schedv, locate=[17,2]

	       ENDELSE
                                                                       
   'FORSEC' : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; go forwards 1 second

		  MOVESCHEDULE, sec=1d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window forward 1 secong

 	     	  SCHEDULE_MODTIME, sec=1d, schedv=schedv, locate=[17,2]

	       ENDELSE

   'FORMIN' : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; go forwards 1 min (in secs)

	  	  MOVESCHEDULE, sec=60d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window forward 1 minute

 	     	  SCHEDULE_MODTIME, sec=60d, schedv=schedv, locate=[14,2]

	       ENDELSE

   'FORHR'  : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; go forwards 1 hour (in secs)

		  MOVESCHEDULE, sec=3600d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window forward 1 hour

 	     	  SCHEDULE_MODTIME, sec=3600d, schedv=schedv, locate=[11,2]

	       ENDELSE

   'FORDAY' : IF schedv.mode EQ 0 THEN BEGIN		; display mode	

		  ; go forwards 1 day * 86400 secs

		  MOVESCHEDULE, sec=86400d, schedv=schedv

	       ENDIF ELSE BEGIN				; edit mode

                  ; Change start time and text window forward 1 day

 	     	  SCHEDULE_MODTIME, sec=86400d, schedv=schedv, locate=[8,2]

	       ENDELSE

   'FORMON' : IF schedv.mode EQ 0 THEN BEGIN	

		  ; go forwards 30 days * 86400 secs

	          MOVESCHEDULE, sec=2592000d, schedv=schedv

	      ENDIF

   'FORYR'  : IF schedv.mode EQ 0 THEN BEGIN	

		  ; go forwards 365 days * 86400
 
		  MOVESCHEDULE, sec=31536000d, schedv=schedv

	      ENDIF


   ;
   ;
   ; Draw events are generated by a cursor click in the window where the 
   ; plan has been plotted.
   ;
   ; In display mode :  Describe what is happening for the selected
   ;			instrument at the selected time.
   ;
   ; In edit mode    :  For a click in the CDS-details line, load the 
   ;			   study name and the times for that study, IF any.
   ;			   Also switches editting control to the CDS-details
   ;			   line.
   ;			For a click in the CDS-plan line,  load the 
   ;			   study name and the times for that study, IF any.
   ;			   Also switches editting control to the CDS-plan
   ;			   line.
   ;


   'DRAW'   : BEGIN
   
	      IF event.press EQ 1 THEN RETURN	; ignore mouse down events
              WSET, schedv.win_index
              !x = schedv.x
              !y = schedv.y
              !p = schedv.p

	      CURSOR, x, y, /NOWAIT		; where did the user click
						;    X hours since start of day
						;    Y op or os_num

	      x = x < !X.CRANGE(1) 	; force click in bounds

   	      datastart = TAI2UTC(schedv.startdis)	; form .DAY and .TIME
   	      datastart.time = 0		; only want day part
   	      daystart = UTC2TAI(datastart)	; start of STARTDIS in TAI
	      time = daystart + (x * 3600d)  	; add in the click location

IF (y LT os_yarr(0).pos) THEN BEGIN
   tmp = schedv.pct_start
   XSET_VALUE, tmp, MIN=0, MAX=100, TITLE='Set Buffer Percentage at Start', $
      GROUP=schedv.base
   schedv.pct_start = tmp
   SCHEDULE_PLOT, schedv
ENDIF ELSE IF (y GE 87) THEN BEGIN ;PRINT, 'above os & op - do nothing for now' $
   val = SELECT_SUBMODE()
   id = WHERE(kap_resource_names EQ "TLM_SUBMODE") & id = id(0)
   ind = WHERE(kap_resource_arr.id EQ id)
   IF (ind(0) GE 0) THEN $
      kap_resource_arr(ind).type = val+1
   SCHEDULE_PLOT, schedv
ENDIF $

ELSE IF (y GE 75) THEN BEGIN	;** selecting an op

	      IF (x GT !X.CRANGE(0)) THEN BEGIN	; ignore clicks left of the y-axis
                 ;** x location determines which op
                 loc = WHERE( (op_arr.op_start LE time) AND (op_arr.op_stop GE time) )
                 IF (loc(0) GE 0) THEN BEGIN
                    op = op_arr(loc(0))
                    WIDGET_CONTROL, schedv.os_label, SENSITIVE=0
;                    WIDGET_CONTROL, schedv.delta_text, SENSITIVE=0
;                    WIDGET_CONTROL, schedv.count_text, SENSITIVE=0
                    WIDGET_CONTROL, schedv.num_text, SET_VALUE='OP_NUM: '+STRTRIM(op.num,2)
                    WIDGET_CONTROL, schedv.start_text, SET_VALUE=TAI2UTC(op.op_start, /ECS)
	            WIDGET_CONTROL, schedv.stop_text, SET_VALUE=TAI2UTC(op.op_stop, /ECS)
	            WIDGET_CONTROL, schedv.delta_text, SET_VALUE=''
	            WIDGET_CONTROL, schedv.count_text, SET_VALUE=''
                 ENDIF ELSE print, 'no OP for time'
	      ENDIF

ENDIF ELSE BEGIN		;** selecting an os

;	      IF (schedv.mode EQ 0) THEN BEGIN	; display mode	;** do the same for both modes

                IF (DATATYPE(os_arr) EQ 'INT') THEN RETURN
		; Y location determines OS_NUM
                ind = WHERE(os_yarr.pos LE y)
                os_num = os_yarr(ind(N_ELEMENTS(ind)-1)).os_num
                print, 'os_num = ', os_num

		;
		; If the user clicks to the left of the yaxis then display
		; information for all instances of the os that matches.
		;
		; Otherwise only display info for the specific instance clicked on.
		;
                WIDGET_CONTROL, schedv.os_label, /SENSITIVE
                WIDGET_CONTROL, schedv.delta_text, /SENSITIVE
                WIDGET_CONTROL, schedv.count_text, /SENSITIVE
                os_ind = -1
                os_arr2 = os_arr(SORT([os_arr.os_start]))  ;** sort on start time
                good_os = WHERE(os_arr2.os_num EQ os_num)
	        IF (x LE !X.CRANGE(0)) THEN BEGIN	; display all instances of os
                   os0 = os_arr2(good_os(0))
                   os1 = os_arr2(good_os(N_ELEMENTS(good_os)-1))
                   WIDGET_CONTROL, schedv.num_text, SET_VALUE='OS_NUM: '+STRTRIM(STRING(os0.os_num),2)

                   WIDGET_CONTROL, schedv.bits_bytes, GET_VALUE=to
                   new_value = SCHEDULE_CONVERT_UNITS(os0.os_size, 0, to)
                   WIDGET_CONTROL, schedv.os_size_text, SET_VALUE=STRING(new_value,FORMAT='(d10.1)')

                   new_value = FLOAT(os0.os_duration * schedv.os_dur_factor)	;** units secs of mins
                   WIDGET_CONTROL, schedv.os_dur_text, SET_VALUE=STRING(new_value,FORMAT='(d6.1)')
	           WIDGET_CONTROL, schedv.start_text, SET_VALUE=TAI2UTC(os0.os_start, /ECS)
	           WIDGET_CONTROL, schedv.stop_text, SET_VALUE=TAI2UTC(os1.os_stop, /ECS)
	           WIDGET_CONTROL, schedv.delta_text, SET_VALUE=''
	           WIDGET_CONTROL, schedv.count_text, SET_VALUE=STRTRIM(STRING(N_ELEMENTS(good_os)),2)
                ENDIF ELSE BEGIN			; display single instance of os
                   FOR i=0, N_ELEMENTS(good_os)-1 DO BEGIN	;** for every matching os_num
                      IF ( (time GE os_arr2(good_os(i)).os_start) AND $;** look for correct time range
                           (time LT os_arr2(good_os(i)).os_stop) ) THEN os_ind = good_os(i)
                   ENDFOR
                   IF (os_ind GE 0) THEN BEGIN
                      os = os_arr2(os_ind)
                      WIDGET_CONTROL, schedv.num_text, SET_VALUE='OS_NUM: '+STRTRIM(STRING(os.os_num),2)

                      WIDGET_CONTROL, schedv.bits_bytes, GET_VALUE=to
                      new_value = SCHEDULE_CONVERT_UNITS(os.os_size, 0, to)
                      WIDGET_CONTROL, schedv.os_size_text, SET_VALUE=STRING(new_value,FORMAT='(d10.1)')

                      new_value = FLOAT(os.os_duration * schedv.os_dur_factor)	;** units secs of mins
                      WIDGET_CONTROL, schedv.os_dur_text, SET_VALUE=STRING(new_value,FORMAT='(d6.1)')
	              WIDGET_CONTROL, schedv.start_text, SET_VALUE=TAI2UTC(os.os_start, /ECS)
	              WIDGET_CONTROL, schedv.stop_text, SET_VALUE=TAI2UTC(os.os_stop, /ECS)
	              WIDGET_CONTROL, schedv.delta_text, SET_VALUE=''
	              WIDGET_CONTROL, schedv.count_text, SET_VALUE=''
                   ENDIF ELSE print, 'no OS for time'
                ENDELSE

ENDELSE

	      END

   'TEXTSTART' : IF schedv.mode EQ 0 THEN BEGIN	; display mode

		   ; 
		   ; The user pressed RETURN in the start text window,
		   ; while in display mode.
		   ; Read in the start value in the text window and
		   ; redisplay the plan starting at that time
		   ;

		   WIDGET_CONTROL, event.id, GET_VALUE=value, SET_VALUE=value

		   ; input was in English but TAI is easier for the program
		   schedv.startdis = UTC2TAI(value(0)) 	; ASCII to TAI

                   ;
		   ; Using the new start time, re-draw the display.
                   ;

		   MOVESCHEDULE, schedv=schedv

                 ENDIF ELSE BEGIN		; edit mode

		   ; 
		   ; The user pressed RETURN in the start text window,
		   ; while in edit mode.
		   ; Read in the start value in the text window and
		   ; save for use later as a study start time.
		   ;

		   WIDGET_CONTROL, event.id, GET_VALUE=value, SET_VALUE=value

		   ; Re-Write start time with all the zeroes
		   WIDGET_CONTROL, schedv.start_text, $	; ASCII to ASCII
				SET_VALUE=UTC2STR(STR2UTC(value(0)),/ECS)

		   ; Input is in English, get time in an easily testable unit
		   schedv.study_start = UTC2TAI(value(0))		; ASCII to TAI

		   ; 
		   ; Since user pressed RETURN, they must be done with start.
	  	   ; Put the input focus in the stop time text widget window
		   ; Highlight the hours part of the time.
		   ;
	
		   WIDGET_CONTROL, schedv.stop_text, /INPUT_FOCUS
		   WIDGET_CONTROL, schedv.stop_text, SET_TEXT_SELECT=[11,2]
	           schedv.last_text = schedv.stop_text	; just a flag

		 ENDELSE


   'TEXTSTOP'  : IF schedv.mode EQ 0 THEN BEGIN	; display mode

		   ; 
		   ; The user pressed RETURN in the stop text window,
		   ; while in display mode.
		   ; Read in the stop value in the text window and
		   ; redisplay the plan starting at stop time minus
		   ; the current duration value.
		   ;

		   WIDGET_CONTROL, event.id, GET_VALUE=value, SET_VALUE=value
			
		   ;
		   ; Passing the routine MOVESCHEDULE a STARTDIS that is 
		   ; really END time that was just input.
		   ; The seconds subtraction will correct in MOVESCHEDULE.
		   ;

                   schedv.startdis = UTC2TAI(value(0))	; ASCII to TAI

		   MOVESCHEDULE, sec=-(schedv.duration), schedv=schedv

		 ENDIF ELSE BEGIN		; edit mode

		   ; 
		   ; The user pressed RETURN in the stop text window,
		   ; while in edit mode.
		   ; Read in the stop value in the text window and
		   ; save for use later as a study start time.
		   ;

		   WIDGET_CONTROL, event.id, GET_VALUE=value, SET_VALUE=value

		   ; Re-Write start time with all the zeroes

		   WIDGET_CONTROL, schedv.stop_text, $	; ASCII to ASCII
				SET_VALUE=UTC2STR(STR2UTC(value(0)),/ECS)

		   ; Input is in English, get time in an easily testable unit
		   schedv.study_stop = UTC2TAI(value(0))		; ASCII to TAI

		   ; 
	  	   ; Put the input focus in the start time text widget window
		   ; Highlight the hour part of the time.
		   ;
	
		   WIDGET_CONTROL, schedv.start_text, /INPUT_FOCUS
		   WIDGET_CONTROL, schedv.start_text, SET_TEXT_SELECT=[11,2]
	           schedv.last_text = schedv.start_text

		 ENDELSE


   'COUNT' : BEGIN
                WIDGET_CONTROL, event.id, GET_VALUE=value
                WIDGET_CONTROL, event.id, SET_VALUE=STRTRIM(STRING(FIX(value)), 2)
             END

   'DELTA' : BEGIN
                IF (DATATYPE(os_arr) EQ 'INT') THEN RETURN
                IF (schedv.mode EQ 0) THEN BEGIN
	           WIDGET_CONTROL, /HOUR
	           WIDGET_CONTROL, schedv.delta_text, GET_VALUE=delta
                   dte = STR2UTC(STRTRIM(delta,2))
                   delta = dte.time / 1000			;** convert msec to sec

                   ;** have user pick which OS's to apply this delta to
                   oss = os_arr.os_num
                   oss = oss(SORT([oss]))
                   oss = oss(UNIQ([oss]))
                   oss = STRTRIM(REVERSE(oss), 2)
                   selected = BYTARR(N_ELEMENTS(oss))
                   result = PICKFILES2(FILES=oss, XSIZE=100, TITLE="Choose OS's to Shift")
                   IF ( result(0) NE '' ) THEN BEGIN
                      FOR i=0, N_ELEMENTS(result) - 1 DO BEGIN 
                         selected_os = LONG(result(i))

                         ;** apply delta to starttime of selected OS's
                         inds = WHERE(os_arr.os_num EQ selected_os)
                         IF (inds(0) NE -1) THEN BEGIN
                            os_arr(inds).os_start = os_arr(inds).os_start + delta
                            os_arr(inds).os_stop = os_arr(inds).os_start + os_arr(inds).os_duration
                         ENDIF

                         ;** apply delta to all camera table loads
                         IF (DATATYPE(load_camtable) EQ 'STC') THEN BEGIN
                            inds = WHERE(load_camtable.os_num EQ selected_os)
                            IF (inds(0) NE -1) THEN $
                               load_camtable(inds).os_start = load_camtable(inds).os_start + delta
                         ENDIF
                         IF (DATATYPE(load_wlexptable) EQ 'STC') THEN BEGIN
                            inds = WHERE(load_wlexptable.os_num EQ selected_os)
                            IF (inds(0) NE -1) THEN $
                               load_wlexptable(inds).os_start = load_wlexptable(inds).os_start + delta
                         ENDIF
                         IF (DATATYPE(do_candc) EQ 'STC') THEN BEGIN
                            inds = WHERE(do_candc.os_num EQ selected_os)
                            IF (inds(0) NE -1) THEN $
                               do_candc(inds).os_start = do_candc(inds).os_start + delta
                         ENDIF
                         IF (DATATYPE(move_fp) EQ 'STC') THEN BEGIN
                            inds = WHERE(move_fp.os_num EQ selected_os)
                            IF (inds(0) NE -1) THEN $
                               move_fp(inds).os_start = move_fp(inds).os_start + delta
                         ENDIF
                      ENDFOR
                      SCHEDULE_PLOT, schedv
                   ENDIF
                ENDIF
             END

   'OS_SIZE' : BEGIN
                WIDGET_CONTROL, schedv.num_text, GET_VALUE=os_num
                os_num = STRTRIM(os_num, 2)
                os_num = LONG(STRMID(os_num, 7, STRLEN(os_num)-7))

                WIDGET_CONTROL, event.id, GET_VALUE=os_size
                ;** convert displayed units to bits
                os_size = STRTRIM(os_size, 2)
                WIDGET_CONTROL, schedv.bits_bytes, GET_VALUE=from
                os_size = SCHEDULE_CONVERT_UNITS(DOUBLE(os_size), from, 0)
                os_size = os_size(0)

                ;** change size of every scheduled instance of this OS
                IF (DATATYPE(os_arr) EQ 'STC') THEN BEGIN
                   ind = WHERE(os_arr.os_num EQ os_num)
                   IF (ind(0) NE -1) THEN BEGIN
                      os_arr(ind).os_size = os_size
                      SCHEDULE_PLOT, schedv
                   ENDIF
                ENDIF
             END

   'OS_DUR' : BEGIN
                WIDGET_CONTROL, schedv.num_text, GET_VALUE=os_num
                os_num = STRTRIM(os_num, 2)
                os_num = LONG(STRMID(os_num, 7, STRLEN(os_num)-7))

                WIDGET_CONTROL, event.id, GET_VALUE=os_duration
                os_duration = STRTRIM(os_duration, 2)
                os_duration = DOUBLE(os_duration(0))
                ;** convert displayed units [secs or mins] to secs
                os_duration = FLOAT(os_duration / schedv.os_dur_factor)
                os_duration = os_duration(0)

                ;** apply duration to starttime of every scheduled instance of this OS
                IF (DATATYPE(os_arr) EQ 'STC') THEN BEGIN
                   ind = WHERE(os_arr.os_num EQ os_num)
                   IF (ind(0) NE -1) THEN BEGIN
                      os_arr(ind).os_duration = os_duration
                      os_start = os_arr(ind).os_start
                      os_arr(ind).os_stop = os_arr(ind).os_start + os_arr(ind).os_duration
                   ENDIF
                ENDIF

                SCHEDULE_PLOT, schedv
             END

   'ADD' : IF schedv.mode EQ 1 THEN BEGIN	; only for edit mode	

		   ;
		   ; Using the current start and stop time and the current
		   ; study name, insert this new study into the plan,
		   ; IF the inputs are valid.
		   ;

		   ;
		   ; Read and save the current values from the text windows,
		   ; incase the user has modified the times without a RETURN.
		   ;

                   ;** see if user is scheduling an OP or an OS
                   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                   num_text = STRTRIM(num_text, 2)

                IF (STRMID(num_text,0,2) EQ 'OP') THEN BEGIN		;** Scheduling an OP

                        op_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
 	                WIDGET_CONTROL, schedv.start_text, GET_VALUE=start_value
		        WIDGET_CONTROL, schedv.stop_text,    GET_VALUE=stop_value
	   	        start_value = UTC2TAI(start_value(0))	; ASCII to TAI
	   	        stop_value  = UTC2TAI(stop_value(0))	; ASCII to TAI
                        loc = WHERE(op_arr.num EQ op_num)
                        IF (loc(0) GE 0) THEN BEGIN		;** OP already scheduled - change it
                           print, 'OP already scheduled - change it'
                           op_arr(loc(0)).op_start = start_value
                           op_arr(loc(0)).op_stop = stop_value
                        ENDIF ELSE BEGIN 			;** scheduling new OP
                           print, 'scheduling new OP'
                           op = op_struct
                           op.num = op_num
                           op.op_start = start_value
                           op.op_stop = stop_value
                           op_arr = [op_arr,op]			;** append to scheduled op_arr
                        ENDELSE

                ENDIF ELSE BEGIN					;** Scheduling an OS

                   WIDGET_CONTROL, schedv.num_text,     GET_VALUE=os_num
		   WIDGET_CONTROL, schedv.start_text,   GET_VALUE=start_value
		   WIDGET_CONTROL, schedv.delta_text,   GET_VALUE=delta
		   WIDGET_CONTROL, schedv.count_text,   GET_VALUE=count
		   WIDGET_CONTROL, schedv.stop_text,    GET_VALUE=stop_value
                   WIDGET_CONTROL, schedv.os_size_text, GET_VALUE=os_size
                   WIDGET_CONTROL, schedv.os_dur_text,  GET_VALUE=os_duration

                   os_num = STRTRIM(os_num, 2)
                   os_num = LONG(STRMID(os_num, 7, STRLEN(os_num)-7))

                   ind = WHERE(defined_os_arr.os_num EQ os_num)
                   IF (ind(0) EQ -1) THEN BEGIN
                      PRINT, '%SCHEDULE_EVENT: OS_NUM: ', os_num, ' not found in defined_os_arr.'
                      RETURN
                   ENDIF

                   ind = ind(0)
                   move_fp_flag = 0
                   ;** only move fp for C1 
                   ;** don't move for Line Scan or Dark LP's
                   IF ( (defined_os_arr(ind).tele EQ 0) AND (defined_os_arr(ind).lp NE 14) AND $
                        (defined_os_arr(ind).lp NE 2) ) THEN move_fp_flag=1

	   	   start_value = UTC2TAI(start_value(0))	; ASCII to TAI
	   	   stop_value  = UTC2TAI(stop_value(0))		; ASCII to TAI
	   	   count  = FIX(count(0))
                   dte = STR2UTC(STRTRIM(delta,2))
                   delta = dte.time / 1000			;** convert msec to sec

                   ;** convert displayed units to bits
                   os_size = STRTRIM(os_size, 2)
                   WIDGET_CONTROL, schedv.bits_bytes, GET_VALUE=from
                   os_size = SCHEDULE_CONVERT_UNITS(DOUBLE(os_size), from, 0)
                   os_size = os_size(0)

                   os_duration = STRTRIM(os_duration, 2)
                   os_duration = DOUBLE(os_duration)
                   ;** convert displayed units [secs or mins] to secs
                   os_duration = FLOAT(os_duration / schedv.os_dur_factor)
                   os_duration = os_duration(0)


                   os = os_struct
                   os.os_num = os_num
                   os.os_duration = os_duration
                   os.os_size = os_size

		   ;** Calculate start and stop times
                   IF (delta GT 0) THEN BEGIN

                                                  ;*************************************
                      IF (count GT 0) THEN BEGIN  ;** schedule count times,  delta apart
                         FOR i=0, count-1 DO BEGIN
                            os.os_start = start_value
                            os.os_stop = start_value + os_duration	;** calc stop time
                            IF (DATATYPE(os_arr) EQ 'INT') THEN os_arr = os ELSE $
                               os_arr = [os_arr, os]			;** add to current schedule
                            IF (DATATYPE(save_start) EQ 'UND') THEN save_start = start_value ELSE $
                               save_start = [save_start, start_value]
                            start_value = start_value + delta
                         ENDFOR

                                                  ;*********************************************************
                      ENDIF ELSE BEGIN		  ;** schedule delta apart between start & stop (calc count)
                         WHILE (start_value LT stop_value) DO BEGIN
                            os.os_start = start_value
                            os.os_stop = start_value + os_duration	;** calc stop time
                            IF (DATATYPE(os_arr) EQ 'INT') THEN os_arr = os ELSE $
                               os_arr = [os_arr, os]			;** add to current schedule
                            IF (DATATYPE(save_start) EQ 'UND') THEN save_start = start_value ELSE $
                               save_start = [save_start, start_value]
                            start_value = start_value + delta
                         ENDWHILE
                      ENDELSE
                   ENDIF ELSE BEGIN	;** delta LE 0

                                                  ;*********************
                      IF (count LE 1) THEN BEGIN  ;** just schedule once
                         os.os_start = start_value
                         os.os_stop = start_value + os_duration		;** calc stop time
                         IF (DATATYPE(os_arr) EQ 'INT') THEN os_arr = os ELSE $
                            os_arr = [os_arr, os]			;** add to current schedule
                         IF (DATATYPE(save_start) EQ 'UND') THEN save_start = start_value ELSE $
                            save_start = [save_start, start_value]

                                                  ;*********************************************************
                      ENDIF ELSE BEGIN	  	  ;** schedule count times between start & stop (calc delta)
                         delta = (stop_value - start_value) / (count-1)
                         FOR i=0, count-1 DO BEGIN
                            os.os_start = start_value
                            os.os_stop = start_value + os_duration	;** calc stop time
                            IF (DATATYPE(os_arr) EQ 'INT') THEN os_arr = os ELSE $
                               os_arr = [os_arr, os]			;** add to current schedule
                            IF (DATATYPE(save_start) EQ 'UND') THEN save_start = start_value ELSE $
                               save_start = [save_start, start_value]
                            start_value = start_value + delta
                         ENDFOR
                      ENDELSE
                   ENDELSE

                   IF (move_fp_flag EQ 1) THEN BEGIN	;** default is to move FP ONLINE
                      ind = WHERE(defined_os_arr.os_num EQ os_num)
                      fw = defined_os_arr(ind).fw
                      INPUT_WL_ORDER, fw, wl, order, ONLINE=online
                      IF (wl NE -1) THEN BEGIN	;** check for cancel
                         fp_stc = {fp_stc, os_num:0L, os_start:0.0D, wl:0.0D, order:0}
                         fp_stc.os_num = os_num
                         fp_stc.wl = wl
                         fp_stc.order = order
                         FOR i=0, N_ELEMENTS(save_start)-1 DO BEGIN
                            fp_stc.os_start = save_start(i)
                            IF (DATATYPE(move_fp) NE 'STC') THEN $	;** first entry
                               move_fp = fp_stc $
                            ELSE $ 
                               move_fp = [move_fp, fp_stc]
                         ENDFOR
                      ENDIF
                   ENDIF

                ENDELSE						;** Scheduling an OS

		   ;
	   	   ; Verify times
		   ; 
		   ;quit = VERIFY_TIMES(baseop=schedv.baseop, $
		;			  study_start=start_value, $
		;			  startonly=1)
		   ;
	   	   ; Put new stud[y|ies] into plan
		   ; 
                   SCHEDULE_PLOT, schedv

		   ; 
		   ; Put the input focus in the start time text widget window
		   ; Highlight the hour part of the time.
		   ;
		
		   WIDGET_CONTROL, schedv.start_text, /INPUT_FOCUS
		   WIDGET_CONTROL, schedv.start_text, SET_TEXT_SELECT=[11,2]
	           schedv.last_text = schedv.start_text


		  ENDIF		


   'DELETE' : IF (schedv.mode EQ 1) THEN BEGIN	; NOT available in display mode

                     ;** see if user is deleting an OP or an OS
                     WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                     num_text = STRTRIM(num_text, 2)

                     IF (STRMID(num_text,0,2) EQ 'OP') THEN BEGIN		;** Deleting an OP

                        op_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
 	                WIDGET_CONTROL, schedv.start_text, GET_VALUE=start_value
	   	        start_value = UTC2TAI(start_value(0))	; ASCII to TAI
                        loc = WHERE(op_arr.num EQ op_num)
                        IF (loc(0) GE 0) THEN BEGIN		;** found valid OP to delete
                           ;** remove the matching OP from the op_arr and re-plot
                           keep = WHERE(op_arr.num NE op_num)
                           op_arr = op_arr(keep)
                           SCHEDULE_PLOT, schedv
                        ENDIF ELSE PRINT, 'No Matching OP_NUM and start_time found to delete.'

                     ENDIF ELSE BEGIN						;** Deleting an OS

 	                IF (DATATYPE(os_arr) EQ 'INT') THEN RETURN
 	                WIDGET_CONTROL, schedv.start_text, GET_VALUE=start_value
	   	        WIDGET_CONTROL, schedv.stop_text,  GET_VALUE=stop_value	
                        os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))

	   	        start_value = UTC2TAI(start_value(0))	; ASCII to TAI
	   	        stop_value  = UTC2TAI(stop_value(0))	; ASCII to TAI

		        ;quit = VERIFY_TIMES(baseop=schedv.baseop, $
			;		  study_start=start_value, $
			;		  startonly=1)
			quit = 0

 		        IF NOT(quit) THEN BEGIN	;** delete all studies with this OS_NUM between start & stop 
                           keep_os = WHERE( (os_arr.os_num NE os_num) OR (os_arr.os_start LT start_value) $
                                                                      OR (os_arr.os_start GT (stop_value+1)) )
                           IF (DATATYPE(load_camtable) EQ 'STC') THEN BEGIN
                              keep_cam = WHERE( (load_camtable.os_num NE os_num) OR (load_camtable.os_start LT start_value) $
                                                                      OR (load_camtable.os_start GT (stop_value+1)) )
                               IF (keep_cam(0) EQ -1) THEN $
                                  load_camtable = -1 $
                               ELSE $
                                  load_camtable = load_camtable(keep_cam)
                           ENDIF

                           IF (DATATYPE(load_wlexptable) EQ 'STC') THEN BEGIN
                              keep_cam = WHERE( (load_wlexptable.os_num NE os_num) OR (load_wlexptable.os_start LT start_value) $
                                                                      OR (load_wlexptable.os_start GT (stop_value+1)) )
                               IF (keep_cam(0) EQ -1) THEN $
                                  load_wlexptable = -1 $
                               ELSE $
                                  load_wlexptable = load_wlexptable(keep_cam)
                           ENDIF

                           IF (DATATYPE(do_candc) EQ 'STC') THEN BEGIN
                              keep_cam = WHERE( (do_candc.os_num NE os_num) OR (do_candc.os_start LT start_value) $
                                                                      OR (do_candc.os_start GT (stop_value+1)) )
                               IF (keep_cam(0) EQ -1) THEN $
                                  do_candc = -1 $
                               ELSE $
                                  do_candc = do_candc(keep_cam)
                           ENDIF

                           IF (DATATYPE(move_fp) EQ 'STC') THEN BEGIN
                              keep_cam = WHERE( (move_fp.os_num NE os_num) OR (move_fp.os_start LT start_value) $
                                                                      OR (move_fp.os_start GT (stop_value+1)) )
                               IF (keep_cam(0) EQ -1) THEN $
                                  move_fp = -1 $
                               ELSE $
                                  move_fp = move_fp(keep_cam)
                           ENDIF

                           IF (keep_os(0) LT 0) THEN BEGIN
                              PRINT, 'deleting all OS'
                              os_arr = -1
                              SCHEDULE_PLOT, schedv
                           ENDIF ELSE IF (N_ELEMENTS(keep_os) EQ N_ELEMENTS(os_arr)) THEN BEGIN
                              PRINT, 'no matching OS to delete'
                           ENDIF ELSE BEGIN
                              ;** remove the matching OS's from the os_arr and re-plot
                              os_arr = os_arr(keep_os)
                              SCHEDULE_PLOT, schedv
                           ENDELSE
 		        ENDIF
                     ENDELSE	;** end deleting an OS
		   ENDIF


   'DELETE_ALL' : IF (schedv.mode EQ 1) THEN BEGIN	; NOT available in display mode
                                                        ; Delete all OS's between two times

 	             IF (DATATYPE(os_arr) EQ 'INT') THEN RETURN
 	             WIDGET_CONTROL, schedv.start_text, GET_VALUE=start_value
	   	     WIDGET_CONTROL, schedv.stop_text,  GET_VALUE=stop_value	

                     quit = RESPOND_WIDG( /COLUMN, $
                                 mess= ["Deleting all OS's between ", $
                                        start_value + ' and ' + stop_value],$
                                 butt=['CONTINUE', 'CANCEL'],$
                                 group=schedv.base)
                     IF (quit NE 0) THEN RETURN

	   	     start_value = UTC2TAI(start_value(0))	; ASCII to TAI
	   	     stop_value  = UTC2TAI(stop_value(0))	; ASCII to TAI

                     keep_os = WHERE( (os_arr.os_start LT start_value) $
                                   OR (os_arr.os_start GT (stop_value+1)) )

                     IF (DATATYPE(load_camtable) EQ 'STC') THEN BEGIN
                        keep_cam = WHERE( (load_camtable.os_start LT start_value) $
                                       OR (load_camtable.os_start GT (stop_value+1)) )
                         IF (keep_cam(0) EQ -1) THEN $
                            load_camtable = -1 $
                         ELSE $
                            load_camtable = load_camtable(keep_cam)
                     ENDIF

                     IF (DATATYPE(load_wlexptable) EQ 'STC') THEN BEGIN
                        keep_cam = WHERE( (load_wlexptable.os_start LT start_value) $
                                       OR (load_wlexptable.os_start GT (stop_value+1)) )
                         IF (keep_cam(0) EQ -1) THEN $
                            load_wlexptable = -1 $
                         ELSE $
                            load_wlexptable = load_wlexptable(keep_cam)
                     ENDIF

                     IF (DATATYPE(do_candc) EQ 'STC') THEN BEGIN
                        keep_cam = WHERE( (do_candc.os_start LT start_value) $
                                       OR (do_candc.os_start GT (stop_value+1)) )
                         IF (keep_cam(0) EQ -1) THEN $
                            do_candc = -1 $
                         ELSE $
                            do_candc = do_candc(keep_cam)
                     ENDIF

                     IF (DATATYPE(move_fp) EQ 'STC') THEN BEGIN
                        keep_cam = WHERE( (move_fp.os_start LT start_value) $
                                       OR (move_fp.os_start GT (stop_value+1)) )
                         IF (keep_cam(0) EQ -1) THEN $
                            move_fp = -1 $
                         ELSE $
                            move_fp = move_fp(keep_cam)
                     ENDIF

                     IF (keep_os(0) LT 0) THEN BEGIN
                        PRINT, 'deleting all OS'
                        os_arr = -1
                        SCHEDULE_PLOT, schedv
                     ENDIF ELSE IF (N_ELEMENTS(keep_os) EQ N_ELEMENTS(os_arr)) THEN BEGIN
                        PRINT, 'no matching OS to delete'
                     ENDIF ELSE BEGIN
                        ;** remove the matching OS's from the os_arr and re-plot
                        os_arr = os_arr(keep_os)
                        SCHEDULE_PLOT, schedv
                     ENDELSE
		  ENDIF

   'SUMMARY' : BEGIN
                   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                   num_text = STRTRIM(num_text, 2)

                   IF (STRMID(num_text,0,2) EQ 'OS') THEN BEGIN		;** OS Summary
                      os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
                      OS_SUMMARY, os_num, schedv.base
                   ENDIF ELSE BEGIN					;** OP Details
                   ENDELSE
                   WSET, schedv.win_index
                   !x = schedv.x & !y = schedv.y & !p = schedv.p
               END

   'DETAILS' : BEGIN
                   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                   num_text = STRTRIM(num_text, 2)

                   IF (STRMID(num_text,0,2) EQ 'OS') THEN BEGIN		;** OS Details
                      os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
                      DEFINE_OS
                      SCHEDULE_CALL_OS, os_num
                   ENDIF ELSE BEGIN					;** OP Details
                      ev = {ev, id:0L, top:0L, handler:0L}
                      ev.id = schedv.base
                      op_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
                      PLAN_OP, ev, op_num
                   ENDELSE
                   WSET, schedv.win_index
                   !x = schedv.x & !y = schedv.y & !p = schedv.p
               END

   'CAMTABLE' : BEGIN	;** mark this instance of this os as needing a camera table upload
                   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                   num_text = STRTRIM(num_text, 2)
		   WIDGET_CONTROL, schedv.start_text,   GET_VALUE=start_value
	   	   start_value = UTC2TAI(start_value(0))	; ASCII to TAI

                   IF (STRMID(num_text,0,2) EQ 'OS') THEN BEGIN
                      os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
                      cam_stc = {cam_stc, os_num:0L, os_start:0.0D}
                      cam_stc.os_num = os_num
                      cam_stc.os_start = start_value
                      IF (DATATYPE(load_camtable) NE 'STC') THEN BEGIN	;** first entry
                         load_camtable = cam_stc
                      ENDIF ELSE BEGIN 
                         ind = WHERE((load_camtable.os_num EQ os_num) AND (load_camtable.os_start EQ start_value))
                         IF (ind(0) EQ -1) THEN BEGIN
                            PRINT, 'Marking OS instance for camera table upload'
                            load_camtable = [load_camtable,cam_stc]
                         ENDIF ELSE BEGIN
                            PRINT, 'This OS instance is already marked for camera table upload'
                            PRINT, 'Removing mark'
                            ind = WHERE((load_camtable.os_num NE os_num) OR (load_camtable.os_start NE start_value))
                            IF (ind(0) EQ -1) THEN $
                               load_camtable = -1 $
                            ELSE $
                               load_camtable = load_camtable(ind)
                         ENDELSE
                      ENDELSE
                      SCHEDULE_PLOT, schedv
                   ENDIF
               END

   'WLEXPTABLE' : BEGIN	;** mark this instance of this os as needing a wavelength/exposure table upload
                   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                   num_text = STRTRIM(num_text, 2)
		   WIDGET_CONTROL, schedv.start_text,   GET_VALUE=start_value
	   	   start_value = UTC2TAI(start_value(0))	; ASCII to TAI

                   IF (STRMID(num_text,0,2) EQ 'OS') THEN BEGIN
                      os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))

      ;** do some checking to see if this is a valid os to mark
      ;** only allow for linescan
      ind = WHERE(defined_os_arr.os_num EQ os_num)
      IF (ind(0) EQ -1) THEN BEGIN
         PRINT, '%SCHEDULE_EVENT: OS_NUM: ', os_num, ' not found in defined_os_arr.'
         RETURN
      ENDIF
      ind = ind(0)
      IF (defined_os_arr(ind).lp NE 14) THEN BEGIN
         PRINT, '%SCHEDULE_EVENT: Move FP only applies to Line Scans'
         RETURN
      ENDIF

                      cam_stc = {cam_stc, os_num:0L, os_start:0.0D}
                      cam_stc.os_num = os_num
                      cam_stc.os_start = start_value
                      IF (DATATYPE(load_wlexptable) NE 'STC') THEN BEGIN	;** first entry
                         load_wlexptable = cam_stc
                      ENDIF ELSE BEGIN 
                         ind = WHERE((load_wlexptable.os_num EQ os_num) AND (load_wlexptable.os_start EQ start_value))
                         IF (ind(0) EQ -1) THEN BEGIN
                            PRINT, 'Marking OS instance for wavelength/exposure table upload'
                            load_wlexptable = [load_wlexptable,cam_stc]
                         ENDIF ELSE BEGIN
                            PRINT, 'This OS instance is already marked for wavelength/exposure table upload'
                            PRINT, 'Removing mark'
                            ind = WHERE((load_wlexptable.os_num NE os_num) OR (load_wlexptable.os_start NE start_value))
                            IF (ind(0) EQ -1) THEN $
                               load_wlexptable = -1 $
                            ELSE $
                               load_wlexptable = load_wlexptable(ind)
                         ENDELSE
                      ENDELSE
                      SCHEDULE_PLOT, schedv
                   ENDIF
               END

   'CANDC' : BEGIN	;** mark this instance of this os as needing a check and correct
                   WIDGET_CONTROL, schedv.num_text,   GET_VALUE=num_text
                   num_text = STRTRIM(num_text, 2)
		   WIDGET_CONTROL, schedv.start_text,   GET_VALUE=start_value
	   	   start_value = UTC2TAI(start_value(0))	; ASCII to TAI

                   IF (STRMID(num_text,0,2) EQ 'OS') THEN BEGIN
                      os_num = LONG(STRMID(num_text, 7, STRLEN(num_text)-7))
                      cam_stc = {cam_stc, os_num:0L, os_start:0.0D}
                      cam_stc.os_num = os_num
                      cam_stc.os_start = start_value
                      IF (DATATYPE(do_candc) NE 'STC') THEN BEGIN	;** first entry
                         do_candc = cam_stc
                      ENDIF ELSE BEGIN 
                         ind = WHERE((do_candc.os_num EQ os_num) AND (do_candc.os_start EQ start_value))
                         IF (ind(0) EQ -1) THEN BEGIN
                            PRINT, 'Marking OS instance for check and correct'
                            do_candc = [do_candc,cam_stc]
                         ENDIF ELSE BEGIN
                            PRINT, 'This OS instance is already marked for check and correct'
                            PRINT, 'Removing mark'
                            ind = WHERE((do_candc.os_num NE os_num) OR (do_candc.os_start NE start_value))
                            IF (ind(0) EQ -1) THEN $
                               do_candc = -1 $
                            ELSE $
                               do_candc = do_candc(ind)
                         ENDELSE
                      ENDELSE
                      SCHEDULE_PLOT, schedv
                   ENDIF
               END

   'MOVEFP' : BEGIN	;** mark this instance of this os as needing to move the FP
                  MARKFP, schedv
               END

    ELSE : PRINT,'Invalid widget event in SCHEDULE_EVENT, ', input
 ENDCASE


; 
; Any changes to the CDS structure in this pass must be resaved into
; the UVALUE, or ELSE they will be lost at the following program RETURN
; to the XMANAGER.
;

  WIDGET_CONTROL, event.top, SET_UVALUE=schedv	


RETURN
END

