pro curval, hd, im, x0=x0, y0=y0, Filename=Filename
;+
; NAME:
;	CURVAL
; PURPOSE:   
;	Display values associated with the cursor as it is moved over an image.
;	CURVAL displays different information depending whether the user 
;	supplied an image array, and/or a FITS header array
;
; CALLING SEQUENCE(S):
;	curval          ;Display x,y and byte intensity (inten)
;	
;	curval, im   ;Display x,y,inten, and also pixel value (from image array)
;	
;	curval, hdr, im, [X0 =, Y0 =, FILE =]        
;
; OPTIONAL INPUTS:
;	Hdr  = FITS Header array
;	Im  = Array containing values that are displayed.  Any type.
;
; OPTIONAL KEYWORD INPUTS:
;	X0,Y0  = coordinates of lower left-hand corner of input image 
;		If not supplied then CURVAL assumes that pixel (0,0) on the
;		image is located at position (0,0) in the window 
;	FILENAME  = name of file to where CURVAL data can be saved.
;		Data will only be saved if left or center mouse button
;		are pressed.
;
; OUTPUTS:
;	None.
;
; SIDE EFFECTS:
;	X and Y values, etc., of the pixel under the cursor are constantly
;	displayed.  
;	Pressing left or center mouse button prints a line of output, and 
;	starts a new line.
;	Pressing right mouse button exits the procedure.
;	If the keyword FILENAME is defined, the date and time, and a heading 
;	will be printed in the file before the data.
;
; REVISION HISTORY:
;	Written,  K. Rhode,  STX  May 1990
;	Added keyword FILENAME	D. Alexander  June 1991
;       Don't write to Journal file   W. Landsman    March 1993
;	Use astrometry structure  W. Landsman      Feb 1994
;	Modified for Mac IDL          I.   Freedman     April 1994
;	Mar 1999   N Rich(NRL)	Change output for SOHO (LASCO) header; also accept
;				header structure
;-
; On_error,2    ;if an error occurs, return to caller

 if !VERSION.OS EQ 'MacOS' then begin
  print,'Macintosh Mouse maps to "LEFT" button and LEFT and RIGHT arrow keys to'
  print,'"CENTER" and "RIGHT" buttons respectively'
 endif

 f_header = 0b           ;True if a FITS header supplied
 f_image =  0b           ;True if an image array supplied
 f_astrom = 0b           ;True if FITS header contains astrometry
 f_bscale = 0b           ;True if FITS header contains BSCALE factors
 f_imhd   = 0b           ;True if image array is in HD (1 parameter)
 f_soho	  = 0b		; True if header is from SOHO
 npar = N_params()
 fileflag=0		;True once left or middle mouse button pressed

 if !D.WINDOW EQ -1 then begin
	message,'ERROR - No image window active',/INF
	return
 endif

 if !VERSION.OS  NE 'MacOS' then begin
  openw,term,filepath(/TERMINAL),/GET_LUN,/STREAM         ;Output terminal
 endif else  term = -1

; Determine if there is an X or Y offset 

 if not keyword_set(X0) then x0 = 0
 if not keyword_set(Y0) then y0 = 0

if (!D.FLAGS and 256) EQ 256 then wshow,!D.WINDOW  ;Bring active window to foreground

; Print formats and header for different astrometry,image, BSCALE combinations

 cr = string("15b)
 line0 = '  X     Y     Byte Inten'
 line1 = '  X     Y     Byte Inten   Value'
 line2 = '  X     Y     Byte Inten        RA                Dec   '
 line3 = '  X     Y   ByteInten   Value        RA             Dec         Flux' 
 line4 = '  X     Y   ByteInten   Value        RA             Dec' 
 line5 = '  X     Y   ByteInten   Value   Flux'
 line6 = '  X     Y   ByteInten	  Radii   Deg from N   Value'
 line7 = '  X     Y   ByteInten   Radii   Deg from N        '

 f0 = "($,a,i4,2x,i4,6x,i4)"
 f1 = "($,a,i4,2x,i4,6x,i4,3x,a)"
 f2 = "($,a,i4,2x,i4,6x,i4,5x,i4,i4,1x,f6.2,3x,i4,i4,1x,f6.2)"
 f3 = "($,a,i4,2x,i4,2x,i4,3x,a,2x,i4,i4,1x,f6.2,2x,i4,i4,1x,f6.2,5x,e8.2)"
 f4 = "($,a,i4,2x,i4,2x,i4,3x,a,2x,i4,i4,1x,f6.2,2x,i4,i4,1x,f6.2)"
 f5 = "($,a,i4,2x,i4,2x,i4,3x,a,5x,e8.2)"
 f6 = "($,a,i4,2x,i4,6x,i4,4x,f6.3,4x,f6.2,2x,a)"
 f7 = "($,a,i4,2x,i4,6x,i4,4x,f6.3,4x,f6.2)"

if (npar gt 0) then begin
   IF datatype(hd) EQ 'STC' THEN hd = struct2fitshdr(hd)
  type = size(hd)
  if (npar eq 1) and (type(0) eq 2) then begin
    f_image = 1b  & f_imhd = 1b 
    imtype = type
  endif else if (type(2) ne 7) or (type(0) ne 1) then begin
    print,'Calling sequence options: CURVAL,'
    print,'                          CURVAL,IM  where IM is a 2-D image,'
    print,'                          CURVAL,HD  where HD is a FITS header,'
    print,'                          or  CURVAL,HD,IM.'
    return
  endif else if (type(2) eq 7) and (type(0) eq 1) then f_header = 1b
  if (npar eq 2) then begin
    f_image = 1b & f_header = 1b
    imtype = size(im)
    if (imtype(0) lt 2) or $
     (imtype(imtype(0)+2) ne imtype(1)*imtype(2)) then $
       message,'Image array (second parameter) is not two dimensional.'
  endif
endif    

; Get information from the header

 if f_header then begin     
   IF trim(fxpar(hd,'TELESCOP')) EQ 'SOHO' THEN BEGIN
	extlasco,hd,astr,noparams
	f_soho = 1b
	date = sxpar(hd,'DATE-OBS')
	yymmdd = utc2yymmdd(str2utc(date))
	solar_ephem,yymmdd,RADIUS=solrad
   ENDIF ELSE extast, hd, astr, noparams             ;Extract astrometry structure
  if (noparams ge 0) then f_astrom = 1b

  if f_image then begin
  bscale = sxpar(hd,'BSCALE')
  if (bscale ne 0) then begin
    bzero = sxpar(hd,'BZERO')
    bunit = sxpar(hd,'BUNIT')
    if !ERR ge 0 then $ 
    if f_astrom then line3 = line3 + '('+bunit+ ')' else $
                     line5 = line5 + '('+bunit+')'
    f_bscale = 1b
  endif
  endif
 endif

 print,'Press left or center mouse button for new output line,'
 print,'... right mouse button to exit.  

; different print statements, depending on the parameters

 case 1 of

(f_image eq 0b) and (f_astrom eq 0b) and (f_soho EQ 0):  begin   
   curtype = 0 & print, line0  & end      ;No image or header info

(f_image) and (f_astrom eq 0b) and (f_bscale eq 0b) and (f_soho EQ 0): begin
   curtype = 1  & print,line1 & end       ;Only image array supplied

(f_image eq 0b) and (f_astrom) and (f_bscale eq 0b) and (f_soho EQ 0): begin 
   curtype = 2  & print,line2 & end       ;Astrometry but no image array

(f_image) and (f_astrom) and (f_bscale) and (f_soho EQ 0): begin
   curtype =3   & print,line3 & end       ;Image array + astrometry + BSCALE

(f_image) and (f_astrom) and (f_bscale eq 0b) and (f_soho EQ 0): begin
   curtype = 4  & print,line4 & end       ;Image array +astrometry

(f_image) and (f_astrom eq 0b) and (f_bscale) and (f_soho EQ 0): begin
   curtype = 5  & print,line5 & end       ;Image array + BSCALE

(f_image) and (f_soho) and (f_bscale eq 0b): begin
   curtype = 6  & print,line6 & end       ;Image array + SOHO/LASCO header

(f_image eq 0b) and (f_soho) and (f_bscale eq 0b): begin 
   curtype = 7  & print,line7 & end       ;SOHO/LASCO header but no image array

endcase
;
;
 LOOP: sv_err = !ERR
 !ERR = 0
 cursor,x,y,2,/DEVICE,/CHANGE                                 

 if !ERR EQ 4 then begin
    if !VERSION.OS NE 'MacOS' then free_lun,term
   
    if fileflag AND !VERSION.OS NE 'MacOS' then free_lun,lun
    return

 endif

 if (!ERR GE 1) and (!ERR LE 3) and (!ERR NE sv_err) then begin  ; have left or center buttons been pressed?
    print,form="($,a)",string("12b)   ; print a form feed
    if keyword_set(filename) and (not fileflag) then begin	; open file & print table header to file
	get_lun,lun
	openw,lun,filename
	printf,lun,'CURVAL:   ',systime(0)	;print time and date to file
        case 1 of  		;different print statements for file, depending on parameters

        (f_image eq 0b) and (f_astrom eq 0b) : begin
           printf, lun, line0  & end			;No image or header info

	(f_image) and (f_astrom eq 0b) and (f_bscale eq 0b) : begin
	   printf, lun, line1 & end			;Only image array supplied

	(f_image eq 0b) and (f_astrom) and (f_bscale eq 0b) : begin
   	   printf, lun, line2 & end			;Astrometry but no image array

	(f_image) and (f_astrom) and (f_bscale) : begin
    	   printf, lun, line3 & end			;Image array + astrometry + BSCALE

	(f_image) and (f_astrom) and (f_bscale eq 0b) : begin
   	   printf, lun, line4 & end			;Image array + astrometry

	(f_image) and (f_astrom eq 0b) and (f_bscale) : begin
   	   printf, lun, line5 & end			;Image array + BSCALE

	(f_image) and (f_soho) and (f_bscale eq 0b): begin
   	   printf, lun, line6 & end       ;Image array + SOHO/LASCO header

	(f_image eq 0b) and (f_soho) and (f_bscale eq 0b): begin 
   	   printf, lun, line7 & end       ;SOHO/LASCO header but no image array

	endcase
 	fileflag=1
    endif
    if keyword_set(filename) then begin
	case curtype of 
	   0: printf, lun, form=f0,'', x, y, inten
	   1: printf, lun, form=f1,'', x, y, inten, value 
	   2: printf, lun, form=f2,'', x, y, inten, ihr, imin, xsec, $
			deg, mn, sc
	   3: printf, lun, form=f3,'', x, y, inten, value, ihr, imin, xsec, $
			deg, mn, sc, flux
	   4: printf, lun, form=f4,'', x, y, inten, value, ihr, imin, xsec, $
			deg, mn, sc
	   5: printf, lun, form=f5,'', x, y, inten, value, flux
	   6: printf, lun, form=f6,'', x, y, inten, radius, angle, value
	   7: printf, lun, form=f7,'', x, y, inten, radius, angle
	endcase
    endif
 endif

  x = x>0 & y = y>0
  inten = fix(tvrd(x,y,1,1))   ; read the byte intensity 

 x = x - x0
 y = y - y0

 if f_image then begin
      if (x LT 0) or (x GE imtype(1)) or $
         (y LT 0) or (y GE imtype(2)) then value = 0 else $
      if f_imhd then value = hd(x,y) else value = im(x,y)
 endif
IF f_soho THEN BEGIN
   xcen = astr.suncen(0)
   ycen = astr.suncen(1)
   r = sqrt(float(x - xcen)^2 + float(y - ycen)^2) ; r in pixels
   rconv = solrad*3600/astr.platescl
   radius = r/rconv
   angle = atan(float(xcen-x), float(y-ycen))*180./!pi
   angle = angle + astr.crota
   if angle lt 0 then angle = angle + 360.
ENDIF ELSE $
 if f_astrom then begin

	case strmid(astr.ctype(0),5,3) of 
	'GSS': gsssxyad, astr, x, y, a, d
	else:  xy2ad, x, y, astr, a, d            ; convert to ra and dec
	endcase

      radec, a, d, ihr, imin, xsec, deg, mn, sc   ; change to hr,min format

 endif

 if f_bscale  then flux = bscale*value + bzero  
;stop

 case curtype of
	0:  printf,term,form=f0,cr,x,y,inten  
	1:  printf,term,form=f1,cr,x,y,inten,value 
	2:  printf,term,form=f2,cr,x,y,inten,ihr,imin,xsec,deg,mn,sc        
	3:  printf,term,form=f3,cr,x,y,inten,value,ihr,imin,xsec,deg,mn,sc,flux
	4:  printf,term,form=f4,cr,x,y,inten,value,ihr,imin,xsec,deg,mn,sc
	5:  printf,term,form=f5,cr,x,y,inten,value,flux
 	6:  printf,term, form=f6,cr, x, y, inten, radius, angle, value
	7:  printf,term, form=f7,cr, x, y, inten, radius, angle
endcase

 goto,LOOP

 end
