pro tvlist, image, dx, dy, TEXTOUT = textout
;+
; NAME
;	TVLIST
; PURPOSE:
;	Cursor controlled listing of a TV image on the terminal. 
;
; CALLING SEQUENCE:
;	TVLIST, image, [ dx, dy, TEXTOUT = ]
;
; OPTIONAL INPUTS:
;	IMAGE - Array containing the image currently displayed on the TV.
;		If omitted, the byte pixel intensities are read from the TV
;		Image array should be the same size, or a multiple of the size
;		of the image in the current window.   
;
;	DX     -Integer scalar giving the number of pixels inthe  X direction 
;		to be displayed.  If omitted then DX = 18 for byte images, and 
;		DX = 14 for integer images.  TVLIST will display REAL data 
;		with more significant figures if more room is availble to 
;		print.  
;
;	DY    - Same as DX, but in Y direction.  If omitted, then DY = DX 
;
; OPTIONAL INPUT KEYWORD:
;	TEXTOUT - Optional keyword that determines output device.
;		The following dev/file is opened for output.
;
;		textout=1	TERMINAL using /more option
;		textout=2	TERMINAL without /more option
;		textout=3	<program>.prt
;		textout=4	laser.tmp
;		textout=5      user must open file
;		textout = filename (default extension of .prt)
;
; OUTPUTS:
;	None.
; PROCEDURE:
;	Program prompts user to place cursor on region of interest in 
;	image display.  Corresponding region of image is then displayed at
;	the terminal.   A compression factor between the image array and the
;	displayed image is determined using the ratio of image sizes.  If 
;	necessary, TVLIST will divide all pixel values in a REAL*4 image by a 
;	(displayed) factor of 10^n (n=1,2,3...) to make a pretty format.
;
; SYSTEM VARIABLE:
;	Set !TEXTOUT = 3 to direct output to a disk file.  TVLIST will then also
;	prompt for a brief description.
;
; RESTRICTIONS:
;	TVLIST may not be able to correctly format all pixel values if the
;	dynamic range near the cursor position is very large.
;
; REVISION HISTORY:
;	Written by rhc, SASC Tech, 3/14/86.
;	Added textout keyword option, J. Isensee, July, 1990
;	Check for readable pixels     W. Landsman   May 1992
;	Use integer format statement from F_FORMAT    W. Landsman   Feb 1994
;-
 On_error,2

 npar = N_params()

 if not keyword_set(TEXTOUT) then textout = !TEXTOUT ;Use default

 if npar GE 2 then $
    if N_elements( dx) NE 1 then $
          message, 'ERROR - Second parameter (format width) must be a scalar'

 textopen,'TVLIST', TEXTOUT = (textout>2)        ;Don't use /MORE

 if npar EQ 0 then begin 	;Read pixel values from TV

        if (!D.FLAGS and 128) NE 128 then message, $
             'ERROR -- Unable to read pixels from current device ' + !D.NAME
	printf,!TEXTUNIT,'No image array supplied, pixel values read from TV' 
	type = 1		;Byte format

 endif else begin

	sz = size(image)
	if (sz(0) LT 2) or (sz(sz(0)+2) NE sz(1)*sz(2)) then $
		message,'Image array (first parameter) not 2-dimensional'
    	type = sz(sz(0)+1)	     ;Byte or Integer image?

 endelse 

 if textout EQ 3 then begin   ;Direct output to a disk file
	printf,!TEXTUNIT,'TVLIST: '+strmid(!STIME,0,17)
        descr = ''
	read,'Enter a brief description to be written to disk: ',descr
        printf,!TEXTUNIT,descr
	printf,!TEXTUNIT,' '
 endif                 

 if (!D.FLAGS AND 256) EQ 256 THEN wshow,!D.WINDOW

 if ( npar GT 0 ) then begin 	;get X and Y dimensions of the image
	xdim = sz(1) - 1 
	ydim = sz(2) - 1 
 endif else begin		;dimensions of TV display
	xdim = !d.x_vsize
	ydim = !d.y_vsize
 endelse

 tvcrs, 1                                    ;Make sure cursor is on
 print, 'Put the cursor on the area you want to list; press any mousse button'
 cursor, xim, yim, /WAIT, /DEVICE
 xim = fix(xim+0.5)
 yim = fix(yim+0.5)

 if npar LT 2 then  $  ;Use default print size? 
    if type EQ 1 then dx = 18 else dx = 15

 if npar LT 3 then dy = dx
; Don't try to print outside the image
  xmax = (xim + dx/2) < xdim
  xmin = (xim - dx/2) > 0 
  ymax = (yim + dy/2) < ydim
  ymin = (yim - dy/2) > 0 

 dx = xmax - xmin + 1 & dy = ymax - ymin + 1

 if xmin GE xmax then $
    message,'ERROR - The cursor is off the image in the x-direction'
 if ymin GE ymax then $
    message,'ERROR - The cursor is off the image in the y-direction'

 fmtsz = (80-4)/dx
 sfmt = strtrim(fmtsz,2)
 cdx = string(dx,'(i2)')
 flt_to_int = 0                   ;Convert floating point to integer?

 case 1 of                                    ;Get proper print format
	type EQ 1 or (npar EQ 0):  fmt = '(i4,' + cdx + 'i' + sfmt + ')'
	type EQ 2:  fmt = '(i4,1x,' + cdx + 'i' + sfmt + ')'
	(type EQ 4) or (type EQ 3) or (type EQ 5):  begin
		temp = image(xmin:xmax,ymin:ymax)
		minval = min(temp, max=maxval) 
		realfmt =  f_format(minval,maxval,factor,fmtsz)
	        if strmid(realfmt,0,1) EQ 'I' then flt_to_int = 1
                fmt = '(i4,1x,' + cdx + realfmt + ')'
		if factor NE 1 then $                                   
       printf,!TEXTUNIT,form='(/,A,E7.1,/)',' TVLIST: Scale Factor ',factor
      end
 endcase 
; Compute and print x-indices above array

 if npar EQ 0 then begin 
    xmin = xmin - (xmin mod 2)
    dx = dx - (dx mod 2)
 endif

 index = indgen(dx)+ xmin

 if type NE 1 then $
     printf,!TEXTUNIT,form='(A,'+ cdx + 'i' + sfmt + ')', ' col ', index  $
 else printf,!TEXTUNIT,form='(A,'+ cdx + 'i' + sfmt + ')', ' col', index 
printf,!TEXTUNIT,'$(A)',' row'

 if npar EQ 0 then begin 
    row = fix( tvrd( xmin,ymin,dx,dy) )
    for i = dy-1,0,-1 do printf, !TEXTUNIT, FORM=fmt, i+ymin, row(*,i)

 endif else begin

 for i = ymax,ymin,-1 do begin	;list pixel values

        row = image(i*sz(1)+xmin:i*sz(1)+xmax)
	if type EQ 1 then row = fix(row)             ;Convert byte array
	if (type EQ 4) or (type EQ 3) or (type EQ 5) then row = row/factor
	if flt_to_int then row = nint( row )
        printf,!TEXTUNIT, FORM=fmt, i, row
 endfor

 endelse

 textclose,TEXTOUT = textout

 return
 end
