;+
; Project     : SOHO - CDS     
;                   
; Name        : ARCMIN2HEL()
;               
; Purpose     : Convert arcmin from sun centre to heliographic coords.
;               
; Explanation : Converts an (x,y) position given in arcmins relative to the 
;               solar disk centre to heliographic coordinates for the date
;               supplied ( default date = today).
;               
; Use         : IDL> helio = arcmin2hel(xx,yy,date=date)
;    
; Inputs      : xx  -  E/W coordinate in arc minutes relative to sun center 
;                      (West is positive)
;               yy  -  S/N coordinate in arc minutes relative to sun center 
;                      (North is positive)
;               
; Opt. Inputs : None
;               
; Outputs     : Function returns 2 element vector: [lat, long] in degrees.
;               
; Opt. Outputs: None
;               
; Keywords    : date  -  date/time in any CDS format
;
;               off_limb  - flag which is true if the coordinates are beyond
;                           the solar limb.
;
; Calls       : PB0R
;               ANYTIM2UTC
;
; Restrictions: If the supplied coordinates are outside the solar disk, the
;               region is projected onto the limb.
;               
; Side effects: None
;               
; Category    : Util, coord
;               
; Prev. Hist. : Original by J P Wuelser.
;
; Written     : CDS version by C D Pike, RAL, 6 Sept 93
;               
; Modified    : Converted to use CDS time and pb0r routines, CDP, 17-May-94
;		Version 3, William Thompson, GSFC, 14 November 1994
;			Modified .DAY to .MJD
;
; Version     : Version 3, 14 November 1994
;-            


function arcmin2hel, xx, yy, date=dat, off_limb=offlimb

;
;  save input x,y position
;
inxy = [xx,yy]

;
;  set up date required in format required by PB0R
;
if n_elements(dat) eq 0 then get_utc, dat   
dat = anytim2utc(dat)                       
if dat.mjd eq 0 then begin
   print,'ARCMIN2HEL: error on time format specification.'
   return,[-99.,-99.]
endif

;
; get B0 and solar radius
;
angles = pb0r(dat)
sunr = angles(2)
b0 = angles(1) / !radeg

;
; normalize, check if off limb, project back to limb
;
inxy = inxy/sunr
inr = sqrt(total(inxy*inxy))
offlimb=0
if inr gt 1.0 then begin
   inxy = inxy/inr
   inr = 1.0
   offlimb=1
endif

;
; assume inxy on sphere, calc. x,y,z. Here: x towards obs., y = W, z = N
;
xyz = [sqrt(1.0-inr*inr),inxy]

;
; rotate around y axis to correct for B0 angle (B0: hel. lat. of diskcenter)
;
rotmx = [[cos(b0),0.0,sin(b0)], [0.0,1.0,0.0], [-sin(b0),0.0,cos(b0)]]
xyz = rotmx # xyz

;
; calc. latitude and longitude.
;
lonlat = fltarr(2)
lonlat(1) = asin(xyz(2))                        ; latitude
lonlat(1) = lonlat(1) < (89.0/!radeg)		; force lat. smaller 89 deg.
lonlat(1) = lonlat(1) > (-89.0/!radeg)		; force lat. bigger -89 deg.
lonlat(0) = asin(xyz(1)/cos(lonlat(1)))         ; longitude

;
; longitude may be larger than 90 degrees due to nonzero B0: get proper value
;
if xyz(0) lt 0.0 then begin
   if xyz(1) ge 0.0 then lonlat(0) = !pi-lonlat(0) $       ; lon = 180 - lon
                    else lonlat(0) = -!pi-lonlat(0)        ; lon = -180 - lon
endif

;
; convert to degrees
;
lonlat = lonlat * !radeg			

return, [lonlat(1),lonlat(0)]

end
