;+
; Project     : SOHO - CDS     
;                   
; Name        : HEL2ARCMIN()
;               
; Purpose     : Compute position relative to sun centre from heliographic.
;               
; Explanation : Using the input heliographic coordinates of a feature,
;               calculate the position in arcmin relative to the sun centre
;               taking account of the sun's orientation (B0).  The current
;               date is assumed unless specified.  West and  North are 
;               considered positive.
;               
; Use         : IDL> print, hel2armin(ns, ew, date = dat)
;                eg  print,hel2arcmin('S34','E23')
;                or  print,hel2arcmin(-34,-23)
;    
; Inputs      : ns      -  the Heliographic latitude in degrees (can be a
;                          string with N/S first character instead of sign).
;               ew      -  the Heliographic longitude in degrees (can be a
;                          string with E/W first character instead of sign).
;               
; Opt. Inputs : None
;               
; Outputs     : Function returns the (x,y) location in arcmins relative to
;               sun disk centre.
;               
; Opt. Outputs: None
;               
; Keywords    : date    -  the date to use in the calculation of B0.
;
; Calls       : PB0R
;               ANYTIM2UTC
;
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Utilities, coordinates.
;               
; Prev. Hist. : Yohkoh routine by Hudson/Wuelser.
;
; Written     : CDS version, C D Pike, RAL, 6 Sept 93
;               
; Modified    : To use CDS time and pb0r routines, CDP, 17-May-94
;		Version 2, William Thompson, GSFC, 14 November 1994
;			Modified .DAY to .MJD
;
; Version     : Version 3, 14 November 1994
;-            


function hel2arcmin, ns, ew, date = datin

if datatype(ns) eq 'STR' then begin
  n = fix(strmid(ns,1,2))
  if strupcase(strmid(ns,0,1)) eq 'S' then n = -n
  w = fix(strmid(ew,1,2))
  if strupcase(strmid(ew,0,1)) eq 'E' then w = -w
endif else begin
  n = ns
  w = ew
endelse

;
;  convert to radians
;
lon = w/!radeg 
colat = (90. - n)/!radeg

;
; get system time, if required
;
if n_elements(datin) eq 0 then get_utc,dat  else dat = anytim2utc(datin)
if dat.mjd eq 0 then begin
   print,'HEL2ARCMIN: error in date format specification.'
   return,[-99.,-99.]
endif
 
;
; get B0 and solar radius
;
angles = pb0r(dat)

;
; convert radius to arcmin
;
sunr = angles(2)                       

;
; vect is the (x,y,z) location of the point for b0 = 0, where x is in the
; direction of Texas, y is west, and z is north. vect1 is rotated by b0. 
;   

answer = fltarr(2,max([n_elements(colat),n_elements(lon),n_elements(sunr)]))

;
;  calculate the result
;
b0 = angles(1)/!radeg
scl = sin(colat)
answer(0,*) = scl * sin(lon) * sunr
answer(1,*) = (-scl * cos(lon) * sin(b0) + cos(colat) * cos(b0) ) * sunr

return, answer

end
