pro dbdelete, list, name
;+
; Project     :	SOHO - CDS
;
; Name        :	
;	DBDELETE
; Purpose     :	
;	Deletes specified entries from data base
; Explanation :	
;	Deletes specified entries from data base
;
;       The procedure is rather slow because the entire database is re-
;       created with the specified entries deleted.
;
; Use         :	
;	DBDELETE, list, [ name ]   
;
;	EXAMPLE: Delete entries in a database STARS where RA=DEC = 0.0
;
;        IDL> !PRIV= 3                           ;Set privileges
;        IDL> dbopen,'STARS',1                   ;Open for update
;        IDL> list = dbfind('ra=0.0,dec=0.0')    ;Obtain LIST vector
;        IDL> dbdelete, list             ;Delete specified entries from db
;
; Inputs      :	
;	list - list of entries to be deleted, scalar or vector
;	name - optional name of data base, scalar string.  If not specified
;		then the data base file must be previously opened for update 
;               by DBOPEN.
;
; Opt. Inputs :	
;        Set the non-standard system variable !DEBUG = 2 to obtain 
;        additional diagnostics   
;
; Outputs     :	None.
;
; Opt. Outputs:	None.
;
; Keywords    :	None.
;
; Calls       :	DB_INFO, DBCLOSE, DBINDEX, DBOPEN, DBPUT, ZPARCHECK
;
; Common      :	
;	DBCOM
;
; Restrictions:	
;	!PRIV must be at least 3 to execute.
;
; Side effects:	
;	The data base file (ZDBASE:name.DBF) is modified by removing the
;	specified entries and reordering the remaining entry numbers
;       accordingly (ie. if you delete entry 100, it will be replaced
;       by entry 101 and the database will contain 1 less entry.
;
; Category    :	Utilities, Database.
;
; Prev. Hist. :	
;	Version 2  D. Lindler  July, 1989
;       Updated documentation   W. Landsman    December 1992
;
; Written     :	D. Lindler, GSFC/HRS, July 1989
;
; Modified    :	Version 1, William Thompson, GSFC, 29 March 1994
;			Incorporated into CDS library
;		Version 2, William Thompson, GSFC, 28 February 1995
;			Fixed bug when external representation used.
;
; Version     :	Version 2, 28 February 1995
;-
;
;-------------------------------------------------------------------------------
  On_error,2

  if N_params() EQ 0 then begin
      print,'Syntax - DBDELETE, entry, [ dbname ]'
      return
  endif 

; data base common block

 common db_com,QDB,QITEMS,QDBREC

; Determine whether or not the database uses external data representation.

 external = qdb(119) eq 1

; Check parameters

 zparcheck, 'DBDELETE', list, 1, [1,2,3], [0,1], 'entry list'
 if N_params() GT 1 then $
	zparcheck, 'dbdelete', name, 2, 7, 0, 'data base name'
 
 if !PRIV lt 3 then $
        message,'!priv must be at least 3 to execute'

; Open data base if name supplied

  if N_params() GT 1 then dbopen,name,1 else begin    ;Open specified database

     if not db_info( 'OPEN') then $
        message,'No database open for update'
     if not db_info('update') then $
            message,'Database '+ db_info('NAME',0) + ' not open for update'
  
   endelse

; Create vector if list is a scalar

  outrec = 0L                           ; Create counter of output record

; loop on entries in data base

  qnentry = db_info('ENTRIES',0)
  
  for i = 1L, qnentry do begin

        ; Is it to be kept?

	found = where( list EQ i, Nfound)

	if !debug ge 2 then print,i,nfound           ; allow diags.

	if ( Nfound LE 0 ) then begin
		outrec = outrec + 1                ; increment counter
		if ( outrec NE i ) then begin
			entry = qdbrec(i)
			tmp = outrec
			if external then byteorder,tmp,/htonl
			dbput, 0, tmp, entry   ; modify entry number
			qdbrec(outrec) = entry
		endif
	endif
  endfor

; Update adjusted total number of entries.

  qdb(84) = byte( outrec,0,4 )

; Update index file

  dbindex

  if N_params() GT 1 then dbclose

  return  ; dbdelete
  end  ; dbdelete
