function dbget,item,values,listin,SILENT=silent, FULLSTRING = fullstring
;+
; Project     :	SOHO - CDS
;
; Name        :	
;	DBGET()
; Purpose     :	
;	Find entry number of fields with specified values.
; Explanation :	
;	Find entry number of fields which contain specified values in
;	a specified item.   DBGET is an altnerative to DBFIND when the
;	desired search values are not easily expressed as a string.  
;
; Use         :	
;	list = dbget( item, values, [ listin ], SILENT = )
;
;	EXAMPLE:  Get info on selected HD stars in Bright Star catalogue
;
;	IDL> dbopen, 'YALE_BS' 
;	IDL> hdno = [1141,2363,3574,4128,6192,6314,6668]    ;Desired HD numbers
;	IDL> list = dbget( 'HD', hdno )        ;Get corresponding entry numbers
;
; Inputs      :	
;	item - Item name or number
;	values -  scalar or vector containing item values to search for.
;
; Opt. Inputs :	
;	listin - list of entries to be searched.  If not supplied, or
;		set to -1, then all entries are searched
;
; Outputs     :	
;	list - vector giving the entry number of entries containing desired
;		item values.  The number of elements in  LIST may be different 
;		from that of VALUE, since a value might be located zero, once, 
;		or many times in the database.  Use the function DBMATCH if a 
;		one to one correspondence is desired between VALUES and LIST. 
;
;	!ERR is set to the number of entries found
;
; Opt. Outputs:	None.
;
; Keywords    :	
;	SILENT - If this keyword is supplied, then DBGET will not display
;		the number of entries found
;	FULLSTRING - By default, one has a match if a search string is 
;		included in any part of a database value (substring match).   
;		But if /FULLSTRING is set, then all characters in the database
;		value must match the search string (excluding leading and 
;		trailing blanks).    Both types of string searches are case
;		insensitive.
;
; Calls       :	DB_INFO, DB_ITEM, DB_ITEM_INFO, DBEXT, DBFIND_SORT, DBSEARCH
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Utilities, Database.
;
; Prev. Hist. :	
;	Written,    W. Landsman      STX     February, 1989
;
; Written     :	W. Landsman, GSFC/UIT (STX), February 1989
;
; Modified    :	Version 1, William Thompson, GSFC, 29 March 1994
;			Incorporated into CDS library.
;		Version 2, William Thompson, GSFC, 14 March 1995
;			Added keyword FULLSTRING
;
; Version     :	Version 2, 14 March 1995
;-
;
 On_error,2                                   ;Return to caller

 if N_params() LT 2 then begin
   print,'Syntax --  list = DBGET( item, values, [listin, SILENT =] )
   return,-1
 endif
   
 if N_params() LT 3 then listin = lonarr(1)-1

 nvals = N_elements(values)

 if nvals EQ 0 then message,'No search values supplied'

 db_item, item, itnum
 index = db_item_info( 'INDEX', itnum)
 list = listin
 
 if nvals EQ 1 then val = [values,values] $  ;Need at least 2 elements
               else val = values 

 if index(0) GE 2 then begin                              ;Sorted item
    if N_elements(list) EQ 1 then list = lonarr(1) + list
    dbfind_sort, itnum(0), nvals, val, list, FULLSTRING = fullstring

 endif else begin                                        ;Non-sorted item
    dbext, list, itnum, itvals
    dbsearch, nvals, val, itvals, good, FULLSTRING = fullstring
    if list(0) ne -1 then list = list(good) else list = good+1
 endelse

 if !ERR LE 0 then begin
     if not keyword_set(SILENT) then $
         print, 'No entries found by DBGET in ' + db_info( 'NAME',0 )
     list = intarr(1)  

 endif else  if not keyword_set( SILENT ) then $
          print,!ERR,' entries found in '+db_info('name',0)

 return, list( sort(list) )

 end
