;+
; Project     : SOHO - CDS     
;                   
; Name        : FIX_ZDBASE()
;               
; Purpose     : To control the value of env. var. ZDBASE.
;               
; Explanation : The environment variable ZDBASE controls access to the
;               CDS databases.  The user may have access to either a private
;               set of data bases or the 'official' CDS set.  This function
;               allows the uset to set the ZDBASE variable to be the
;               equivalent of either ZDBASE_CDS or ZDBASE_USER depending
;               on the database access required.  These latter two variables
;               must be defined before this routine is called.  The original
;               definition of ZDBASE is stored in common and can be restored to
;               that variable by use of the /ORIGINAL keyword.
;               
; Use         : IDL> status = fix_zdbase(/user, errmsg=errmsg)
;
;               first time used so record the current value of ZDBASE and set
;               its current value to be that of ZDBASE_USER
;
;               IDL> status = fix_zdbase(/cds, errmsg=errmsg)
;               
;               if not the first time used then just set current ZDBASE
;               definition to that of ZDBASE_CDS
;
;               IDL> status = fix_zdbase(/orig, errmsg=errmsg)
;
;               all finished, so restore ZDBASE to its original value.
;
;               Note that this routine is more likely to be used within other
;               procedures rather than at a user level as in the above
;               example.
;
;    
; Inputs      : None (see keywords).
;               
; Opt. Inputs : None
;               
; Outputs     : None (Values in common may change)
;               
; Opt. Outputs: None
;               
; Keywords    : USER      -   switch ZDBASE to value of ZDBASE_USER
;               CDS       -   switch ZDBASE to value of ZDBASE_CDS
;               ORIGINAL  -   restore original value of ZDBASE
;               ERRMSG    -   if defined on entry any error messages
;                             will be returned in it.
;
; Calls       : getenv
;               setenv
;
; Common      : zdbase_def
;               
; Restrictions: Uses common block variable to signal whether original value
;               of ZDBASE has been saved or not.  Be careful of the common
;               block's memory.
;               
; Side effects: Environment variable ZDBASE is changed.
;               
; Category    : Util, database
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 17-Jan-95
;               
; Modified    : 
;
; Version     : Version 1, 17-Jan-95
;-            

function fix_zdbase, user=user, cds=cds, original=original, errmsg=errmsg

;
;  common to store original value of ZDBASE
;
common zdbase_def, zdb_initialised, orig_zdbase

;
; check only one keyword set
;
if keyword_set(cds)+keyword_set(user)+keyword_set(original) ne 1 then begin
   if keyword_set(errmsg) then begin
      errmsg='Use: One and only one of /user,/cds,/original keywords must be set.'
   endif else begin
      print,'Use: One and only one of /user,/cds,/original keywords must be set.'
   endelse
   return,0
endif

;
;  first use (original value undefined or set to zero)?  
;  If so record value of ZDBASE
;
first = 0
if n_elements(zdb_initialised) eq 0 then first = 1
if not first then begin
   if zdb_initialised eq 0 then first = 1
endif
if first then begin
   orig_zdbase = getenv('ZDBASE')
   if orig_zdbase eq '' then begin   
      message = 'Variable ZDBASE not defined.'
      if keyword_set(errmsg) then begin
         errmsg = message
         return,0
      endif else begin
         message, message
      endelse
   endif
   zdb_initialised = 1
endif 

;
;  USER option
;
if keyword_set(user) then begin
   u = getenv('ZDBASE_USER')
   if u ne '' then begin
      command = "setenv,'ZDBASE="+u+"'"
      status = execute(command)
      if not status then begin
         message = 'Unable to set variable ZDBASE to user value.'
         if keyword_set(errmsg) then begin
            errmsg = message
            return,0
         endif else begin
            message, message
         endelse
      endif else begin
         return,1
      endelse   
   endif else begin
      message = 'Variable ZDBASE_USER is not defined.'
      if keyword_set(errmsg) then begin
         errmsg = message
         return,0
      endif else begin
         message, message
      endelse
   endelse    
endif


;
;  CDS option
;
if keyword_set(cds) then begin
   u = getenv('ZDBASE_CDS')
   if u ne '' then begin
      command = "setenv,'ZDBASE="+u+"'"
      status = execute(command)
      if not status then begin
         message = 'Unable to set variable ZDBASE to cds value.'
         if keyword_set(errmsg) then begin
            errmsg = message
            return,0
         endif else begin
            message, message
         endelse
      endif else begin
         return,1
      endelse   
   endif else begin
      message = 'Variable ZDBASE_CDS is not defined.'
      if keyword_set(errmsg) then begin
         errmsg = message
         return,0
      endif else begin
         message, message
      endelse
   endelse    
endif


;
;  return to original definition, first check initialised flag is defined
;  and not set to zero
;
if keyword_set(original) then begin
   command = "setenv,'ZDBASE="+orig_zdbase+"'"
   status = execute(command)
   if not status then begin
      message = 'Unable to reset ZDBASE to original value.'
      if keyword_set(errmsg) then begin
         errmsg = message
         return,0
      endif else begin
         message, message
      endelse
   endif else begin
      zdb_initialised = 0
      orig_zdbase = ''
      return,1
   endelse   
endif

end


