pro fits_info, filename, SILENT=silent,TEXTOUT=textout, N_ext=n_ext
;+
; Project     :	SOHO - CDS
;
; Name        :	
;	FITS_INFO
;
; Purpose     :	Provides info about FITS file, primary and extensions.
;
; Explanation :	
;	Provide information about the contents of a FITS file, (primary header
;	and data if any + extensions).  Information is printed at the terminal
;	and/or stored in a common block
;
; Use         :	
;	FITS_INFO, filename, [ SILENT = , TEXTOUT = , N_ext = ]
;
;	Example:  Display info about all FITS files of the form '*.fit' in the
;	current directory
;
;		IDL> fits_info, '*.fit'
;
;	Any time a *.fit file is found which is *not* in FITS format, an error 
;	message is displayed at the terminal and the program continues
;
; Inputs      :	
;	FILENAME = Scalar or vector string giving the name of the FITS file(s)
;		Can include wildcards such as '*.fits'
;
; Opt. Inputs :	None.
;
; Outputs     :	Information is printed to the screen.
;
; Opt. Outputs:	None.
;
; Keywords    :	
;	SILENT - This key word will suppress display of the file description
;		on the terminal
;	TEXTOUT - specifies output device.
;		textout=1        TERMINAL using /more option
;		textout=2        TERMINAL without /more option
;		textout=3        <program>.prt
;		textout=4        laser.tmp
;		textout=5        user must open file, see TEXTOPEN
;		textout = filename (default extension of .prt)
;	N_ext - Returns an integer scalar giving the number of extensions in
;		the FITS file
;
; Calls       :	
;	GETTOK, STRN, SXPAR, TEXTOPEN, TEXTCLOSE 
;
; Common      :	
;	DESCRIPTOR = File descriptor string of the form N_hdrrec Naxis IDL_type
;		Naxis1 Naxis2 ... Naxisn [N_hdrrec table_type Naxis
;		IDL_type Naxis1 ... Naxisn] (repeated for each extension) 
;		See the procedure RDFITS_STRUCT for an example of the
;		use of this common block
;
; Restrictions:	
;	Uses the non-standard system variable !TEXTOUT to select an output
;	device if the TEXTOUT keyword is not supplied
;
; Side effects:	None.
;
; Category    :	Data handling, I/O, FITS, Generic
;
; Prev. Hist. :	
;	Written, K. Venkatakrishna, Hughes STX, May 1992
;	Added N_ext keyword, and table_name info, G. Reichert
;	Some cleaning up of the code W. Landsman   October 1992
;	Work on *very* large FITS files   October 92
;
; Written     :	K. Venkatakrishna, Hughes STX, May 1992
;
; Modified    :	Version 1, William Thompson, GSFC, 10 June 1994
;			Incorporated into CDS library.
;		Version 2, William Thompson, GSFC, 9 January 1995
;			Incorporated following changes:
;
;       More checks to recognize corrupted FITS files     February, 1993
;	Proper check for END keyword    December 1994
;	Correctly size variable length binary tables   December 1994
;
; Version     :	Version 2, 9 January 1995
;-
;
 COMMON descriptor,fdescript

 if N_params() lt 1 then begin
     print,'Syntax - fits_info, filename, [/SILENT, TEXTOUT =, N_ext = ]'
     return
 endif

 fil = findfile( filename, COUNT = nfiles)
 if nfiles EQ 0 then message,'No files found'

 silent = keyword_set( SILENT )
 if not keyword_set( TEXTOUT ) then textout = !TEXTOUT    
 textopen, 'FITS_INFO', TEXTOUT=textout

 for nf = 0, nfiles-1 do begin

    file = fil(nf)

    openr, lun1, file, /GET_LUN, /BLOCK

    N_ext = -1
    fdescript = ''
    extname = ['']

   START:  
   ON_IOerror, BAD_FILE
   descript = ''
   
   point_lun, -lun1, pointlun
   test = bytarr(8)
   readu, lun1, test

   if N_ext EQ -1 then begin
        if string(test) NE 'SIMPLE  ' then goto, BAD_FILE
   endif else begin
        if string(test) NE 'XTENSION' then goto, END_OF_FILE
   endelse
   point_lun, lun1, pointlun

;                               Read the header
   hdr = bytarr(80, 36, /NOZERO)
   N_hdrblock = 1
   readu, lun1, hdr
   hd = string( hdr > 32b)
;                               Get values of BITPIX, NAXIS etc.
   bitpix = sxpar(hd, 'BITPIX')
   if !ERR EQ -1 then $ 
          message, 'WARNING - FITS header missing BITPIX keyword',/CON
   Naxis = sxpar( hd, 'NAXIS')
   if !ERR EQ -1 then message, $ 
           'WARNING - FITS header missing NAXIS keyword',/CON
   simple = sxpar( hd, 'SIMPLE')
   exten = sxpar( hd, 'XTENSION')
   Ext_type = strmid( strtrim( exten ,2), 0, 8)      ;Use only first 8 char
   gcount = sxpar( hd, 'GCOUNT') > 1
   pcount = sxpar( hd, 'PCOUNT')

   isel = where( strpos(hd,'EXTNAME =') GE 0, N_extname)  ;find extension name

   if ( N_extname GE 1 ) then begin

            hdd = hd( isel(0) )
            dum = gettok( hdd, '=' )
            extname = [ extname, strtrim(hdd,2) ]

   endif else extname = [ extname, '' ]

   if strn(Ext_type) NE '0' then begin
        if (gcount NE 1) or (pcount NE 0) then $
	     ext_type = 'VAR_' + ext_type
	descript = descript + ' ' + Ext_type
  endif

   descript = descript + ' ' + strn(Naxis)

   case BITPIX of
      8:   IDL_type = 1     ; Byte
     16:   IDL_type = 2     ; Integer*2
     32:   IDL_type = 3     ; Integer*4
    -32:   IDL_type = 4     ; Real*4 
    -64:   IDL_type = 5     ; Real*8
   ELSE: begin 
         message, ' Illegal value of BITPIX = ' + strn(bitpix) + $
         ' in header',/CON
         goto, SKIP
         end
   endcase

  if Naxis GT 0 then begin
         descript = descript + ' ' + strn(IDL_type)
         Nax = sxpar( hd, 'NAXIS*')
         if N_elements(Nax) LT Naxis then begin 
              message, $
                 'ERROR - Missing required NAXISi keyword in FITS header',/CON
                  goto, SKIP
         endif
         for i = 1, Naxis do descript = descript + ' '+strn(Nax(i-1))
  endif

  end_rec = where( strtrim(strmid(hd,0,8),2) EQ  'END')

;  Read header records, till end of header is reached

 while (end_rec(0) EQ -1) and (not eof(lun1) ) do begin
       hdr = bytarr(80, 36, /NOZERO)
       readu,lun1,hdr
       hd = string( hdr > 32b)
       end_rec = where( strtrim(strmid(hd,0,8),2) EQ  'END')
       n_hdrblock = n_hdrblock + 1
 endwhile

 n_hdrec = 36*(n_hdrblock-1) + end_rec(0) + 1         ; size of header
 descript = strn( n_hdrec ) + descript

;  If there is data associated with primary header, then find out the size

 if Naxis GT 0 then begin
         ndata = Nax(0) 
         if naxis GT 1 then for i = 2, naxis do ndata=ndata*Nax(i-1)
 endif else ndata = 0

 nbytes = (abs(bitpix)/8) * gcount * (pcount + ndata)
 nrec = long(( nbytes +2879)/ 2880)

; Skip the headers and data records

   point_lun, -lun1, pointlun
   pointlun = pointlun + nrec*2880L
   point_lun, lun1, pointlun

; Check if all headers have been read 

 if ( simple EQ 0 ) AND ( strlen(strn(exten)) EQ 1) then goto, END_OF_FILE  

    N_ext = N_ext + 1

; Append information concerning the current extension to descriptor

    fdescript = fdescript + ' ' + descript

; Check for EOF
    if not eof(lun1) then goto, START
;
 END_OF_FILE:  
 extname = extname(1:*)            ;strip off bogus first value
                                  ;otherwise will end up with '' at end

 if not (SILENT) then begin
 printf,!textunit,file,' has ',strn(N_ext),' extensions'
 printf,!textunit,'Primary header: ',gettok(fdescript,' '),' records'

 Naxis = gettok( fdescript,' ' ) 

 If Naxis NE '0' then begin

 case gettok(fdescript,' ') of

 '1': image_type = 'Byte'
 '2': image_type = 'Integer*2'    
 '3': image_type = 'Integer*4'
 '4': image_type = 'Real*4'
 '5': image_type = 'Real*8'

 endcase

 image_desc = 'Image -- ' + image_type + ' array ('
 for i = 0,fix(Naxis)-1 do image_desc = image_desc + ' '+ gettok(fdescript,' ')
 image_desc = image_desc+' )'

 endif else image_desc = 'No data'
 printf,!textunit, format='(a)',image_desc

 if N_ext GT 0 then begin
  for i = 1,N_ext do begin

  printf, !TEXTUNIT, 'Extension ' + strn(i) + ' -- '+extname(i)

  header_desc = '               Header : '+gettok(fdescript,' ')+' records'
  printf, !textunit, format = '(a)',header_desc

  table_type = gettok(fdescript,' ')

  case table_type of
   'A3DTABLE' : table_desc = 'Binary Table'
   'BINTABLE' : table_desc = 'Binary Table'
   'VAR_BINTABLE': table_desc = 'Variable length Binary Table'
   'TABLE':     table_desc = 'ASCII Table'
    ELSE:       table_desc = table_type
  endcase

  table_desc = '               ' + table_desc + ' ( '
  table_dim = fix( gettok( fdescript,' ') )
  table_type = gettok(fdescript,' ')

  for j = 0, table_dim-1 do $
         table_desc = table_desc + gettok(fdescript,' ') + ' '
  table_desc = table_desc + ')'

  printf,!textunit, format='(a)',table_desc
 endfor
 endif

  printf, !TEXTUNIT, ' '
  endif 
  SKIP: free_lun, lun1
  endfor
  textclose, TEXTOUT=textout
  return

 BAD_FILE:
     message, 'Error reading FITS file ' + file, /CON
    goto,SKIP
 end
