Pro sxaddpar, Header, Name, Value, Comment, Location, before=before, $
                                       after=after , format=format
;+
; PROJECT     : SOHO - CDS
;
; NAME:
;	SXADDPAR
; PURPOSE:
;	Add or modify a parameter in a FITS or STSDAS header array.
;
; CALLING SEQUENCE:
;	sxaddpar, Header, Name, Value, [ Comment,  Location,
;				BEFORE =, AFTER = , FORMAT= ]
;
; INPUTS:
;	Header = String array containing FITS or STSDAS header.    The
;		length of each element must be 80 characters.    If not 
;		defined, then SXADDPAR will create an empty FITS header array.
;
;	Name = Name of parameter. If Name is already in the header the value 
;		and possibly comment fields are modified.  Otherwise a new 
;		record is added to the header.  If name = 'HISTORY' then the 
;		value will be added to the record without replacement.  In 
;		this case the comment parameter is ignored.
;
;	Value = Value for parameter.  The value expression must be of the 
;		correct type, e.g. integer, floating or string.  String values
;		 of 'T' or 'F' are considered logical values.
;
; OPTIONAL INPUT PARAMETERS:
;	Comment = String field.  The '/' is added by this routine.  Added 
;		starting in position 31.    If not supplied, or set equal to 
;		'', then the previous comment field is retained (when found) 
;
;	Location = Keyword string name.  The parameter will be placed before the
;		location of this keyword.    This parameter is identical to
;		the BEFORE keyword and is kept only for consistency with
;		earlier versions of SXADDPAR.
;
; OPTIONAL INPUT KEYWORD PARAMETERS:
;	BEFORE	= Keyword string name.  The parameter will be placed before the
;		location of this keyword.  For example, if BEFORE='HISTORY'
;		then the parameter will be placed before the first history
;		location.  This applies only when adding a new keyword;
;		keywords already in the header are kept in the same position.
;
;	AFTER	= Same as BEFORE, but the parameter will be placed after the
;		location of this keyword.  This keyword takes precedence over
;		BEFORE.
;
;	FORMAT	= Specifies FORTRAN-like format for parameter, e.g. "F7.3".  A
;		scalar string should be used.  For complex numbers the format
;		should be defined so that it can be applied separately to the
;		real and imaginary parts.
;
; OUTPUTS:
;	Header = updated FITS header array.
;
; RESTRICTIONS:
;	Warning -- Parameters and names are not checked
;		against valid FITS parameter names, values and types.
;
; MODIFICATION HISTORY:
;	DMS, RSI, July, 1983.
;	D. Lindler Oct. 86  Added longer string value capability
;	Converted to NEWIDL  D. Lindler April 90
;	Added Format keyword, J. Isensee, July, 1990
;	Added keywords BEFORE and AFTER. K. Venkatakrishna, May '92
;	Pad string values to at least 8 characters   W. Landsman  April 94
;
;       Version 1, Liyun Wang, GSFC/ARC, September 19, 1994
;          Incoporated into CDS library.
;
; Version     :
;       Version 1, September 19, 1994
;-
   IF N_params() LT 3 THEN BEGIN ;Need at least 3 parameters
      PRINT,'Syntax - Sxaddpar, Header, Name,  Value, [Comment, Postion
      PRINT,'                      BEFORE = ,AFTER = , FORMAT = ]
      RETURN
   ENDIF

; Define a blank line and the END line

   ENDLINE = 'END' +STRING(REPLICATE(32b,77)) ;END line
   BLANK = STRING(REPLICATE(32b,80)) ;BLANK line

;  If Location parameter not defined, set it equal to 'END     '
;
   IF ( N_params() GT 4 ) THEN loc = STRUPCASE(location) ELSE $
      IF KEYWORD_SET( BEFORE) THEN loc = STRUPCASE(before) ELSE $
      IF KEYWORD_SET( AFTER)  THEN loc = STRUPCASE(after) ELSE $
      loc = 'END'

   WHILE STRLEN(loc) LT 8 DO loc = loc + ' '

   IF N_params() LT 4 THEN comment = '' ;Is comment field specified?

   n = N_elements(header)       ;# of lines in FITS header
   IF (n EQ 0) THEN BEGIN       ;header defined?
      header=STRARR(10)         ;no, make it.
      header(0)=ENDLINE
      n=10
   ENDIF ELSE BEGIN
      s = SIZE(header)          ;check for string type
      IF (s(0) NE 1) OR (s(2) NE 7) THEN $
         MESSAGE,'FITS Header (first parameter) must be a string array'
   ENDELSE

;  Make sure Name is 8 characters long

   nn = STRING(REPLICATE(32b,8)) ;8 char name
   STRPUT,nn,STRUPCASE(name)    ;insert name

;  Extract first 8 characters of each line of header, and locate END line

   keywrd = STRMID(header,0,8)  ;Header keywords
   iend = WHERE(keywrd EQ 'END     ',nfound)
   IF nfound EQ 0 THEN header(0)=ENDLINE ;no end, insert at beginning
   iend = iend(0) > 0           ;Make scalar

   IF nn EQ 'HISTORY ' THEN BEGIN ;add history record?
      IF iend LT (n-1) THEN $
         header(iend+1)=ENDLINE $ ;move end up
      ELSE BEGIN
         header = [header,REPLICATE(blank,5)] ;yes, add 5.
         header(n) = ENDLINE
      ENDELSE
      newline = blank
      STRPUT,newline,nn+STRING(value),0
      header(iend) = newline    ;add history rec.
      RETURN
   ENDIF                        ;history

; Find location to insert keyword

   ipos  = WHERE(keywrd EQ nn,nfound)
   IF nfound GT 0 THEN BEGIN
      i = ipos(0)
      IF comment EQ '' THEN comment=STRMID(header(i),32,48) ;save comment?
      GOTO, REPLACE    
   ENDIF

   IF loc NE '' THEN BEGIN
      iloc =  WHERE(keywrd EQ loc,nloc)
      IF nloc GT 0 THEN BEGIN
         i = iloc(0)
         IF KEYWORD_SET(after) AND loc NE 'HISTORY ' THEN i = i+1 < iend 
         IF i GT 0 THEN header=[header(0:i-1),blank,header(i:n-1)] $
         ELSE header=[blank,header(0:n-1)]
         GOTO, REPLACE  
      ENDIF
   ENDIF

; At this point keyword and location parameters were not found, so a new
; line is added at the end of the FITS header

   IF iend LT (n-1) THEN BEGIN	;Not found, add more?
      header(iend+1) = ENDLINE	;no, already long enough.
      i = iend                  ;position to add.
   ENDIF ELSE BEGIN		;must lengthen.
      header = [header,REPLICATE(blank,5)] ;add an element on the end
      header(n)=ENDLINE		;save "END"
      i =n-1			;add to end
   END

; Now put value into keyword at line i

REPLACE:    
   h=blank			;80 blanks
   STRPUT,h,nn+'= '             ;insert name and =.
   apost = "'"                  ;quote a quote
   type = SIZE(value)           ;get type of value parameter
   IF type(0) NE 0 THEN $
      MESSAGE,'Keyword Value (third parameter) must be scalar'

   CASE type(1) OF		;which type?

      7: BEGIN
         upval = STRUPCASE(value) ;force upper case.
         IF (upval EQ 'T') OR (upval EQ 'F') THEN BEGIN
            STRPUT,h,upval,29   ;insert logical value.
         END ELSE BEGIN		;other string?
            IF STRLEN(value) GT 18 THEN BEGIN ;long string
               STRPUT, h, apost + value + apost+' /' + comment,10
               header(i) = h
               RETURN
            ENDIF
            STRPUT, h, apost + value,10 ;insert string val
            STRPUT, h, apost, 11 + (STRLEN(value)>8) ;pad string vals
         ENDELSE                ;to at least 8 chars
      ENDCASE

      ELSE: BEGIN
         IF (N_elements(FORMAT) EQ 1) THEN $ ;use format keyword
            v = STRING(value,'('+FORMAT+')' ) ELSE $
            v = STRTRIM(value,2) ;convert to string, default format
         s = STRLEN(v)          ;right justify
         STRPUT,h,v,(30-s)>10   ;insert
      END
   ENDCASE

   STRPUT,h,' /',30             ;add ' /'
   STRPUT, h, comment, 32	;add comment
   header(i) = h		;save line

   RETURN
END
