PRO writefits, filename, data, header, NaNvalue = NaNvalue, Append = Append
;+
; PROJECT     : SOHO - CDS
;
; NAME:
;	WRITEFITS
; PURPOSE:
;	Write an an IDL array into a disk FITS file.    Works with all types
;	of FITS files except random groups
;
; CALLING SEQUENCE:
;	writefits, filename, data [, header, NaNvalue = , /APPEND] 
;
; INPUTS:
;	FILENAME = String containing the name of the file to be written.
;
;	DATA = Image array to be written to FITS file.    If DATA is 
;              undefined or a scalar, then only the FITS header (which
;              must have NAXIS = 0) will be written to disk
;
; OPTIONAL INPUT:
;	HEADER = String array containing the header for the FITS file.
;		 If variable HEADER is not given, the program will generate
;		 a minimal FITS header.
;
; OPTIONAL INPUT KEYWORD:
;       NaNvalue - Value in the data array to be set to the IEEE NaN
;                 condition.   This is the FITS representation of undefined
;                 values 
;       APPEND - If this keyword is set then the supplied header and data
;                array are assumed to be an extension and are appended onto
;                the end of an existing FITS file.    Note that the primary
;                header in the existing file must already have an EXTEND
;                keyword to indicate the presence of an FITS extension.
;
; OUTPUTS:
;	None
;
; CALLS:
;       SXADDPAR, SXDELPAR, CHECK_FITS, HOST_TO_IEEE, HEADFITS
;
; RESTRICTIONS:
;       (1) It recommended that BSCALE and BZERO not be used (or set equal
;           to 1. and 0) with REAL*4 or REAL*8 data.
;       (2) WRITEFITS will remove any group parameters from the FITS header
;
; EXAMPLE:
;       Write a randomn 50 x 50 array as a FITS file creating a minimal header.
;
;       IDL> im = randomn(seed, 50, 50)        ;Create array
;       IDL> writefits, 'test', im             ;Write to a FITS file "test"
;
; PROCEDURES USED:
;       CHECK_FITS, HOST_TO_IEEE, SXDELPAR, SXADDPAR, SXPAR
;
; MODIFICATION HISTORY:
;	WRITTEN, Jim Wofford, January, 29 1989
;       MODIFIED, Wayne Landsman, added BITPIX = -32,-64 support for UNIX
;       Use new BYTEODER keywords 22-Feb-92
;       Modify OPENW for V3.0.0   W. Landsman       Dec 92
;       Work for "windows"   R. Isaacman            Jan 93
;	More checks for null data                   Mar 94
;      Version 1, Liyun Wang, GSFC/ARC, September 19, 1994
;         Incoporated into CDS library
; Version:
;      Version 1, September 19, 1994
;-
   On_error, 2

   IF N_params() LT 2 THEN MESSAGE, /NONAME, $
      'Syntax: WRITEFITS, filename, data,[ header, NaNvalue =, /APPEND ]

; Get information about data

   siz = SIZE( data )      
   naxis = siz(0)               ;Number of dimensions
   IF naxis GT 0 THEN nax = siz( 1:naxis ) ;Vector of dimensions
   lim = siz( naxis+2 )         ;Total number of data points
   type = siz(naxis + 1)        ;Data type

   IF N_elements(header) LT 2 THEN mkhdr, header, data $
   ELSE IF naxis GT 0 THEN $         
      check_FITS, data, header, /UPDATE, /FITS

   hdr = header
   IF NOT KEYWORD_SET( APPEND) THEN BEGIN 
      sxaddpar, hdr, 'SIMPLE', 'T', ' Written by IDL:  ' + !STIME 
      sxdelpar, hdr, [ 'GCOUNT', 'GROUPS', 'PCOUNT', 'PSIZE' ]
   ENDIF
   
; For floating or double precision test for NaN values to write

   IF naxis NE 0 THEN BEGIN

      NaNtest = KEYWORD_SET(NaNvalue) AND ( (type EQ 4) OR (type EQ 5) )
      IF NaNtest THEN NaNpts = WHERE( data EQ NaNvalue, N_NaN)
      
; If necessary, byte-swap the data.    Do not destroy the original data

      vax = (!VERSION.ARCH EQ "vax") OR (!VERSION.ARCH EQ "alpha")
      Little_endian = ( !VERSION.ARCH EQ "mipsel" ) OR $ 
         ( !VERSION.ARCH EQ '386i') OR  $
         ( !VERSION.ARCH EQ '386' ) OR $
         ( !VERSION.OS EQ 'windows')
      
      IF (VAX OR Little_endian) THEN BEGIN
         newdata = data
         host_to_ieee, newdata
      ENDIF

; Write the NaN values, if necessary

      IF NaNtest THEN BEGIN
         IF (N_NaN GT 0) THEN BEGIN
            IF type EQ 4 THEN data(NaNpts) = $
               FLOAT( [ 127b, 255b, 255b, 255b ], 0, 1 ) $
            ELSE IF type EQ 8 THEN data(NaNpts) = $
               DOUBLE( [ 127b, REPLICATE( 255b,7)], 0 ,1)
         ENDIF
      ENDIF
   ENDIF

; Open file and write header information

   IF KEYWORD_SET( APPEND) THEN BEGIN
      IF (STRMID( hdr(0),0,8 ) NE 'XTENSION') THEN BEGIN
         MESSAGE, $
            'ERROR - "XTENSION" must be first keyword in header extension',/CON
         RETURN
      ENDIF
      test = findfile( filename, COUNT = n)
      IF n EQ 0 THEN MESSAGE, $
         'ERROR - FITS file ' + filename + ' not found'
      hprimary = headfits( filename )
      extend = WHERE( STRMID(hprimary,0,8) EQ 'EXTEND  ', Nextend)
      OPENU, unit, filename, /BLOCK, /GET_LUN
      IF Nextend EQ 0 THEN BEGIN
         MESSAGE,'EXTEND keyword not found in primary FITS header',/CON
         MESSAGE,'Recreate primary FITS header with EXTEND keyword ' + $
            'before adding extensions', /CON
         RETURN
      ENDIF
      
      file = fstat(unit)
      nbytes  = file.size
      point_lun, unit, nbytes
      npad = nbytes MOD 2880
      IF npad NE 0 THEN WRITEU, unit, REPLICATE(32b, 2880 - npad)

   ENDIF ELSE BEGIN

      IF !VERSION.OS EQ "vms" THEN $
         OPENW, unit, filename, /NONE, /BLOCK, /GET_LUN, 2880 ELSE $
         OPENW, unit, filename, /GET_LUN

   ENDELSE

; Determine if an END line occurs, and add one if necessary

   endline = WHERE( STRMID(hdr,0,8) EQ 'END     ', Nend)
   IF Nend EQ 0 THEN BEGIN

      MESSAGE,'WARNING - An END statement has been appended to the FITS header',/INF
      hdr = [ hdr, 'END' + STRING( REPLICATE(32b,77) ) ]
      endline = N_elements(hdr) - 1 

   ENDIF
   nmax = endline(0) + 1

; Convert to byte and force into 80 character lines

   bhdr = REPLICATE(32b, 80l*nmax)
   FOR n = 0l, endline(0) DO bhdr(80*n) = byte( hdr(n) )
   npad = 80l*nmax MOD 2880
   WRITEU, unit, bhdr
   IF npad GT 0 THEN WRITEU, unit,  REPLICATE(32b, 2880 - npad)

; Write data
   IF naxis EQ 0 THEN GOTO, DONE
   bitpix = sxpar( hdr, 'BITPIX' )
   nbytes = N_elements( data) * (abs(bitpix) / 8 )
   npad = nbytes MOD 2880

   IF VAX OR LITTLE_ENDIAN THEN $

   WRITEU, unit, newdata  $

   ELSE WRITEU, unit, data 

; ASCII Tables padded with blanks (32b) otherwise pad with zeros
   IF KEYWORD_SET( APPEND) THEN BEGIN
      exten = sxpar( header, 'XTENSION')
      IF exten EQ 'TABLE   ' THEN padnum = 32b ELSE padnum = 0b
   ENDIF ELSE padnum = 0b
   
   IF npad GT 0 THEN WRITEU, unit, REPLICATE( padnum, 2880 - npad)
DONE:
   FREE_LUN, unit  

   RETURN
END
