;+
; Project     : SOHO - CDS     
;                   
; Name        : OUTPLOT
;               
; Purpose     : To overplot time series data on a plot created by UTPLOT
;               
; Explanation : Converts input x-variable to seconds, applies the time offset
;               stored in common and overplots data on a plot created by
;               UTPLOT.
;               
; Use         : IDL> outplot,x,y + normal oplot keywords.
;    
; Inputs      : x - the time variable.  May be either an array of CDS internal
;                                       time structures (as produced by the 
;                                       routine STR2UTC, say) or...
; 
;                                       a string array of date/times in a 
;                                       format translatable by STR2UTC.
;
;               y - the data value to be plotted.
;
;               
; Opt. Inputs : Various keywords.
;               
; Outputs     : None
;               
; Opt. Outputs: None
;               
; Keywords    : dmy        - if the x-variable is a list of date/time strings
;                            this keyword must be used if they are in the
;                            format dd-mm(m)-yyyy as opposed to the CDS 
;                            'official' format of yyyy-mm-dd
;
;               + other standard oplot keywords
;
; Calls       : STR2UTC
;               UTC2SEC
;               SEC2UTC
;               VCHECK
;
;
; Common      : cds_utplot_com  (for passing of time axis offset to outplot)
;               
; Restrictions: Any 2-character year specifications are translated as being
;               between 1950 and 2049 (use 4-characters as necessary).
;
;               
; Side effects: None
;               
; Category    : Data display, plotting
;               
; Prev. Hist. : This is a version of the old OUTPLOT as used on
;               Yohkoh and developed by Tolbert, Schwartz and Morrison.
;       
;
; Written     : This version for CDS by C D Pike, RAL, 20-Apr-94
;               
; Modified    : 
;
; Version     : Version 1, 20-Apr-94
;-            

pro OUTPLOT, x_in, y, $

      channel=channel, charsize=charsize, charthick=charthick,$
      color=color,$

      data=data, device=device, dmy=dmy,$

      font=font,$

      linestyle=linestyle,$

      normal=normal, nsum=nsum,     nolabel=nolabel,$

      psym=psym,$

      symsize=symsize, $

      thick=thick



;
;  common for offset value passed from utplot
;
common cds_utplot_com, cds_xoffset

;
;  tidy up on error
;
on_error,2

;
;  validate input
;
;
;  either CDS internal time structure or array of date/time strings
;
case datatype(x_in,1) of
   'Structure': x = x_in
   'String'   : begin
                   if keyword_set(dmy) then begin
                      x = str2utc(x_in,/dmy)
                   endif else begin
                      x = str2utc(x_in)
                   endelse
                end
    else      : begin 
                   print,' '
                   print,'OUTPLOT: x-variable must be either:'
                   print,'1) an array of CDS internal time structures.
                   print,'  or ...'
                   print,'2) an array of CDS date/time strings.'
                   print,' '
                   return
                end
endcase
;
;  save current settings
;
psave = !p
        !p.channel=vcheck(channel,!p.channel)
        !p.clip=vcheck(clip,!p.clip)
        !p.color=vcheck(color,!p.color)
        !p.linestyle=vcheck(linestyle,!p.linestyle)
        !p.noclip=vcheck(noclip,!p.noclip)
        !p.nsum=vcheck(nsum,!p.nsum)
        !p.psym=vcheck(psym,!p.psym)
        !p.t3d=vcheck(t3d,!p.t3d)
        !p.thick=vcheck(thick,!p.thick)



;
;  perform the over plotting using the defined !x variable
;
oplot,utc2sec(x)-cds_xoffset,y, $
         symsize=vcheck(symsize,1.0)

;
;  restore setup
;
!p = psave

return
end

