;+
; Project     : SOHO - CDS     
;                   
; Name        : PCONVERT()
;               
; Purpose     : Convert Plot DEVICE, DATA, NORMAL and PIXEL coordinates
;               
; Explanation : Uses the data stored for each Plot Region to calculate
;		to and from different coordinate systems.
;               
; Use         : DEVICE_X = PCONVERT(PLOTREG,COORD)
;    
; Inputs      : PLOTREG : A plot region ID as returned by PSTORE or PFIND
;
;		COORD : The coordinate to be converted. See Restrictions.
;               
; Opt. Inputs : None.
;               
; Outputs     : Return value.
;               
; Opt. Outputs: None.
;               
; Keywords    : Y : 	Set to mark that it's the Y coordinate that is
;			supplied.
;
;		DEVICE/DATA/
;		NORMAL/
;		PIXEL  : Set to indicate the type of the supplied coordinate.
;			Default is DEVICE.
;
;		TO_DEVICE/TO_DATA/
;		TO_NORMAL/
;		TO_PIXEL: Set to indicate the type of coordinate to convert
;			into. Default is DATA.
;		
;		LOWER: In converting PIXEL -> other units, it is normally
;			assumed that the supplied coordinate is the 
;			pixel _number_, i.e., the expected output is the
;			value at the center of the pixel. If you are 
;			supplying 'real' pixel values (with pixel 
;			number 2 lying in the interval <1.0, 2.0>) or if
;			you want the coordinates at the left/lower border of
;			where the pixel is displayed, use this keyword to
;			notify. E.g:
;
;			pconvert(P,0,/pixel,/to_data) yields 0.000
;			but:
;			pconvert(P,0,/pixel,/to_data,/lower) yields -0.500
;			given that the data coordinate system is set up
;			to correspond with pixel index numbers.
;
; Calls       : TRIM()
;
; Common      : WSTORE
;               
; Restrictions: There must exist a data coordinate system for the
;		specified plot region, stored with PSTORE()
;
;		When converting to and from PIXEL values it is assumed
;		that:
;
;		For one-dimensional data, the plot is generated
;		with XRANGE = [MIN(x),MAX(x)],XSTYLE=1, and it is
;		assumed that the pixels are placed at constant intervals.
;
;		For two-dimensional data, the plot is generated with
;		XRANGE = [MIN(x)-XSTEP,MAX(x)+XSTEP],XSTYLE=1, (and
;		vice versa for Y) where XSTEP is the (DATA coordinate) 
;		distance between two consecutive pixels.
;		This is also necessary to get the axis ticks correct
;		on the plot/TV'ed data.
;               
; Side effects: None.
;               
; Category    : Utilities, Graphics
;               
; Prev. Hist. : Combined earlier ppix2dat/ppix2dev .... etc.
;
; Written     : SV Hagfors Haugan, 30-May-1994
;               
; Modified    : SVHH, 1-June-1994 -- Found that IDL sometimes chops a pixel
;			when calculating !X.S/!Y.S - fixed.
;		SVHH, 25-November-1994 -- The above "fix" removed -- better
;			to make sure that IDL is not missing the pixel
;			using e.g., plot,[..],position=pos+0.0001,/dev
;			--  this (hopefully) keeps IDL from miscalculations
;			of !P.clip etc.
;			Included bounds on /TO_PIXEL conversions.
;
; Version     : 1.2, 25-November-1994
;-            


FUNCTION pconvert,P_reg,C,y=y,lower=lower,$
	DEVICE=DEVICE,data=data,normal=normal,pixel=pixel,$
	to_device=to_device,to_data=to_data,$
	to_normal=to_normal,to_pixel=to_pixel
  
  common wstore,D,P,N,XX,YY,dataxx,datayy
  
  IF N_params()	lt 2 THEN message,'Use: result = pconvert(Plot_reg,COORD)'
  
  IF N_elements(P_reg) eq 0 or N_elements(P_reg) gt 1 THEN $
			  message,'Plot_region must be scalar'
  
  i = P_reg ; shorthand
  
  IF i lt 0 or i ge N_elements(D) THEN $
	message,'Plot region number must be [0,..,'+trim(N_elements(D)-1)+']'
    
  IF NOT Keyword_SET(Y)	THEN BEGIN
      ddata = dataxx(i)	>1
      odata= datayy(i) >1
      scrn = P(i).clip(2)-P(i).clip(0)
      j	= P(i).clip(0)
      Size = D(i).X_size
      s	= XX(i).s
  END ELSE BEGIN
      ddata = datayy(i)	>1
      odata= dataxx(i) >1
      scrn = P(i).clip(3)-P(i).clip(1)
      j	= P(i).clip(1)
      Size = D(i).Y_size
      s	= YY(i).s
  EndELSE

;
; Default input is DEVICE
;
  IF NOT (Keyword_SET(DEVICE) or Keyword_SET(data) $
	  or Keyword_SET(normal) or Keyword_SET(pixel))	THEN DEVICE=1
  
;
; Default output is DATA
;
  IF NOT (Keyword_SET(to_device) or Keyword_SET(to_data) $
	  or Keyword_SET(to_normal) or Keyword_SET(to_pixel)) THEN to_data=1
;
;
;
  
  IF (Keyword_SET(pixel)) and odata gt 1  $
	      and NOT Keyword_SET(lower) THEN C	= C + .5d
  
  IF Keyword_SET(DEVICE) THEN BEGIN
      
      IF Keyword_SET(to_device)	THEN BEGIN
	  return,c
      END ELSE IF Keyword_SET(to_data) THEN BEGIN
	  return,(double(C)/Size-s(0))/s(1)
      END ELSE IF Keyword_SET(to_normal) THEN BEGIN
	  return,double(C)/double(Size)
      END ELSE IF Keyword_SET(to_pixel)	THEN BEGIN
	  return,(((C-J)*double(ddata)/scrn) >0) < (ddata-0.0001)
      END
      
  END ELSE IF Keyword_SET(data)	THEN BEGIN
      
      IF Keyword_SET(to_device)	THEN BEGIN
	  return,(C*s(1)+s(0))*Size
      END ELSE IF Keyword_SET(to_data) THEN BEGIN
	  return,C
      END ELSE IF Keyword_SET(to_normal) THEN BEGIN
	  return,C*s(1)+s(0)
      END ELSE IF Keyword_SET(to_pixel)	THEN BEGIN
	  dev =	(C*s(1)+s(0))*Size
	  return,(((dev - j)*ddata/scrn) > 0) < (ddata-0.0001)
      END
      
  END ELSE IF Keyword_SET(normal) THEN BEGIN
      
      IF Keyword_SET(to_device)	THEN BEGIN
	  return,C*Size
      END ELSE IF Keyword_SET(to_data) THEN BEGIN
	  return,(C-s(0))/s(1)
      END ELSE IF Keyword_SET(to_normal) THEN BEGIN
	  return,c
      END ELSE IF Keyword_SET(to_pixel)	THEN BEGIN
	  return,(C*Size - j)*ddata/scrn > 0 < (ddata-0.0001)
      END
      
  END ELSE IF Keyword_SET(pixel) THEN BEGIN
      
      IF Keyword_SET(to_device)	THEN BEGIN
	  return, J + C*scrn/ddata
      END ELSE IF Keyword_SET(to_data) THEN BEGIN
	  return, ((J +	double(C)*scrn/ddata)/(Size) -s(0))/s(1)
      END ELSE IF Keyword_SET(to_normal) THEN BEGIN
	  return,(J + C*scrn/ddata)/Size
      END ELSE IF Keyword_SET(to_pixel)	THEN BEGIN
	  return,c
      END
      
  EndIF
  
END

