;+
; Project     : SOHO - CDS     
;                   
; Name        : UTPLOT
;               
; Purpose     : To plot time series data with sensible time axis labelling.
;               
; Explanation : Uses the routine DEF_UTPLOT to manipulate the IDL plotting
;               variables to set up sensible time axis labels and intervals.
;               After the setup the data are plotted with a normal call to 
;               the routine PLOT.
;               
; Use         : IDL> utplot,x,y,timerange=[] + normal plot keywords.
;    
; Inputs      : x - the time variable.  May be either an array of CDS internal
;                                       time structures (as produced by the 
;                                       routine STR2UTC, say) or...
; 
;                                       a string array of date/times in a 
;                                       format translatable by STR2UTC.
;
;               y - the data value to be plotted.
;
;               
; Opt. Inputs : Various keywords.
;               
; Outputs     : None
;               
; Opt. Outputs: None
;               
; Keywords    : timerange  - a 2-element array each element is either a CDS
;                            internal time structure or a CDS date/time string.
;                            Note that this time range is still adjusted 
;                            slightly for actual plotting unless the keyword
;                            xstyle=1 is specified.
;
;               dmy        - if the x-variable is a list of date/time strings
;                            this keyword must be used if they are in the
;                            format dd-mm(m)-yyyy as opposed to the CDS 
;                            'official' format of yyyy-mm-dd
;
;               + other standard plot keywords
;
; Calls       : DEF_UTPLOT
;               STR2UTC
;               UTC2SEC
;               SEC2UTC
;               VCHECK
;
;
; Common      : cds_utplot_com  (for passing of time axis offset to outplot)
;               
; Restrictions: The MJD of any dates used must be positive.
;               Any 2-character year specifications are translated as being
;               between 1950 and 2049 (use 4-characters as necessary).
;
;               The time array passed to UTPLOT must be in ascending order
;               of time - sort it if necessary!
;
; Side effects: None
;               
; Category    : Data display, plotting
;               
; Prev. Hist. : This is a severely trimmed version of the old UTPLOT as used on
;               Yohkoh and developed by Tolbert, Schwartz and Morrison.
;               I may have trimmed too much, please let me know.
;
; Written     : This version for CDS by C D Pike, RAL, 20-Apr-94
;               
; Modified    : Version 1, C D Pike, RAL, 20-Apr-94
;		Version 2, William Thompson, GSFC, 14 November 1994
;			Modified .DAY to .MJD
;			 , Nathan Rich, NRL/LASCO, Feb 2000    
;		Add [XY]GRIDSTYLE keywords
;
; Version     : Version 2, 14 November 1994
;-            

pro UTPLOT, x_in, y, $
      background=background,$

      channel=channel, charsize=charsize, charthick=charthick,$
      clip=clip,       color=color,$

      data=data, device=device, dmy=dmy,$

      font=font,$

      linestyle=linestyle,$

      noclip=noclip, nodata=nodata, noerase=noerase,$
      normal=normal, nsum=nsum,     nolabel=nolabel,$

      position=position, psym=psym,$

      subtitle=subtitle, symsize=symsize, $

      thick=thick, ticklen=ticklen,$
      title=title, tick_unit=tick_unit, timerange=timerange,$

      xcharsize=xcharsize, xmargin=xmargin,$
      xstyle=xstyle,       xticklen=xticklen,$
      xtitle=xtitle,       xgridstyle=xgridstyle, $

      ycharsize=ycharsize, ymargin=ymargin, yminor=yminor,$
      yrange=yrange,       ystyle=ystyle,   yticklen=yticklen,$
      ytickname=ytickname, yticks=yticks,   ytickv=ytickv,$
      ytitle=ytitle,       ytype=ytype,     ynozero=ynozero, $
      ygridstyle=ygridstyle


;
;  common for offset value passinf to outplot
;
common cds_utplot_com, cds_xoffset

;
;  tidy up on error
;
on_error,2

;
;  validate input
;
if n_elements(x_in) lt 2 then begin
   print,'Error:  UTPLOT x-variable must have >= 2-elements.'
   return
endif

;
;  either CDS internal time structure or array of date/time strings
;
case datatype(x_in,1) of
   'Structure': x = x_in
   'String'   : begin
                   if keyword_set(dmy) then begin
                      x = str2utc(x_in,/dmy)
                   endif else begin
                      x = str2utc(x_in)
                   endelse
                end
    else      : begin 
                   print,' '
                   print,'UTPLOT: x-variable must be either:'
                   print,'1) an array of CDS internal time structures.
                   print,'  or ...'
                   print,'2) an array of CDS date/time strings.'
                   print,' '
                   return
                end
endcase

;
;  save current global x plotting variable 
;
xsave = !x

;
;  if the time span required is < 10 days then to maintain accuracy, subtract
;  the first time and add it on later (stored in variable cds_xoffset)
;
n = n_elements(x)
if (x(n-1).mjd - x(0).mjd) lt 10 then begin
   cds_xoffset = x(0).mjd * 86400.d0
   x.mjd = x.mjd - x(0).mjd 
endif else begin
   cds_xoffset = 0.0d0
endelse

;
;  do the same with the time range if it was specified.
;
if (keyword_set(timerange)) then begin
    if datatype(timerange,1) ne 'Structure' then begin 
       t0 = str2utc(timerange(0),/dmy)
       t1 = str2utc(timerange(1),/dmy)
    endif else begin
       t0 = timerange(0)
       t1 = timerange(1)
    endelse
    if (t1.mjd - t0.mjd) lt 10 then begin
       t1.mjd = t1.mjd  - t0.mjd 
       t0.mjd = 0  
    endif
    !x.range(0) = utc2sec(t0)
    !x.range(1) = utc2sec(t1)
endif else begin
    xs = utc2sec(x)  
    !x.range(0) = min(xs)
    !x.range(1) = max(xs)
endelse

;
;  set global variable
;
xrange = !x.range

;
;  if xstyle keyword not used then use system value
;
xstyle = vcheck(xstyle,!x.style)


def_utplot,$
           xstyle=xstyle,$ 
           xrange=xrange,$
           cds_xoffset = cds_xoffset


;
;  save initial global plotting variables
;
psave = !p
ysave = !y

;
;  update with any specified in the call
;
!p.background=vcheck(background,!p.background)
!p.channel=vcheck(channel,!p.channel)
!p.charsize=vcheck(charsize,!p.charsize)
!p.charthick=vcheck(charthick,!p.charthick)
!p.clip=vcheck(clip,!p.clip)
!p.color=vcheck(color,!p.color)
!p.font=vcheck(font,!p.font)
!p.linestyle=vcheck(linestyle,!p.linestyle)
!p.noclip=vcheck(noclip,!p.noclip)
!p.noerase=vcheck(noerase,!p.noerase)
!p.nsum=vcheck(nsum,!p.nsum)
!p.position=vcheck(position,!p.position)
!p.psym=vcheck(psym,!p.psym)
!p.subtitle=vcheck(subtitle,!p.subtitle)
!p.thick=vcheck(thick,!p.thick)
!p.ticklen=vcheck(ticklen,!p.ticklen)
!p.title=vcheck(title, !p.title)

!x.charsize=vcheck(xcharsize, !x.charsize)
!x.margin=vcheck(xmargin, !x.margin)
!x.ticklen=vcheck(xticklen, !x.ticklen)
!x.title=vcheck(xtitle,!x.title)
!x.style=vcheck(xstyle,!x.style)  
!x.gridstyle=vcheck(xgridstyle,!x.gridstyle)

!y.charsize=vcheck(ycharsize, !y.charsize)
!y.margin=vcheck(ymargin, !y.margin)
!y.minor=vcheck(yminor, !y.minor)
!y.range=vcheck(yrange, !y.range)
!y.style=vcheck(ystyle, !y.style)
!y.ticklen=vcheck(yticklen,!y.ticklen)
!y.tickname=vcheck(ytickname,!y.tickname)
!y.ticks=vcheck(yticks, !y.ticks)
!y.tickv=vcheck(ytickv, !y.tickv)
!y.title=vcheck(ytitle,!y.title)
!y.type=vcheck(ytype, 0)
!y.gridstyle=vcheck(ygridstyle,!y.gridstyle)

;
;  create a time axis label if none given
;
if not keyword_set(xtitle) then begin
   !x.title = 'Start time: '+utc2str(sec2utc(!x.crange(0)+cds_xoffset))
endif else begin
   !x.title = xtitle
endelse

;
;  perform the plotting using the defined !x variable
;
plot,utc2sec(x),y, $
         data=vcheck(data),$
         device=vcheck(device), $
         nodata=vcheck(nodata), $
         normal=vcheck(normal), $
         symsize=vcheck(symsize,1.0),$
         ynozero=vcheck(ynozero)

;
;save new system variables needed for overplotting
;
xcrange = !x.crange
xs      = !x.s
xwindow = !x.window
xregion = !x.region

ycrange = !y.crange
ys      = !y.s
ywindow = !y.window
yregion = !y.region
ytype   = !y.type

zcrange = !z.crange
zs      = !z.s
zwindow = !z.window
zregion = !z.region
ztype   = !z.type

pmulti  = !p.multi
pclip   = !p.clip

;
;restore original system variables
;
!x=xsave
!p=psave
!y=ysave

;
;replace system variables needed for overplotting
;
!x.crange = xcrange
!x.s      = xs
!x.window = xwindow
!x.region = xregion

!y.crange = ycrange
!y.s      = ys
!y.window = ywindow
!y.region = yregion
!y.type   = ytype


!p.multi  = pmulti
!p.clip   = pclip


return
end

