;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Document name: which.pro
; Created by:    Liyun Wang, GSFC/ARC, September 20, 1994
;
; Last Modified: Thu Jan 26 16:59:44 1995 (lwang@achilles.nascom.nasa.gov)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO WHICH, name, all=all, search=search
;+
; PROJECT:
;       SOHO - CDS
;
; NAME:
;       WHICH
;
; PURPOSE:
;       Search for and print file or routine in IDL !path
;
; EXPLANATION:
;       Use to find where IDL finds a program file, and, if
;	multiple definitions exist, which is loaded (the first
;	one). Splits path into all different directories,
;	searches for a file with the given NAME + '.PRO'.
;
; CALLING SEQUENCE:
;       WHICH, NAME
;
; INPUTS:
;       NAME - Name of the routine to search for (string scalar).
;
; OPTIONAL INPUTS:
;       None.
;
; OUTPUTS:
;       None.
;
; OPTIONAL OUTPUTS:
;       None.
;
; KEYWORD PARAMETERS:
;       ALL    - Report all occurrences if set. Usually WHICH reports the
;                first occurrence of a found routine (which is to be
;                executed by IDL). WHICH can take a while, especially on
;                VMS system, to search through all directories (and text
;                libraries on VMS system) if ALL is set. ALL is
;                automatically set if SEARCH is set.
;       SEARCH - Turn on the search mode, if set, that would match any
;                given string pattern found in the path.
;
; CALLS:
;       CONCAT_DIR, STR_CHOP, GREP
;
; COMMON BLOCKS:
;       WHICH -- Mainly for speeding things up
;
; RESTRICTIONS:
;       None.
;
; SIDE EFFECTS:
;       None.
;
; CATEGORY:
;       General utility
;
; PREVIOUS HISTORY:
;       Written Stein Vidar Haugan, 1993
;
; MODIFICATION HISTORY:
;       19 May, 1994, SVHH, Doc. added
;	21 May, 1994, SVHH, Version 2, with on_error,2 and 'Use:'
;       Liyun Wang, GSFC/ARC, September 20, 1994
;          Added IDL internal routine checkings.
;       Liyun Wang, GSFC/ARC, October 5, 1994
;          Current directory also gets searched now
;       Version 3, Liyun Wang, GSFC/ARC, December 16, 1994
;          Made it capable of finding files in text libraries on VMS system
;          Added the ALL keyword
;       Version 4, Liyun Wang, GSFC/ARC, January 23, 1995
;          Added the SEARCH keyword
;       Version 5, Liyun Wang, GSFC/ARC, January 24, 1995
;          Used an undocumented IDL function routine ROUTINE_NAMES to get IDL
;             intrinsic routine names.
;
; VERSION:
;       Version 5, January 24, 1995
;-

   COMMON which, internal
   ON_ERROR,2

;----------------------------------------------------------------------
;  A list of IDL internal routine can be obtained by entering the "?"
;  command at IDL prompt, provided that the current device is set to
;  'tek'.
;----------------------------------------------------------------------

   IF N_ELEMENTS(name) EQ 0 THEN BEGIN
      PRINT, 'WHICH -- Syntax error.'
      PRINT, '   Usage: WHICH, ''filename'' [,/ALL]'
      PRINT, ' '
      RETURN
   ENDIF

;----------------------------------------------------------------------
;  In case a file name with an extension is entered, strip off the
;  extension:
;----------------------------------------------------------------------
   ipos =  STRPOS(name,'.')
   IF (ipos NE -1) THEN name = STRMID(name,0,ipos)
   cd,current = cur_dir
   path_sep = ':'
   IF !version.os EQ 'vms' THEN path_sep = ','

;----------------------------------------------------------------------
;  First check to see if it is an IDL internal routine. We will call an
;  undocumented IDL function called ROUTINE_NAMES to get all names of
;  the IDL built-in function and procedure names.
;----------------------------------------------------------------------
   IF N_ELEMENTS(internal) EQ 0 THEN BEGIN
      f_name = ROUTINE_NAMES(/s_functions)
      p_name = ROUTINE_NAMES(/s_procedures)
      internal = [f_name, p_name]
   ENDIF
   IF KEYWORD_SET(search) THEN BEGIN
      exact = 0
      all = 1
      ff = grep(name, internal)
      IF ff(0) NE '' THEN BEGIN
         temp = str_comb(ff,', ')
         PRINT, 'IDL Built-in: ',temp
      ENDIF
   ENDIF ELSE BEGIN
      exact = 1
      aa = WHERE(STRUPCASE(name) EQ internal, count)
      IF count NE 0 THEN BEGIN
         PRINT, STRUPCASE(name)+' is an IDL built-in routine.'
         IF N_ELEMENTS(all) EQ 0 THEN RETURN
      ENDIF
   ENDELSE

   p = cur_dir+path_sep+!path
   dirs = str_sep(p,path_sep)

   CASE (!version.os) OF
      'vms': BEGIN
         FOR i = 0,N_ELEMENTS(dirs)-1 DO BEGIN
            allfiles = str_chop(STRUPCASE(get_mod(dirs(i))),'.PRO')
            ff = grep(name, allfiles, exact=exact)
            IF (ff(0) NE '') THEN BEGIN
               FOR j = 0, N_ELEMENTS(ff)-1 DO BEGIN
                  filename = concat_dir(dirs(i),ff(j))
                  PRINT, filename+'.PRO'
               ENDFOR
               IF N_ELEMENTS(all) EQ 0 THEN GOTO, cleanup
            ENDIF
         ENDFOR
cleanup:
;----------------------------------------------------------------------
;        Clean up scratch files when library is listed
;----------------------------------------------------------------------
         look = findfile('sys$login:*._sdac_*;*',count=nf)
         IF nf GT 0 THEN BEGIN
            FOR i=0,nf-1 DO BEGIN
               spawn,'DELETE/NOLOG/NOCONFIRM '+look(i)
            ENDFOR
         ENDIF
      END
      ELSE: BEGIN
         IF KEYWORD_SET(search) THEN BEGIN
            FOR i = 0, N_ELEMENTS(dirs)-1 DO BEGIN
               filename = findfile(concat_dir(dirs(i),'*'+name+'*.pro'),$
                           count = cnt)
               IF cnt GT 0 THEN BEGIN
                  FOR j = 0, cnt-1 DO PRINT, filename(j)
               ENDIF
            ENDFOR
         ENDIF ELSE BEGIN
            FOR i = 0,N_ELEMENTS(dirs)-1 DO BEGIN
               filename = concat_dir(dirs(i),name+'.pro')
               OPENR, unit, filename, /GET_LUN, error=rr
               IF rr EQ 0 THEN BEGIN
                  PRINT, filename
                  CLOSE, unit
                  FREE_LUN, unit
                  IF NOT KEYWORD_SET(all) THEN RETURN
               ENDIF
            ENDFOR
         ENDELSE
      END
   ENDCASE
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; End of 'which.pro'.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
