;+
; Project     : SOHO - CDS     
;                   
; Name        : CDS_SLINE()
;               
; Purpose     : Select from basic CDS Kelly or BB spectral line list
;               
; Explanation : Restores the basic parameters (wavelength and line ID) from
;               the SAVE file created by cds_sline_data and filters the list
;               according to user requirements.
;
;            eg IDL> list = cds_sline(345,350,elem=['ca','ne'],ion=['v','vi'])
;
;               will select all lines in the range 345-350 A originating
;               from Ca V, Ca VI, Ne V or Ne VI
;               
; Use         : list = cds_sline([wave_start, wave_end, elem=elem, ion=ion,$
;                                 /hard, /see, /cds, /kelly])
;    
; Inputs      : None
;               
; Opt. Inputs : wave_start  - start wavelength for search, default = 153 A
;               wave_end    - end wavelength for search, default = 789 A
;               
; Outputs     : Function returns details of chosen lines in (3,n) array.
;               Also get listing to screen and optional hard copy. 
;               
; Opt. Outputs: listing to printer
;               
; Keywords    : ELEM  - the element of interest (2 character std abbreviation)
;                       can be an array.
;               ION   - ionization stage  (up to 4 character Roman style)
;                       can be an array.
;               HARD  - if set the listing will be sent to the printer
;               SEE   - if set, listing is output to screen
;               CDS   - use line list from Blue Book
;               KELLY - use abbreviated Kelly line list (default)
;
; Calls       : None
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Util, spectrum
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 9-Nov-94
;               
; Modified    : Added output variable.  CDP, 24-Nov-94
;               Changed to a function.  CDP, 7-Mar-95
;               Added CDS BB line list option.  CDP, 15-Mar-95
;
; Version     : Version 4, 15-Mar-95
;-            

function cds_sline, w1, w2, elem=elem, ion=ion, hard=hard, $
                            see=see, kelly=kelly, cds=cds

;
;  selects lines from the list prepared by cds_sline_data and saved 
;  in  CDS_INFO/Kelly_wav_id.save or CDS_INFO/CDS_wav_id.save.
;        
if keyword_set(cds) then begin
   restore,concat_dir('$CDS_INFO','CDS_wav_id.save')
endif else begin
   restore,concat_dir('$CDS_INFO','Kelly_wav_id.save')
endelse
               
if n_params() lt 2 then w2 = 900.
if n_params() eq 0 then w1 = 100.

if not keyword_set(elem) then elem = ['ALL','ALL']
if not keyword_set(ion) then ion = ['ALL','ALL']

;
; limit wavelength
;
n = where (line(0,*) ge w1 and line(0,*) le w2)

;
;  if some valid wavelengths move on to elements selection
;
if n(0) ge 0 then begin

   line = line(*,n)
   if elem(0) ne 'ALL' then begin
      if n_elements(elem) eq 1 then telem = [elem,elem] else telem = elem
      n = find_common(strtrim(strupcase(telem),2), $
                      strtrim(strupcase(reform(line(1,*))),2))
      if n(0) eq -1 then begin
         return, -1
      endif
      line = line(*,n)
   endif
;
;  still ok so select ionization stage
;
   if ion(0) ne 'ALL' then begin
      if n_elements(ion) eq 1 then tion = [ion,ion] else tion = ion
      n = find_common(strupcase(tion), strupcase(reform(line(2,*))))
      if n(0) eq -1 then begin
         return, -1
      endif
      line = line(*,n)
   endif

;
;  print the selection
;
   if keyword_set(see) then begin
      openw,screen, filepath(/terminal), /more, /get_lun
      ns = strtrim(string(n_elements(n),form='(i6)'),2)
      printf,screen,' '
      printf,screen,ns+' lines selected.'
      printf,screen,' '
      for i=0,(n_elements(line)/3)-1 do begin
         printf,screen,line(*,i)
      endfor
   endif

;
;  hard copy requested?
;
   if keyword_set(hard) then begin
      openw,lun,'cds_sline_list',/get_lun
      for i=0,(n_elements(line)/3)-1 do begin
         printf,lun,line(*,i)
      endfor
      free_lun,lun
      spawn,'lpr cds_sline_list'
      bell
      print,'Listing sent to printer.'
   endif

;
;  nothing in this wavelength range so return empty handed
;
endif else begin
   line = -1
endelse

;
;  switch details around for hook up to ANYNAME2STD etc.
;
if n_elements(line) gt 1 then begin
   temp = line(0,*)
   line(0,*) = strpad(line(1,*),2,/after)
   line(1,*) = strpad(line(2,*),5,/after)
   line(2,*) = temp
endif

return,line
end
