;+
; Project     : SOHO - CDS     
;                   
; Name        : CDS_SLINE_DATA
;               
; Purpose     : Create save file from Kelly or CDS spectral line list.
;               
; Explanation : The CDS version of the Kelly line list or the Blue Book CDS
;               line list is read and the wavelengths and lines 
;               identifications are SAVEd for pickup by cds_sline.
;               
; Use         : IDL> cds_sline_data [, /kelly, /cds]
;    
; Inputs      : None
;               
; Opt. Inputs : None
;               
; Outputs     : File is created in CDS_INFO
;               
; Opt. Outputs: None
;               
; Keywords    : kelly - uses Kelly line list (default)
;               cds   - uses CDS Blue Book line list
;
; Calls       : None
;
; Common      : None
;               
; Restrictions: Must have write access to CDS_INFO directory.
;               
; Side effects: None
;               
; Category    : Util, spectrum
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 9-Nov-94
;               
; Modified    : To include BB line list.  CDP, 15-Mar-95
;               Add /xdr key for VMS compatibility.  CDP, 3-Apr-95
;
; Version     : Version 3, 3-Apr-95
;-            

pro cds_sline_data, kelly=kelly, cds=cds

;
;  Pick out lines from CDS Kelly list or BB list
;
text = ' '
roman=['?','I','II','III','IV','V','VI','VII','VIII','IX','X',$
       'XI','XII','XIII','XIV','XV','XVI','XVII','XVIII','XIX','XX',$
       'XXI','XXII','XXII','XXIII','XXIV','XXV','XXVI','XXVI','XXVIII','XXIX']
line = strarr(3,10000)

;
;  which list
;
if keyword_set(cds) then begin
   openr,lun,concat_dir('$CDS_ATOMIC','cds_lines.dat'),/get_lun
endif else begin
   openr,lun,concat_dir('$CDS_ATOMIC','kelly_lines.dat'),/get_lun
endelse
num = 0

while not eof(lun) do begin
   readf,lun,text
   if keyword_set(cds) then begin
      txt = rem_mult_char(text,' ')
      txt = str_sep(txt,' ')
      e = txt(0)
      i = where(roman eq txt(1))
      w = txt(2)
   endif else begin   
      w = strmid(text,3,7)
      e = strmid(text,11,2)
      i = strmid(text,13,2)
      if i eq '' or i eq ' ' or i eq '  ' then i = '0'
      e = strlowcase(e)
      e = strupcase(strmid(e,0,1))+strmid(e,1,10)
   endelse

   line(0,num) = w
   line(1,num) = e
   line(2,num) = roman(fix(i))

   num = num + 1
endwhile

free_lun,lun

;
;  Trim list
;
line = line(*,0:num)

;
;  and sort by wavelength
;
w = line(0,*)
ns = sort(w)
line = line(*,ns)

;
;  and save
;
if keyword_set(cds) then begin
   save,/xdr,line,file=concat_dir('$CDS_INFO','CDS_wav_id.save')
endif else begin
   save,/xdr,line,file=concat_dir('$CDS_INFO','Kelly_wav_id.save')
endelse
end
