;----------------------------------------------------------------------
; Document name: mk_gif.pro
; Created by:    Liyun Wang, GSFC/ARC, December 7, 1994
;
; Last Modified: Thu Feb  9 09:52:18 1995 (lwang@achilles.nascom.nasa.gov)
;----------------------------------------------------------------------
;
PRO MK_GIF, ifile, red=red, green=green, blue=blue, frac=frac, $
            reverse=reverse, sig=sig, _extra=extra_keywords
;+
; PROJECT:
;       SOHO - CDS
;
; NAME:
;       MK_GIF
;
; PURPOSE:
;       Convert FITS files to GIF image files
;
; EXPLANATION:
;       Reads one or more FITS files, byte scales them, and then
;       writes them to GIF files. If a title is to be plotted, it
;       will be plotted in upper center position.
;
; CALLING SEQUENCE:
;       MK_GIF, FILE_STC
;
; INPUTS:
;       FILE_STC - A structure with three tags:
;          FILENAME -- Name of the FITS file
;          TITLE    -- Title to be attached in the GIF image
;          COLOR    -- Color table to be loaded; defaults to 0
;          FLIP     -- Flag indicating if the image should be flipped
;                      before conversion.
;
; OPTIONAL INPUTS:
;       None.
;
; OUTPUTS:
;       None.
;
; OPTIONAL OUTPUTS:
;       None.
;
; KEYWORD PARAMETERS:
;       RED, GREEN, BLUE - optional color vectors to override COLOR
;       FRAC    - fraction by which to increase image
;                 size in Y-direction to fit title [def = 10%]
;       ROTATE  - value for rotate (see ROTATE function)
;       FLIP    - flip image to to bottom
;       REVERSE - flip image left to right
;       SIG     - select significant range of image
;
; CALLS:
;       WRITE_GIF, CONCAT_DIR, BREAK_FILE, FXREAD
;
; COMMON BLOCKS:
;       None.
;
; RESTRICTIONS:
;       None.
;
; SIDE EFFECTS:
;       None.
;
; CATEGORY:
;
; PREVIOUS HISTORY:
;       Written December 7, 1994, Liyun Wang, GSFC/ARC
;
; MODIFICATION HISTORY:
;       Version 1, Liyun Wang, GSFC/ARC, December 7, 1994
;          Modified from FITS2GIF by Dominic Zarro (ARC)
;       Version 2, Liyun Wang, GSFC/ARC, February 1, 1995
;          Added one tag (FLIP) to the input file structure.
;
; VERSION:
;       Version 2, February 1, 1995
;-
;
   ON_ERROR, 2

   IF N_ELEMENTS(ifile.filename) EQ 0 OR datatype(ifile) NE 'STC' THEN $
      MESSAGE,'Syntax --> mk_gif, file_struct'

   sav_white=!d.n_colors-1
   xpos = .5
   FOR i = 0, N_ELEMENTS(ifile.filename)-1 DO BEGIN
      break_file, ifile(i).filename, dsk, direc, file, ext
      ofile = concat_dir(direc,file+'.gif')
      fxread,ifile(i).filename,data,header
      image = data
      IF KEYWORD_SET(sig) THEN image = sigrange(temporary(image))
      image = bytscl(temporary(image))
      IF KEYWORD_SET(reverse) THEN image = reverse(temporary(image))
      IF ifile(i).flip EQ 1 THEN image = reverse(rotate(temporary(image),2))
      sav_dev = !d.name
;----------------------------------------------------------------------
;     use Z-buffer to adjoin title
;----------------------------------------------------------------------
      sz = SIZE(image)
      IF ifile(i).title(0) NE '' THEN BEGIN
         IF N_ELEMENTS(ifile(i).title) EQ 2 THEN BEGIN
            ypos = .96
         ENDIF ELSE BEGIN
            ypos = .95
         ENDELSE
         IF N_ELEMENTS(frac) EQ 0 THEN frac = 10.
         fac = (1.+frac/100.)
         new = bytarr(sz(1),sz(2)*fac)
         new(0:sz(1)-1,0:sz(2)-1) = temporary(image)
         image = new & delvarx,new
         set_plot,'z'
         DEVICE,set_resolution=[sz(1),sz(2)*fac]
         tvscl, image
;----------------------------------------------------------------------
;        Plot the centered title
;----------------------------------------------------------------------
         xyouts, xpos, ypos, ifile(i).title(0), norm=1, SIZE=1.0, $
            charthick=1.0, font=-1, alignment = 0.5,$
            _extra=extra_keywords,color=sav_white
         IF N_ELEMENTS(ifile(i).title) EQ 2 THEN BEGIN
            xyouts, xpos, ypos-0.03, ifile(i).title(1), norm=1, SIZE=1.0, $
               charthick=1.0, font=-1, alignment = 0.5,$
               _extra=extra_keywords,color=sav_white
         ENDIF
         image = tvrd()
         set_plot,sav_dev
      ENDIF
;----------------------------------------------------------------------
;     use X-windows mode to load color table
;----------------------------------------------------------------------
;      set_plot,'x'
      loadct, ifile(i).color,/silent
;----------------------------------------------------------------------
;     Override color table if red, green, and blue are defined
;----------------------------------------------------------------------
      IF N_ELEMENTS(red)*N_ELEMENTS(green)*N_ELEMENTS(blue) EQ 0 THEN $
         cload = 0 ELSE cload = 1
      IF cload THEN write_gif, ofile, image, red, green, blue  $
      ELSE write_gif, ofile, image
      PRINT, 'File '+ofile+' created.'
      set_plot, sav_dev
   ENDFOR
   RETURN
END

;----------------------------------------------------------------------
; End of 'mk_gif.pro'.
;----------------------------------------------------------------------
