;+
; Project     : SOHO - CDS     
;                   
; Name        : TLM_SUMMARY
;               
; Purpose     : To summarize a telemetry file.
;               
; Explanation : Reads a telemetry files, packet by packet and records 
;               the details of every packet whose ID matches that requested.
;               
; Use         : IDL> tlm_summary, filename, id_spec [,packs=packs,packe=packe]
;    
; Inputs      : filename - the telemetry file name
;               id_spec  - integer array giving ID wanted eg [1,2] to pick
;                          GIS/NIS raster header packets 
;                          (watch decimal/hex switch!)
;               
; Opt. Inputs : None
;               
; Outputs     : Outputs information to screen and file 'filename'_summ in 
;               current directory. 
;               
; Opt. Outputs: None
;               
; Keywords    : packs  -  the packet at which to start logging
;               packe  -  the packet at which to stop logging
;
; Calls       : None
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Telemetry
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 23-May-1994
;               
; Modified    : Make it write summary file in current directory and change
;               extension added to be _summ.  CDP, 15-Jun-94
;
; Version     : Version 2, 15-Jun-94
;-            

pro tlm_summary, file, packet_ids, packs=packs, packe=packe

;
;  insist on correct input
;
if n_params() lt 2 then begin
   print,'Use:  tlm_summary, filename, packet_ids
   return
endif

;
;  allow user to have own version of telemetry files
;
tfile = ''
if  file_exist(file) then begin
   tfile = file
endif else begin
   if file_exist(concat_dir('$CDS_TM_DATA',file)) then begin
      tfile = concat_dir('$CDS_TM_DATA',file)
   endif
endelse

;
;  telemetry file doesn't exist in user or std directory
;
if tfile eq '' then begin
   print,'TLM_SUMMARY:  Input file does not exist.'
   return
endif

;
;  Define general packet header structure
;
PacketHeader = {PacketHeaderDef,                $
                Msgid:               FIX(0),    $
		MsgLen:              FIX(0),    $
		MsgReqid:            FIX(0),    $    
		ap_id:               BYTARR(2), $
		counter:             FIX(0),    $
		length:              FIX(0),    $
		Time:                BYTARR(6), $
		ID:                  BYTE(0),   $
		StreamPacketCounter: BYTE(0)}

;
; Define Science data header
;
SciHeader = {SciHeaderDef, $
		StudyID:               BYTE(0),   $
		DataCompScheme:        BYTE(0),   $ 
		DataExtWindowID:       FIX(0),    $
		SequenceID:            FIX(0),    $
		RasterLengthM:         BYTE(0),   $
		RasterLengthS:         BYTE(0),   $
		OPSLCoordinate:        FIX(0),    $
		OPSRCoordinate:        FIX(0),    $
                SlitPosition:          FIX(0),    $
		SlitNumber:            BYTE(0),   $
		MirrorPosition:        BYTE(0),   $
		ExposureTime:          FIX(0),    $
		StudyCounter:          FIX(0),    $
		RasterCounter:         FIX(0),    $
                DataCompOption:        FIX(0),    $
                SciProcStatus:         FIX(0)}

;
;Exposure header
;
;
Exphdr= {ExpHeaderDef, $
                count:                 FIX(0),      $
		SolarX:                FIX(0),      $
		ExposureStart:         BYTARR(6),   $
		ExposureNumber:        FIX(0),      $
		OPSLCoordinate:        FIX(0),      $
		OPSRCoordinate:        FIX(0),      $
		SlitPosition:          FIX(0),      $
                SunSensorPitch:        FIX(0),      $
                SunSensorYaw:          FIX(0),      $
                SolarY:                FIX(0),      $
                MirrorPosition:        BYTE(0),     $
                EventStatus:           BYTE(0),     $
                EventLevel:            FIX(0),      $
		StudyCounter:          FIX(0),      $
		RasterCounter:         FIX(0),      $
                Science:               BYTARR(252), $
		ChkSum:                FIX(0)}

;
;  dummy data arrays
;
b260 = bytarr(260)
b286 = bytarr(286)
b284 = bytarr(284)
;
;  accumulative totals for info
;
tot = intarr(50)
sp_tot = intarr(50)


;
;  open input file 
;
openr, unit, tfile, /get_lun

;
;  open output file in the current directory
;
break_file,tfile,disk,direct,ff,ext
openw, lun, ff+ext+'_summ',/get_lun,width=130

;
;  human readable interpretation of ID
;
name = strarr(100)
name(1) = 'GIS-raster'
name(2) = 'VDS-raster'
name(3) = 'Abort'
name(17) = 'GIS-exp'
name(18) = 'VDS-exp'
name(33) = 'GIS-data'
name(34) = 'VDS-data'
name(48) = 'Special'

;
;  and of special packet formats
;
sp_name = strarr(100)
sp_name(1)  = '(MCU)'
sp_name(2)  = '(Deferred)'
sp_name(3)  = '(Data line list)'
sp_name(4)  = '(Raster table)'
sp_name(5)  = '(Sequence table)'
sp_name(6)  = '(CDS h/k table)'
sp_name(7)  = '(Error)'
sp_name(8)  = '(Health)'
sp_name(9)  = '(IMIF)'
sp_name(10)  = '(Hardware list)'
sp_name(11)  = '(Mode table)'
sp_name(12)  = '(Engineering table)'
sp_name(13)  = '(Sequence controller table)'
sp_name(14) = '(IEF)'

;
;  title
;
print,' Packet      Type             Time            Seq-id   Raster-count  '+$
      'M-step S-step   Exp #   Slit Posn    Mirror Posn'
printf,lun,' Packet      Type              Time           Seq-id   Raster-count  '+$
           'M-step S-step   Exp #   Slit Posn    Mirror Posn'

;
;  any packets to be skipped?
;
if not keyword_set(packs) then packs = 0L
if not keyword_set(packe) then packe = 1000000L

counter = packs

;
;  skip to start place
;
if packs gt 0 then point_lun, unit, long(packs)*306L

while not eof(unit) and (counter le packe) do begin
   readu, unit, PacketHeader

;
;  produce totals
;
if packetheader.id lt 50 then tot(packetheader.id) = tot(packetheader.id)+1
if packetheader.id eq '30'x then begin
   temp = read_sp_id(unit) 
   if (temp(1) lt 50) and (temp(1) gt 0) then begin
      sp_tot(temp(1)) = sp_tot(temp(1))+1
   endif else begin
      print,counter,'Special packet format ID out of range. Value = ',temp(1),$
                     form='(1x,i5,4x,a,i4)'
      printf,lun,counter,$
                    'Special packet format ID out of range. Value = ',temp(1),$
                     form='(1x,i5,4x,a,i4)'
   endelse
   readu,unit,b284
endif

;
;  was ID wanted?
;
   hit = where(packet_ids eq PacketHeader.id)
   if hit(0) ge 0 then begin
      case packetheader.id of
         '1'x: begin
                  readu,unit,sciheader
                  readu,unit,b260
               end
         '2'x: begin
                  readu,unit,sciheader
                  readu,unit,b260
               end
        '11'x: begin
                  readu,unit,exphdr
               end
        '12'x: begin
                  readu,unit,exphdr
               end
        '30'x:
         else: readu,unit,b286
      endcase
         
      if packetheader.id eq '1'x or packetheader.id eq '2'x then begin
          time = anytim2cal(tai2utc(obt2tai(packetheader.time)),form=1)
          print,counter, name(packetheader.id), time, $
                         byteswap(sciheader.sequenceid),$
                         byteswap(sciheader.rastercounter),$
                         sciheader.rasterlengths,$
                         sciheader.rasterlengthm,$
                         form='(1x,i5,4x,a10,2x,a20,5x,z4.4,5x,i6,4x,i6,1x,i6)'
          printf,lun,counter, name(packetheader.id), time, $
                         byteswap(sciheader.sequenceid),$
                         byteswap(sciheader.rastercounter),$
                         sciheader.rasterlengths,$
                         sciheader.rasterlengthm,$
                         form='(1x,i5,4x,a10,2x,a20,5x,z4.4,5x,i6,4x,i6,1x,i6)'
      endif

      if packetheader.id eq '11'x or packetheader.id eq '12'x then begin
          time = anytim2cal(tai2utc(obt2tai(exphdr.exposurestart)),form=1)
          slitpos_decode,exphdr.slitposition,sb,sp
          print,counter, name(packetheader.id), time, $
                         byteswap(exphdr.exposurenumber),$
                         sp,$
                         exphdr.mirrorposition,$
                         format='(1x,i5,4x,a10,2x,a20,36x,i10,7x,i4,9x,i4)'
          printf,lun,counter, name(packetheader.id), time, $
                         byteswap(exphdr.exposurenumber),$
                         sp,$
                         exphdr.mirrorposition,$
                         format='(1x,i5,4x,a10,2x,a20,36x,i10,7x,i4,9x,i4)'
      endif

      if packetheader.id eq '21'x or packetheader.id eq '22'x then begin
          time = anytim2cal(tai2utc(obt2tai(packetheader.time)),form=1)
          print,counter, name(packetheader.id),time,form='(1x,i5,4x,a10,2x,a20)'
          printf,lun,counter, $
                     name(packetheader.id),time,form='(1x,i5,4x,a10,2x,a20)'
      endif

      if packetheader.id eq '30'x then begin
          sp_id = temp(1) 
          time = anytim2cal(tai2utc(obt2tai(packetheader.time)),form=1)
          if (sp_id lt 50) and (sp_id ge 0) then begin
             print,counter, name(packetheader.id),$
                       time,sp_name(sp_id),form='(1x,i5,4x,a10,2x,a20,2x,a30)'
             printf,lun,counter, name(packetheader.id),$
                       time,sp_name(sp_id),form='(1x,i5,4x,a10,2x,a20,2x,a30)'
          endif else begin
             print,counter, name(packetheader.id),$
               time,'(Incorrect format ID)',form='(1x,i5,4x,a10,2x,a20,2x,a30)'
             printf,lun,counter, name(packetheader.id),$
               time,'(Incorrect format ID)',form='(1x,i5,4x,a10,2x,a20,2x,a30)'
          endelse
      endif

   endif else begin
      if packetheader.id ne '30'x then readu,unit,b286
   endelse

;
;  next packet
;
   counter = counter + 1

endwhile

;
;  print statistics summary
;

pname = strarr(50)
pname(*) = '          ?           '
pname(10) = 'Engineering A'
pname(11) = 'Engineering B'
pname(1)  = 'GIS Raster header'
pname(2)  = 'VDS Raster header'
pname(3)  = 'Raster abort'
pname(17)  = 'GIS Exposure header'
pname(18)  = 'VDS Exposure header'
pname(33)  = 'GIS Science data'
pname(34)  = 'VDS Science data'
pname(4)  = 'Primary Proc. Mem. Dump'
pname(5)  = 'Secondary Proc. Mem. Dump'
pname(6)  = 'VDS Memory Dump'
pname(7)  = 'GIS raw data dump'
pname(48)  = 'Special Packet'

spname = strarr(50)
spname(*) = '   ?   '
spname(2) = 'Deferred'
spname(8) = 'Health'
spname(14) = 'IEF'
spname(1) = 'MCU'
spname(9) = 'IMIF'
spname(7) = 'Error'





print,' '
print,'   Packet statistics.'
print,'   -----------------'
print,' '
print,'For packets ',fmt_vect([packs,counter-1],delim=' to ')
print,' '
print,'Packet ID               Type                Number of packets'
print,'  (hex)
for i=0,49 do begin
   if tot(i) gt 0 then print,i,pname(i),tot(i),form='(3x,z2.2,7x,a25,15x,i5)
endfor
print,' '
if total(sp_tot) gt 0 then begin
   print,'Specials....'
   for i=0,49 do begin
      if sp_tot(i) gt 0 then print,i,spname(i),sp_tot(i),$
                                                 form='(3x,z2.2,7x,a25,15x,i5)
   endfor
endif

;
;  hard copy
;

printf,lun,' '
printf,lun,'   Packet statistics.'
printf,lun,'   -----------------'
printf,lun,' '
printf,lun,'For packets ',fmt_vect([packs,counter-1],delim=' to ')
printf,lun,' '
printf,lun,'Packet ID               Type                Number of packets'
printf,lun,'  (hex)
for i=0,49 do begin
   if tot(i) gt 0 then begin
      printf,lun,i,pname(i),tot(i),form='(3x,z2.2,7x,a25,15x,i5)
   endif
endfor
printf,lun,' '
if total(sp_tot) gt 0 then begin
   printf,lun,'Specials....'
   for i=0,49 do begin
      if sp_tot(i) gt 0 then printf,lun,i,spname(i),sp_tot(i),$
                                                 form='(3x,z2.2,7x,a25,15x,i5)
   endfor
endif


  
;
;  close files
;
free_lun, unit
free_lun, lun

end
   
