;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Document name: add_path.pro
; Created by:    Liyun Wang, GSFC/ARC, October 8, 1994
;
; Last Modified: Tue Nov 15 21:28:56 1994 (lwang@orpheus.nascom.nasa.gov)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO ADD_PATH, path_name, index=index, append=append, expand=expand
;+
; PROJECT:
;       SOHO - CDS
;
; NAME:
;       ADD_PATH
;
; PURPOSE:
;       Add directory (and optionally all its subdirs) into IDL path
;
; EXPLANATION:
;
; CALLING SEQUENCE:
;       ADD_PATH, path_name [,/append] [,index=index]
;
; INPUTS:
;       path_name -- A string scalar of a valid directory name.
;
; OPTIONAL INPUTS:
;       INDEX -- Position in the !path where the directory name is inserted;
;                ignored if the keyword APPEND is set.
;
; OUTPUTS:
;       None, but !path is changed.
;
; OPTIONAL OUTPUTS:
;       None.
;
; KEYWORD PARAMETERS:
;       APPEND -- By default, the given directory name is added in the
;                 beginning of !path. Set this keyword will add the directory
;                 name in the end of !path.
;       EXPAND -- Set this keyword if the PATH_NAME needs to be expanded.
;
; CALLS:
;       CHK_DIR, CONCAT_DIR, STR_SEP, STR_COMB, DATATYPE
;
; COMMON BLOCKS:
;       None.
;
; RESTRICTIONS:
;       None.
;
; SIDE EFFECTS:
;       None.
;
; CATEGORY:
;       Utilities, OS
;
; PREVIOUS HISTORY:
;       Written October 8, 1994, by Liyun Wang, GSFC/ARC
;
; MODIFICATION HISTORY:
;       Version 2, Liyun Wang, GSFC/ARC, October 17, 1994
;          Added EXPAND keyword
;
; VERSION:
;       Version 2, October 17, 1994
;-
;
   ON_ERROR, 2
   IF N_ELEMENTS(path_name) EQ 0 THEN $
      MESSAGE, 'Syntax: ADD_PATH, dir_name [,/append]'

   IF datatype(path_name) NE 'STR' THEN $
      MESSAGE, 'Input parameter must be of string type.'

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  Check the validity of the path name
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
   IF NOT chk_dir(path_name) THEN BEGIN
      PRINT, 'Sorry, but '+path_name+' is NOT a valid direcotry name, '+$
         'and cannot be'
      PRINT, 'added into the IDL path.'
      RETURN
   ENDIF

   IF N_ELEMENTS(index) EQ 0 THEN index = 0

   CASE (!version.os) OF
      'vms': delimit = ','
      'windows': delimit = ';'
      ELSE: BEGIN
         delimit = ':'
         home_dir = getenv('HOME')
         IF STRMID(path_name,0,2) EQ '~/' THEN $
            path_name = concat_dir(home_dir,STRMID(path_name,2,9999)) $
         ELSE BEGIN ; if it is a subdirectory, attatch the whole path to it
            cd,current = curr_dir
            full_path = concat_dir(curr_dir, path_name)
            IF chk_dir(full_path) THEN $
               path_name = full_path
         ENDELSE
      END
   ENDCASE

   dir_names = str_sep(!path, delimit)
   nd = N_ELEMENTS(dir_names)
   IF index GT nd-1 THEN index = nd-1

   IF KEYWORD_SET(expand) THEN BEGIN
      p_name = expand_path('+'+path_name)
      IF p_name EQ '' THEN BEGIN
         PRINT, 'Apparently there are no .pro or .sav files in '+path_name+$
            '...'
         PRINT, 'No action is taken.'
         RETURN
      ENDIF
      names = str_sep(p_name,delimit)
      FOR i = 0, N_ELEMENTS(names)-1 DO BEGIN
         id = WHERE(dir_names EQ names(i))
         IF id(0) NE -1 THEN PRINT, names(i)+' already in the IDL path.' $
         ELSE BEGIN
            IF N_ELEMENTS(idd) EQ 0 THEN idd = i ELSE idd = [idd,i]
         ENDELSE
      ENDFOR
      IF N_ELEMENTS(idd) EQ 0 THEN RETURN
      path_name = str_comb(names(idd),delimit)
   ENDIF 

   IF KEYWORD_SET(append) THEN BEGIN
      !path = !path+delimit+path_name
   ENDIF ELSE BEGIN
      IF index EQ 0 THEN $
         !path = path_name+delimit+!path $
      ELSE $
         !path = str_comb(dir_names(0:index-1),delimit)+delimit+$
         path_name+delimit+$
         str_comb(dir_names(index:nd-1),delimit)
   ENDELSE
   names = str_sep(path_name,delimit)
   n_path = N_ELEMENTS(names)-1
   FOR i = 0, n_path DO PRINT, names(n_path-i)+' added to the IDL path.'
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; End of 'add_path.pro'.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
