pro host_to_ieee, data, IDLTYPE = idltype
;+
; Project     : SOHO - CDS
;
; Name        : 
;	HOST_TO_IEEE
; Purpose     : 
;	Translate host variables to IEEE format.
; Explanation : 
;	Translates an IDL variable in the host architecture into the IEEE-754
;	standard (e.g. as used in FITS files).
;
;	BYTEORDER is used to convert from the host format to network format for
;	integer arrays or XDR format for floating point arrays.  These are
;	equivalent to IEEE format.
;
; Use         : 
;	HOST_TO_IEEE, DATA
; Inputs      : 
;	DATA	 = Any IDL variable, scalar or vector.  DATA will be modified
;		   by HOST_TO_IEEE to convert it to IEEE format.  Byte and
;		   string variables are returned by HOST_TO_IEEE unchanged.
; Opt. Inputs : 
;	None.
; Outputs     : 
;	The output value of DATA is the variable converted to the format of the
;	host architecture.
; Opt. Outputs: 
;	None.
; Keywords    : 
;	IDLTYPE - scalar integer (1-7) specifying the IDL datatype according
;		to the code given by the SIZE function.      This keyword
;		will usually be used when suppying a byte array that needs
;		to be interpreted as another data type (e.g. FLOAT).
; Calls       : 
;	None.
; Common      : 
;	None.
; Restrictions: 
;	Assumes the IDL version is since 2.2.2 when the /XDRTOF keyword 
;	became available to BYTEORDER.    There were two bad implementations
;	in BYTEORDER for double precision: (1) in IDL V3.* for DecStations
;	(!VERSION.ARCH = 'mipsel') and (2) on Dec Alpha OSF machines.
;	IEEE_TO_HOST works around these cases by swapping the byte order
;	directly.
; Side effects: 
;	None.
; Category    : 
;	Utilities, Operating_system
; Prev. Hist. : 
;	Adapted from CONV_UNIX_VAX, W. Landsman   Hughes/STX    January, 1992
;	Fixed Case statement for Float and Double      September, 1992
;	Workaround for /DTOXDR on DecStations          January, 1993
;	Workaround for /DTOXDR on Alpha OSF            July 1994
;	Assume since Version 2.2.2, Ultrix problems persist   November 1994
; Written     : 
;	Wayne Landsman, GSFC/UIT, January 1992.
; Modified    : 
;	Version 4, William Thompson, GSFC, 27 February 1995
;		Using version from IDL Astronomy User's Library for better
;		compatibility.  See previous history above.  IDLTYPE keyword
;		reinstated.
; Version     : 
;	Version 4, 27 February 1995
;-
;
 On_error,2 

 if N_params() EQ 0 then begin
    print,'Syntax - HOST_TO_IEEE, data, [IDLTYPE = ]
    return
 endif  

 npts = N_elements( data )
 if npts EQ 0 then $
     message,'ERROR - IDL data variable (first parameter) not defined'

 sz = size(data)
 if not keyword_set( idltype) then idltype = sz( sz(0)+1)

 case idltype of

      1: return                             ;byte

      2: byteorder, data, /HTONS            ;integer

      3: byteorder, data, /HTONL            ;long

      4: byteorder, data, /FTOXDR           ;float

      5: begin                              ;double
 
     bad_ultrix =   (!VERSION.ARCH EQ 'mipsel') and $
                    strmid(!VERSION.RELEASE,0,1) EQ '3'
     bad_osf  =     (!VERSION.ARCH EQ 'alpha') and (!VERSION.OS EQ 'OSF')

     if bad_ultrix or bad_osf then begin       ;Swap byte order directly

                    dtype = sz( sz(0) + 1)
                    if ( dtype EQ 5 ) then data = byte(data, 0, npts*8) $
                                      else npts = npts/8
                    data = reform( data, 8 , npts ,/OVER)
                    data = rotate( data, 5)
                    if ( dtype EQ 5 ) then data = double(data, 0, npts)
                    if sz(0) gt 0 then data = reform( data, sz(1:sz(0)), /OVER )

      endif else byteorder, data, /DTOXDR

      end
     
      6: BEGIN                              ;complex
           fdata = float(data)
           byteorder, fdata, /FTOXDR
           idata = imaginary( temporary(data) )
           byteorder, idata, /FTOXDR
           data = complex( fdata, idata )
         END

      7: return                             ;string

       8: BEGIN				    ;structure

	Ntag = N_tags( data )

	for t=0,Ntag-1 do  begin
          temp = data.(t)
          host_to_ieee, temp
          data.(t) = temp
        endfor 

       END
 ENDCASE

 return
 end 
