;+
; Project     : SOHO - CDS     
;                   
; Name        : ANYNAME2STD()
;               
; Purpose     : Converts user format of line name to CDS standard.
;               
; Explanation : Spectral lines in the CDS software have an associated 'name'.
;               This name is also allocated to the data extraction window 
;               derived from the line.  For ease of use the internal format
;               used is of the form N_5_1234_5 ie the element, the ionization
;               level and the wavelength separated by underscores.  (Note the
;               decimal point is replaced by an underscore also). Since users
;               may prefer to refer to the line by a name using another format
;               this routine is provided to try and translate between the two.
;               
; Use         : IDL> std = anyname2std(element [, ion, wave, errmsg=errmsg])   
;                               
; Inputs      : element - the two-letter element ID  eg He, N, Ne
;               ion     - the ionization stage (can be roman or arabic)
;               wave    - the wavelength in Angstroms (will be truncated to 
;                         nearest 0.01A)
;
;               Note that various combinations of the above are possible.
;               If 3 parameters are supplied they will be interpreted as above.
;
;               If only 2 are given, the first will be assumed to contain
;               both the element and the ion, separated by spaces or under-
;               scores. 
;               eg IDL> std = anyname2std('He II',301.1)
;
;               If only one (string) parameter is given, it will be assumed to
;               contain all three pieces of information.
;               eg IDL> std = anyname2std('He II 301.1')
;
;               If only one parameter is given but it is a 3-element array
;               this will be treated as if it were 3 separate parameters
;               (for compatibility with cds_sline)
;
; Opt. Inputs : As above
;               
; Outputs     : Function returns standard version of name or null string
;               if an error is detected.
;               
; Opt. Outputs: None
;               
; Keywords    : errmsg  - if this variable is defined on entry, any error
;                         messages will be returned in it.
;
;                         eg IDL> errmsg = ''
;                            IDL> std = anyname2std('He II', errmsg=errmsg)
;                            IDL> if errmsg ne '' then print,errmsg
;
;                            ----> 'Missing wavelength'
;
;
; Calls       : None
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Util, string
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 6-Mar-95
;               
; Modified    : Add 3-element array input.  CDP, 7-Mar-95
;               Rework wavelength format.  CDP, 16-Mar-95
;
; Version     : Version 3, 16-Mar-95
;-            

function anyname2std, element, ion, wave, errmsg=errmsg

;
;  Roman ionization states
;
ions = ['I','II','III','IV','V','VI','VII','VIII','IX','X','XI','XII',$
        'XIII','XIV','XV','XVI','XVII','XVIII','XIX','XX','XXI','XXII',$
        'XXIII','XXIV','XXV','XXVI']

;
;  check parameters
;
if n_params() eq 0 then begin
   print,'Use: IDL> std = anyname2std(element,ion,wave)'
   print,'see doc. for variations.'
   return,''
endif


;
;  output array
;
if (size(element))(0) eq 0 then out = strarr(1)  
if (size(element))(0) eq 1 then out = strarr(1)  
if (size(element))(0) eq 2 then out = strarr((size(element))(2))


;
;  loop to handle input arrays
;

for num = 0,n_elements(out)-1 do begin
;
; figure out what the input is
;
   case 1 of
;
;  full specification of all 3 parameters separately
;
     (n_params() eq 3): begin
                           if n_elements(element) eq n_elements(ion) and $
                             n_elements(element) eq n_elements(wave) then begin
                              e = element(num)
                              i = ion(num)
                              w = wave(num)
                           endif else begin
                              message = 'Input parameters must have same dimensions.'
                              if n_elements(errmsg) gt 0 then begin
                                 errmsg = message
                                 return,''
                              endif else begin
                                 message,message
                              endelse
                           endelse
                        end

;
;  only 2 parameters given so assume they are 'element/ion' and wavelength
;
     (n_params() eq 2): begin
                           wave = ion(num)
                           e = rem_mult_char(repchar(strcompress(strtrim(element(num),2)),' ','_'),'_')
                           nu = strpos(e,'_')
                           if nu eq -1 then begin
                              message = 'Element/ionization parameter has only one field.'
                              if n_elements(errmsg) gt 0 then begin
                                 errmsg = message
                                 return,''
                              endif else begin
                                 message,message
                              endelse
                           endif
                           z = str_sep(e,'_')
                           e = z(0) 
                           i = z(1)
                           w = wave
                        end

   
;
;  only one parameter but it is 3-element array
;
   (n_params() eq 1 and (size(element))(1) eq 3): begin
            e = element(0,num)
            i = element(1,num)
            w = element(2,num)
          end


;
;  only one parameter so assume all 3 are packaged up in a string
;
   (n_params() eq 1 and datatype(element,1) eq 'String'): begin
      e = rem_mult_char(repchar(strcompress(strtrim(element(num),2)),' ','_'),'_')
      nu = strpos(e,'_')
      if nu eq -1 then begin
         message = 'A single parameter input must have 3 fields.'
         if n_elements(errmsg) gt 0 then begin
            errmsg = message
            return,''
         endif else begin
            message,message
         endelse
      endif
      z = str_sep(e,'_')
      e = z(0)
      i = z(1)
      w = z(2)
   end
   
;
;  don't understand input
;
     else:  begin
               message = 'Cannot interpret input parameters'
               if n_elements(errmsg) gt 0 then begin
                  errmsg = message
                  return,''
               endif else begin
                  message,message
               endelse
            end
   endcase


;
;  now validate the 3 parameters in turn
;
;
;   first the element
;
   e = strupcase(strtrim(e,2))
   if strlen(e) gt 2 then begin
      message = 'Element has > 2 characters'
      if n_elements(errmsg) gt 0 then begin
         errmsg = message
         return,''
      endif else begin
         message,message
      endelse
   endif
   
   if strlen(e) gt 1 then begin
      e = strmid(e,0,1) + strlowcase(strmid(e,1,1))
   endif
   
;
;  then the ionization state
;
   if i ne '?' then begin
      if valid_num(i,/integer) then begin
         i = strtrim(string(i,form='(i2)'),2)
      endif else begin
         n = where(ions eq strupcase(i))
         if n(0) ge 0 then begin
            i = strtrim(string(n(0)+1,form='(i2)'),2)
         endif else begin
            message = 'Cannot interpret ionization state'
            if n_elements(errmsg) gt 0 then begin
               errmsg = message
               return,''
            endif else begin
               message,message
            endelse
         endelse
      endelse
   endif

;
;  and finally the wavelength
;
   if valid_num(w,value) then begin
      if value lt 100.0 or value gt 1700.0 then begin
         message = 'Wavelength is outside CDS/SUMER range'
         if n_elements(errmsg) gt 0 then begin
            errmsg = message
            return,''
         endif else begin
            message,message
         endelse
      endif else begin
         w = strtrim(string(value,form='(f7.2)'),2)
      endelse
   endif else begin
      message = 'Wavelength parameter is not a valid number'
      if n_elements(errmsg) gt 0 then begin
         errmsg = message
         return,''
      endif else begin
         message,message
      endelse
   endelse
 
;
;  success if reached this far and replace decimal point by underscore
;
   out(num) = repchar(e+'_'+i+'_'+w,'.','_')
endfor

;
;  return final list in std format 
;
if n_elements(out) eq 1 then out = out(0)
return,out

end
