;+
; Project     : SOHO - CDS     
;                   
; Name        : STDNAME2USER()
;               
; Purpose     : Converts CDS standard line names to user format.
;               
; Explanation : Spectral lines in the CDS software have an associated 'name'.
;               This name is also allocated to the data extraction window 
;               derived from the line.  For ease of use the internal format
;               used is of the form N_5_1234.5 ie the element, the ionization
;               level and the wavelength separated by underscores.  Since users
;               may prefer to refer to the line by a name using another format
;               this routine is provided to try and translate between the two.
;               
; Use         : IDL> uname = stdname2user(std_name[,form=form,$
;                                                   errmsg=errmsg,$
;                                                   no_wave=no_wave])   
;    
; Inputs      : std_name - standard internal CDS line name as created by
;                          ANYNAME2STD().
;
; Opt. Inputs : form  -  specifies option for format of the output.
;                        Default (form=0) is 'Fe 10 123.4'
;                                 form=1  is 'Fe X 123.4'
;                        (More later)
;               
; Outputs     : Function returns userversion of name or null string
;               if an error is detected.
;               
; Opt. Outputs: None
;               
; Keywords    : ERRMSG  - if this variable is defined on entry, any error
;                         messages will be returned in it.
;
;                         eg IDL> errmsg = ''
;                            IDL> un = stdname2user(std, errmsg=errmsg)
;                            ISL> if errmsg ne '' then print,errmsg
;
;                            ----> 'Input not in standard format'
;
;               NO_WAVE  - if present the wavelength field is suppressed.
;
; Calls       : None
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Util, string
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 6-Mar-95
;               
; Modified    : Handle '?' fields from CDS line list.  CDP, 7-Mar-95
;
; Version     : Version 2, 7-Mar-95
;-            

function stdname2user, std, form=form, no_wave=no_wave, errmsg=errmsg

;
;  Roman ionization states
;
ions = ['I','II','III','IV','V','VI','VII','VIII','IX','X','XI','XII',$
        'XIII','XIV','XV','XVI','XVII','XVIII','XIX','XX','XXI','XXII',$
        'XXIII','XXIV','XXV','XXVI']



;
;  check parameters
;
if n_params() eq 0 then begin
   print,"Use:  IDL> print,stdname2user('He_2_123.4')"
   print," prints -->  He II 123.4"
   return,''
endif

;
;  user format specified?
;
if n_elements(form) eq 0 then form = 0

   out = strarr(n_elements(std))

;
;  loop to cater for arrays
;
for num=0,n_elements(std)-1 do begin

;
;  check input format
;
;
;  break it up, 2 underscores?
;
   text = str_sep(std(num),'_')
   if n_elements(text) ne 3 then begin
      message = 'Error in standard input format (not 2 underscores)'
      if n_elements(errmsg) gt 0 then begin
         errmsg = message
         return,''
      endif else begin
         message,message
      endelse
   endif

;
;  handle individual bits
;
   e = text(0)
   i = text(1)
   if keyword_set(no_wave) then w = '' else w = text(2)

;
;  Roman or arabic?
;
   if form eq 1 and i ne '?' then i = ions(i-1)

;
;  make result
;
   out(num) = strtrim(e+' '+i+' '+w,2)

endfor

;
;  return
;
return,out

end
