;+
; Project     : SOHO - CDS     
;                   
; Name        : ANYTIM2TAI()
;               
; Purpose     : Converts any standard CDS time format to TAI.
;               
; Explanation : Tests the type of input and tries to use the appropriate
;               conversion routine to create the date/time in CDS TAI
;               time format.
;               
; Use         : IDL>  utc = anytim2tai(any_format)
;    
; Inputs      : any_format - date/time in any of the acceptable CDS 
;                            time formats -- for acceptable formats see file 
;                            aaareadme.txt.
;               
; Opt. Inputs : None
;               
; Outputs     : Function returns CDS TAI double precision time value.
;               
; Opt. Outputs: None
;               
; Keywords    : ERRMSG    = If defined and passed, then any error messages 
;                           will be returned to the user in this parameter 
;                           rather than being printed to the screen.  If no
;                           errors are encountered, then a null string is
;                           returned.  In order to use this feature, the 
;                           string ERRMSG must be defined first, e.g.,
;
;                                ERRMSG = ''
;                                UTC = ANYTIM2TAI ( DT, ERRMSG=ERRMSG, ...)
;                                IF ERRMSG NE '' THEN ...
;
; Calls       : DATATYPE, INT2UTC, STR2UTC
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Util, time
;               
; Prev. Hist. : Based on ANYTIM2UTC by C. D. Pike.
;
; Written     : William Thompson, GSFC, 20 May 1996
;               
; Modified    :	Version 1, William Thompson, GSFC, 20 May 1996
;
; Version     : Version 1, 20 May 1996
;-            

function anytim2tai, dt, errmsg=errmsg
 
;
;  set default return value
;
tai = 0.0D0

on_error, 2   ;  Return to the caller of this procedure if error occurs.
message=''    ;  Error message returned via ERRMSG if error is encountered.
;
;  see if any parameters were passed
;
if n_params() eq 0 then begin
	message = 'Syntax:  TAI = ANYTIM2TAI(DATE-TIME)'
	goto, handle_error
endif

;
;  determine type of input 
;
type = datatype(dt,1)

;
; see if the input is an array
;
np = n_elements(dt)
if np gt 1 then tai = replicate(tai, np)

;
; act accordingly
;
case type of
      'Double':  tai = dt
          else:  begin
		message=''
		tai = utc2tai(dt,errmsg=message)
		end		
endcase

if message eq '' then goto, finish
;
; Error handling point.
;
handle_error:
	if n_elements(errmsg) eq 0 then message, message
	errmsg = message
;
finish:
	return, tai
	end
