	FUNCTION OBT2TAI, OBT, ERRMSG=ERRMSG
;+
; Project     :	SOHO - CDS
;
; Name        :	OBT2TAI()
;
; Purpose     :	Converts OBT/LOBT to UTC
;
; Explanation :	This procedure takes the 6 byte (local) on-board time
;		from the spacecraft and converts it into a double precision
;		number containing the Atomic International Time (TAI) in
;		seconds from midnight, 1 January 1958.
;
; Use         :	Result = OBT2TAI( OBT )
;
; Inputs      :	OBT	= The on-board time, as downloaded from the spacecraft.
;			  This should be a byte array with the first dimension
;			  having six elements.
;
; Opt. Inputs :	None.
;
; Outputs     :	The result of the function will be a double precision number
;		containing the TAI time in seconds.  The result is a simple
;		conversion from the 6 byte satellite format to a double
;		precision time in seconds--no corrections are made to this
;		time.
;
; Opt. Outputs:	None.
;
; Keywords    :	ERRMSG	= If defined and passed, then any error messages 
;			  will be returned to the user in this parameter 
;			  rather than being handled by the IDL MESSAGE 
;			  utility.  If no errors are encountered, then a null 
;			  string is returned.  In order to use this feature, 
;			  the string ERRMSG must be defined first, e.g.,
;
;				ERRMSG = ''
;				RESULT = OBT2TAI( OBT, ERRMSG=ERRMSG )
;				IF ERRMSG NE '' THEN ...
;
; Calls       :	DATATYPE
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	If an error is encountered and the ERRMSG keyword is set, 
;		OBT2TAI returns a double precision value of -1.D0.
;
; Category    :	Utilities, Time.
;
; Prev. Hist. :	None.
;
; Written     :	William Thompson, GSFC, 12 September 1993.
;
; Modified    :	Version 1, William Thompson, GSFC, 12 September 1993.
;		Version 2, Donald G. Luttermoser, GSFC/ARC, 28 December 1994
;			Added the keyword ERRMSG.  Note that there are no
;			internally called procedures that use the ERRMSG 
;			keyword.
;		Version 3, Donald G. Luttermoser, GSFC/ARC, 30 January 1995
;			Made the error handling procedure more robust.  Note
;			that this routine can handle both scalars and vectors
;			as input.
;
; Version     :	Version 3, 30 January 1995.
;-
;
	ON_ERROR, 2  ; Return to the caller of this procedure if error occurs.
	MESSAGE=''   ; Error message that is returned if ERRMSG keyword set.
;
;  Check the input array.
;
	IF N_PARAMS() NE 1 THEN BEGIN
		MESSAGE = 'Syntax:  Result = OBT2TAI( OBT )'
	ENDIF ELSE BEGIN
		IF DATATYPE(OBT,1) NE 'Byte' THEN BEGIN
			MESSAGE = 'OBT must be a byte array.'
		ENDIF ELSE BEGIN
			SZ = SIZE(OBT)
			IF SZ(1) NE 6 THEN MESSAGE = $
			 'The first dimension of OBT must be 6.'
			IF SZ(0) EQ 0 THEN MESSAGE = 'OBT must be an array.'
		ENDELSE
	ENDELSE
	IF MESSAGE NE '' THEN GOTO, HANDLE_ERROR
;
;  Reformat OBT into a 6 by N array.
;
	OB = REFORM(OBT,6,N_ELEMENTS(OBT)/6)
;
;  Convert the on-board time into seconds since midnight, 1 January 1958.
;
	TAI = OB(0,*)*2.D0^24 + OB(1,*)*2.D0^16 + OB(2,*)*2.D0^8 + OB(3,*) + $
		OB(4,*)/2.D0^8 + (OB(5,*) AND 'E0'XB)/2.D0^16
;
;  If the original on-board time was a simple 6 byte array, then the result
;  should be a scalar.  Otherwise, the result should have the same
;  dimensionality as OBT, except for the first dimension of 6.
;
	IF SZ(0) EQ 1 THEN TAI = TAI(0) ELSE TAI = REFORM(TAI,SZ(2:SZ(0)))
;
	IF N_ELEMENTS(ERRMSG) NE 0 THEN ERRMSG = MESSAGE
	RETURN, TAI
;
;  Now handle any errors encountered.
HANDLE_ERROR:
	IF N_ELEMENTS(ERRMSG) EQ 0 THEN MESSAGE, MESSAGE
	ERRMSG = MESSAGE
	RETURN, -1.D0
;
	END
