	FUNCTION TAI2OBT, TAI, ERRMSG=ERRMSG
;+
; Project     :	SOHO - CDS
;
; Name        :	TAI2OBT()
;
; Purpose     :	Converts TAI to OBT/LOBT
;
; Explanation :	This procedure takes a double precision number containing the
;		Atomic International Time (TAI) in seconds from midnight, 1
;		January 1958, and converts it into the 6 byte (local) on-board
;		time from the spacecraft.
;
; Use         :	Result = TAI2OBT( TAI )
;
; Inputs      :	TAI = The time in TAI format as a double precision number or
;		      array.
;
; Opt. Inputs :	None.
;
; Outputs     :	The result of the function will be a byte array with the first
;		dimension having six elements.
;
; Opt. Outputs:	None.
;
; Keywords    :	ERRMSG	= If defined and passed, then any error messages 
;			  will be returned to the user in this parameter 
;			  rather than being handled by the IDL MESSAGE 
;			  utility.  If no errors are encountered, then a null 
;			  string is returned.  In order to use this feature, 
;			  the string ERRMSG must be defined first, e.g.,
;
;				ERRMSG = ''
;				RESULT = TAI2OBT( TAI, ERRMSG=ERRMSG )
;				IF ERRMSG NE '' THEN ...
;
; Calls       :	DATATYPE
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	If an error is encountered and the ERRMSG keyword is set, 
;		TAI2OBT returns a byte array of all zeros.
;
; Category    :	Utilities, Time.
;
; Prev. Hist. :	None.
;
; Written     :	William Thompson, GSFC, 6 July 1995
;
; Modified    :	Version 1, William Thompson, GSFC, 6 July 1995
;
; Version     :	Version 1, 6 July 1995
;-
;
	ON_ERROR, 2  ; Return to the caller of this procedure if error occurs.
	MESSAGE=''   ; Error message that is returned if ERRMSG keyword set.
;
;  Check the input array.
;
	IF N_PARAMS() NE 1 THEN BEGIN
		MESSAGE = 'Syntax:  Result = TAI2OBT( TAI )'
	ENDIF ELSE BEGIN
		IF DATATYPE(TAI,1) NE 'Double' THEN BEGIN
			MESSAGE = 'TAI must be a double precision array.'
		ENDIF ELSE SZ = SIZE(TAI)
	ENDELSE
	IF MESSAGE NE '' THEN GOTO, HANDLE_ERROR
;
;  Make an output array of the right size.
;
	OBT = BYTARR(6,N_ELEMENTS(TAI))
;
;  Calculate the first four bytes 
;
	SEC = LONG(TAI)
	REMAIN = TAI - SEC
	HOST_TO_IEEE, SEC
	OBT(0,0) = BYTE(SEC, 0, 4, N_ELEMENTS(SEC))
;
;  Calculate the fifth byte.
;
	REMAIN = REMAIN * 2.D0^8
	NEXT = BYTE(REMAIN)
	REMAIN = REMAIN - NEXT
	OBT(4,*) = NEXT
;
;  Calculate the sixth byte.
;
	REMAIN = REMAIN * 2.D0^8
	OBT(5,*) = BYTE(ROUND(REMAIN))
;
;  If the original on-board time was a scalar, then the result should be a one
;  dimensional array.  Otherwise, the result should have the same
;  dimensionality as TAI, except for the first dimension of 6.
;
	IF SZ(0) GT 1 THEN OBT = REFORM(OBT,[6,SZ(1:SZ(0))])
;
	IF N_ELEMENTS(ERRMSG) NE 0 THEN ERRMSG = MESSAGE
	RETURN, OBT
;
;  Now handle any errors encountered.
;
HANDLE_ERROR:
	IF N_ELEMENTS(ERRMSG) EQ 0 THEN MESSAGE, MESSAGE
	ERRMSG = MESSAGE
	RETURN, BYTARR(6)
;
	END
