;---------------------------------------------------------------------------
; Document name: get_cds_point.pro
; Created by:    Liyun Wang, GSFC/ARC, March 24, 1995
;
; Last Modified: Wed Mar 29 12:00:06 1995 (lwang@achilles.nascom.nasa.gov)
;---------------------------------------------------------------------------
;
FUNCTION get_cds_point, cds_detail
;+
; PROJECT:
;       SOHO - CDS
;
; NAME:
;       GET_CDS_POINT
;
; PURPOSE: 
;       Create a structure from the given CDS plan to be used in IMAGE_TOOL
;
; EXPLANATION:
;       In order for IMAGE_TOOL to handle the pointing(s) of a study,
;       the following information must be provided to IMAGE_TOOL:
;       Number of pointings needs to be done; for each pointing, width
;       and height of the pointing area (e.g., a raster). If there are
;       several rasters in the same pointing, the relation of the
;       rest of raster with respect to the first one (in CDS case,
;       determined by the POINTING value in study definition) has to
;       be given (can be, in CDS case, 0: fixed alreay, 1: same as the
;       first raster, or -1: offset from the first one). This routine
;       is to extract such information and put it into a structure to
;       be passed into IMAGE_TOOL.
;
; CALLING SEQUENCE: 
;       result = get_cds_point(cds_detail)
;
; INPUTS:
;       CDS_DETAIL - CDS plan structure containing the detailed science plan
;                    record. It contains the following tags:
;
;         STRUCT_TYPE - The character string 'CDS-DETAIL'
;         PROG_ID     - Program ID, linking one or more studies together
;         STUDY_ID    - Number defining the study
;         STUDYVAR    - Number giving the study variation ID
;         SCI_OBJ     - Science objective from the daily science meeting
;         SCI_SPEC    - Specific science objective from meeting
;         CMP_NO      - Campaign number
;         OBJECT      - Code for object planned to be observed
;         OBJ_ID      - Object identification
;         DATE_OBS    - Date/time of beginning of observation, in TAI format
;         DATE_END    - Date/time of end of observation, in TAI format
;         N_RASTERS1  - Variable number of rasters parameter
;         TIME_TAGGED - True (1) if the start of the study is to be a 
;                       time-tagged event.  Otherwise, the study will begin 
;                       immediately after the previous study.
;         TRACKING    - True (1) if feature tracking to be used, or false (0) 
;                       otherwise
;         N_POINTINGS - Number of pointings to apply to the study.
;         N_REPEAT_S  - Number of times to repeat study.
;         POINTINGS   - An array (with N_POINTINGS elements) of pointings
;                       to use with the study. POINTINGS is itself a
;                       structure of type PLAN_PNT_STRUC with the
;                       following tags: 
;
;                       INS_X  - Pointing in X relative to sun center
;                       INS_Y  - Pointing in Y relative to sun center
;
; OPTIONAL INPUTS: 
;       None.
;
; OUTPUTS:
;       RESULT - A pointing structure that can be used by IMAGE_TOOL. 
;                It has the following tags:
;
;          INSTRUME   - Code specifying the instrument; e.g., 'C' for CDS
;          G_LABEL    - Generic label for the pointing; e.g., 'RASTER'
;          X_LABEL    - Label for X coordinate of pointing; e.g., 'INS_X'
;          Y_LABEL    - Label for Y coordinate of pointing; e.g., 'INS_Y'
;          DATE_OBS   - Date/time of beginning of observation, in TAI format   
;          DO_POINTING- An integer of value 0 or 1 indicating whether pointing
;                       should be handled at the planning level (i.e., by
;                       IMAGE_TOOL)
;          N_POINTINGS- Number of pointings to be performed by IMAGE_TOOL
;          POINTINGS  - A structure array (with N_POINTINGS elements) of type
;                       "DETAIL_POINT" to be handled by IMAGE_TOOL. It has 
;                       the following tags: 
;
;                       POINT_ID - A string scalar for pointing ID
;                       INS_X    - X coordinate of pointing area center in arcs
;                       INS_Y    - Y coordinate of pointing area center in arcs
;                       WIDTH    - Area width (E/W extent)  in arcsec
;                       HEIGHT   - Area height (N/S extent) in arcsec
;                       OFF_LIMB - An interger with value 1 or 0 indicating 
;                                  whether or not the pointing area should
;                                  be off limb 
;
;          N_RASTERS  - Number of rasters for each pointing (this is
;                       irrelevant to the SUMER)
;          RASTERS    - A structure array (N_RASTERS-element) of type
;                       "RASTER_POINT" that contains raster size and pointing
;                       information (this is irrelevant to the SUMER). It has
;                       the following tags: 
;
;                       POINTING - Pointing handling code; valis
;                                  values are: 1, 0, and -1
;                       INS_X    - Together with INS_Y, the pointing to use
;                                  when user-supplied values are not
;                                  allowed.  Only valid when POINTING=0
;                                  (absolute) or POINTING=-1 (relative to
;                                  1st raster).
;                       INS_Y    - ...
;                       WIDTH    - Width (E/W extent) of the raster, in arcs
;                       HEIGHT   - Height (N/S extent) of the raster, in arcs
;
;      Note: For CDS case, pointings.width, pointings.height,
;            pointings.x_coord, and pointings.y_coord should match the first
;            raster's rasters.width, rasters.height, rasters.ins_x, and
;            rasters.ins_y, respectively.
;
; OPTIONAL OUTPUTS:
;       None.
;
; KEYWORD PARAMETERS: 
;       None.
;
; CALLS:
;       DATATYPE, GET_CDS_STUDY, GET_CDS_RASTER, MK_POINT_STC, RASTER_SIZE
;
; COMMON BLOCKS:
;       None.
;
; RESTRICTIONS: 
;       None.
;
; SIDE EFFECTS:
;       None.
;
; CATEGORY:
;       
; PREVIOUS HISTORY:
;       Written March 24, 1995, Liyun Wang, GSFC/ARC
;
; MODIFICATION HISTORY:
;       Version 1, created, Liyun Wang, GSFC/ARC, March 24, 1995
;
; VERSION:
;       Version 1, March 24, 1995
;-
;
   ON_ERROR, 2
   IF datatype(cds_detail) NE 'STC' THEN MESSAGE,'Invalid input data type. '+$
      'Reuqires a plan structure.'
   
   n_pointings = cds_detail.n_pointings
   
   IF n_pointings EQ 0 THEN n_pointings = 1

;---------------------------------------------------------------------------
;  From the plan structure, get the study from the DB
;---------------------------------------------------------------------------
   study = get_cds_study(cds_detail.study_id, cds_detail.studyvar)
   IF study.study_id LT 0 THEN MESSAGE, 'Study with ' + $
      'ID='+STRTRIM(cds_detail.study_id,2)+$
      ' and VAR='+STRTRIM(cds_detail.studyvar,2)+$
      ' not found in DB'
   n_rasters = N_ELEMENTS(study.rasters)
   dprint, 'n_rasters=', n_rasters
   
;---------------------------------------------------------------------------
;  Make an "empty" pointing structure based on number of pointings and rasters
;---------------------------------------------------------------------------
   mk_point_stc, point_stc, n_pointings=n_pointings, n_rasters=n_rasters
   
   dhelp,/str, point_stc
   
;---------------------------------------------------------------------------
;  Fill out some tags specific to CDS
;---------------------------------------------------------------------------
   point_stc.instrume = 'C'
   point_stc.g_label = 'INDEX'
   point_stc.x_label = 'INS_X'
   point_stc.y_label = 'INS_Y'
   point_stc.date_obs = cds_detail.date_obs
   
;---------------------------------------------------------------------------
;  Check to see if pointing needs to be handled by IMAGE_TOOL
;---------------------------------------------------------------------------
   IF study.var_point EQ 'N' OR cds_detail.n_pointings EQ 0 THEN $
      point_stc.do_pointing = 0
   
   FOR i = 0, n_rasters-1 DO BEGIN
      cds_raster = get_cds_raster(study.rasters(i).ras_id,$
                                  study.rasters(i).ras_var)
      raster_size, cds_raster, width = width, height = height
;---------------------------------------------------------------------------
;     Look for the target raster that appears first in the raster list and has
;     POINTING value being 1 (the pointing of this raster will be handled by
;     IMAGE_TOOL) 
;---------------------------------------------------------------------------
      IF N_ELEMENTS(idx) EQ 0 THEN BEGIN
         IF cds_raster.pointing EQ 1 THEN idx = i
      ENDIF
      point_stc.rasters(i).width = width
      point_stc.rasters(i).height = height
   ENDFOR
;---------------------------------------------------------------------------
;  If the target raster is not found, the first one in the list is assumed
;---------------------------------------------------------------------------
   IF N_ELEMENTS(idx) EQ 0 THEN idx = 0
   
   point_stc.rasters.ins_x = study.rasters.ins_x
   point_stc.rasters.ins_y = study.rasters.ins_y
   point_stc.rasters.pointing = study.rasters.pointing
   
;---------------------------------------------------------------------------
;  Assign the target raster's width and height to pointings.width and .height
;---------------------------------------------------------------------------
   FOR i = 0, n_pointings-1 DO BEGIN
      point_stc.pointings(i).point_id = '   '+STRTRIM(i,2)
      point_stc.pointings(i).width = point_stc.rasters(idx).width
      point_stc.pointings(i).height = point_stc.rasters(idx).height
   ENDFOR

   point_stc.pointings.ins_x = cds_detail.pointings.ins_x
   point_stc.pointings.ins_y = cds_detail.pointings.ins_y
   
   RETURN, point_stc
END

;---------------------------------------------------------------------------
; End of 'get_cds_point.pro'.
;---------------------------------------------------------------------------
