;+
; Project     :	SOHO - CDS
;
; Name        :	WIDG_HELP
;
; Purpose     :	Widget to select help topics.
;
; Explanation :	Create a widget that lists and displays the help topic from
;		which the user may select.
;
; Use         :	WIDG_HELP, FILENAME
;
; Inputs      :	FILENAME = The name of a file that contains the help
;			   information to display.  This program searches for a
;			   file with this name first in the current directory,
;			   and then in !PATH, and searches for the name by
;			   itself, and with '.hlp' appended after it.
;
;			   The file consists of a series of topic headers
;			   followed by a text message about that topic.  The
;			   topic headers are differentiated by starting with
;			   the "!" character in the first column, e.g.
;
;				!Overview
;				This is the overview for the
;				topic at hand.
;
;				!Button1
;				This is the help explanation
;				for button 1
;
;				etc.
;
;			   The program assumes that the first line in the file
;			   contains the first topic.  Also, there must be at
;			   least one line between any two topics.  Thus,
;
;				!Button2
;				!Button3
;
;			   is not allowed, but
;
;				!Button2
;				
;				!Button3
;
;			   is allowed.  The last topic in the file must have at
;			   least one non-topic line after it.
;
; Opt. Inputs :	None.
;
; Outputs     :	None.
;
; Opt. Outputs:	None.
;
; Keywords    :	GROUP_LEADER = The widget ID of a calling widget. Destroying
;			       that widget will kill this widget.
;
;		TITLE	     = The text to be displayed on the widget title
;			       bar.  If not passed, then "Widget Help" is used.
;               FONT         = Font name for the help text widget. If
;                              not passed, the text font is determined
;                              by GET_DFONT
;               SEP_CHAR     = Character used to differentiate topic
;                              headers. The default SEP_CHAR is '!'
;
; Calls       :	FIND_WITH_DEF, DATATYPE, GET_DFONT
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Utilities, Widget
;
; Prev. Hist. :	Based on PLANHELP by Elaine Einfalt, GSFC (HSTX), April 1994
;
; Written     :	William Thompson, GSFC, 2 September 1994
;
; Modified    :	Version 1, William Thompson, GSFC, 2 September 1994
;                          P. Brekke, GSFC, 21 September 1994
;                                     Text widget width = 60  (was 50)
;               Version 2, Liyun Wang, GSFC/ARC, April 3, 1995
;                          Added keywords FONT and SEP_CHAR
;                          Allowed commentary lines (starting with a ';'
;                             in the first column) in help text file
;
; Version     :	Version 2, April 3, 1995
;-
;

;==============================================================================
;			  The event handler routine.
;==============================================================================

	PRO WIDG_HELP_EVENT, EVENT
;
	ON_ERROR, 2
;
;  Get the event ID and the text window widget ID.
;
	WIDGET_CONTROL, EVENT.ID, GET_UVALUE=INPUT
	WIDGET_CONTROL, EVENT.TOP, GET_UVALUE=HELP
;
;  Process the widget event.  If the "Quit button was called, then exit this
;  procedure.
;
	CASE INPUT OF
		'CANCEL':  WIDGET_CONTROL, EVENT.TOP, /DESTROY
;
;  If one of the list items, then display the help information for the selected
;  topic.
;
		'LIST':  BEGIN
			I = EVENT.INDEX
			WIDGET_CONTROL, HELP.TEXT_WIDG, SET_VALUE=	$
				HELP.TEXT(HELP.STARTLINE(I):HELP.STOPLINE(I))
			END
		ELSE : PRINT,'WIDG_HELP - Programmer error ' + INPUT
	ENDCASE
;
	RETURN
	END

;==============================================================================
;			The widget definition routine.
;==============================================================================

PRO WIDG_HELP, FILENAME, GROUP_LEADER=GROUP, TITLE=TITLE, FONT=FONT,$
               SEP_CHAR=SEP_CHAR
;
;  Define the error status.
;
	ON_ERROR, 2
;
;  If already running return immediately.
;
	IF XREGISTERED('widg_help') THEN RETURN
;
;  Check the number of arguments.
;
	IF N_PARAMS() NE 1 THEN MESSAGE, 'Syntax:  WIDG_HELP, FILENAME'
;
;  Set font for text widget
;
        IF datatype(font) NE 'STR' THEN font = (get_dfont(wfont))(0)
;
;  See if SEP_CHAR is passed
;
        IF datatype(sep_char) NE 'STR' THEN sep_char = '!'
        comment = ';'
;
;  Open the help file.
;
	FILE = FIND_WITH_DEF(FILENAME,!PATH,'.hlp')
	IF FILE EQ '' THEN MESSAGE, 'Unable to open file '+FILE
	OPENR, UNIT, FILE, /GET_LUN
;
;  Read in the first topic header, and the first
;
	N_TOPICS = 1
	CUR_LINE = 0
	LINE = ''
;	TOPIC = ''
	TEXT = ''
        STARTLINE = 0
        WHILE (N_ELEMENTS(topic) EQ 0) DO BEGIN
           READF, UNIT, LINE
           char = STRMID(LINE,0,1) 
           IF char NE comment AND char EQ sep_char THEN BEGIN
              TOPIC = STRMID(LINE,1,STRLEN(LINE)-1)
              READF, UNIT, TEXT
           ENDIF
        ENDWHILE

; 	READF, UNIT, LINE
;         IF STRMID(LINE,0,1) NE comment THEN $
;            TOPIC = STRMID(LINE,1,STRLEN(LINE)-1)
; 	READF, UNIT, TEXT
;
;  Keep reading the file, and collect all the topics and text.
;
	WHILE NOT EOF(UNIT) DO BEGIN
           READF, UNIT, LINE
           IF STRMID(LINE,0,1) NE comment THEN BEGIN
              IF STRMID(LINE,0,1) EQ sep_char THEN BEGIN
                 IF N_TOPICS EQ 1 THEN STOPLINE = CUR_LINE ELSE	$
                    STOPLINE = [STOPLINE, CUR_LINE]
                 N_TOPICS = N_TOPICS + 1
                 TOPIC = [TOPIC, STRMID(LINE,1,STRLEN(LINE)-1)]
                 STARTLINE = [STARTLINE, CUR_LINE+1]
              END ELSE BEGIN
                 CUR_LINE = CUR_LINE + 1
                 TEXT = [TEXT, LINE]
              ENDELSE
           ENDIF
	ENDWHILE
	STOPLINE = [STOPLINE, CUR_LINE]
	FREE_LUN, UNIT
;
;  Create the base widget.
;
	IF N_ELEMENTS(TITLE) EQ 1 THEN WIDGET_TITLE = TITLE ELSE	$
		WIDGET_TITLE = 'Widget Help'
	BASE = WIDGET_BASE(TITLE=WIDGET_TITLE, /ROW, SPACE=20)
;
;  Create the quit button and the list of help topics.
;
	LEFT = WIDGET_BASE(BASE, /COLUMN, SPACE=15)	
	QUITIT = WIDGET_BUTTON(LEFT, VALUE='Quit', UVALUE='CANCEL')
        LIST = WIDGET_LIST(LEFT, VALUE=TOPIC, YSIZE=25, UVALUE='LIST')
;
;  Create the topic output window.
;
        TEXT_WIDG = WIDGET_TEXT(BASE, /SCROLL, /FRAME, XSIZE=60, YSIZE=30, $
                                font = font)
;
;  Realize the widget, and pass it a structure containing all the relevant
;  information.
;
	HELPPASS = {			$
		TEXT_WIDG: TEXT_WIDG,	$
		TOPIC: TOPIC,		$
		TEXT: TEXT,		$
		STARTLINE: STARTLINE,	$
		STOPLINE: STOPLINE}
	WIDGET_CONTROL, BASE, /REALIZE
	WIDGET_CONTROL, BASE, SET_UVALUE=HELPPASS
;
;  Start the widget.
;
	XMANAGER, 'widg_help', BASE, GROUP_LEADER=GROUP
;
	RETURN
	END
