;+
; Project     :	SOHO - CDS
;
; Pro Name    :	XMATRIX
;
; Purpose     : Widget display of fields within an arbitrary structure
;
; Use         : xmatrix,struct,wbase
;
; Inputs      :
;               STRUCT = input structure
;               WBASE = parent widget base into which place matrix
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;               NX = # of columns by which to arrange widgets (def=2)
;               WTAGS = text widget id's for each tag
;               TITLE = optional widget title
;               EDITABLE = make fields editable
; Procedure   :
;               Arranges structures in a matrix format with the tag
;               name in label widget and the tag value in a text widget.
;               If wbase and wtags exist, then widget is just updated
;               with input field values. This procedure is called by XSTRUCT
;
; Calls       : None.
;
; Common      : None.
;
; Restrictions:
;              Input must be a structure.
;              WBASE must be a valid parent base
;
; Side effects: None.
;
; Category    : Widgets
;
; Prev. Hist. : None.
;
; Written     :	Zarro (ARC/GSFC) 20 August 1994
;
; Modified    :
;       Version 2, Liyun Wang, GSFC/ARC, October 12, 1994
;          Made the WIDGET_LABEL right justified by using the 'fixed' font
;       Version 3, Zarro, (GSFC/ARC) 8 March, 1994
;          Added nested structure display
;
; VERSION:
;       Version 3
;-

PRO xmatrix,struct,wbase,nx=nx,wtags=wtags,title=title,editable=editable


   blank = '                                                              '
   ON_ERROR,1

   IF (N_ELEMENTS(struct) NE 1) OR (datatype(struct) NE 'STC') THEN $
      MESSAGE,'input must be a 1-d structure'

   IF N_ELEMENTS(wbase) EQ 0 THEN wbase=0
   IF NOT WIDGET_INFO(wbase,/valid) THEN MESSAGE,$
      'define a top level parent base first'

   IF N_ELEMENTS(wtags) EQ 0 THEN wtags=0l
   wtags=LONG(wtags)
   update=WIDGET_INFO(wtags(0),/valid)

   IF N_ELEMENTS(nx) EQ 0 THEN nx=2

;-- get tag definitions

   tags=tag_names(struct)
   stc_name=tag_names(struct,/structure_name)
   IF stc_name EQ '' THEN stc_name='ANONYMOUS'
   IF N_ELEMENTS(title) EQ 0 THEN title=stc_name
   ntags=N_ELEMENTS(tags)
   temp = ntags/nx
   IF (nx*FIX(temp) EQ ntags) THEN ny = temp ELSE ny=1+temp

;-- make label and text widgets

   IF (NOT update) THEN BEGIN
      wtags=lonarr(ntags)
      wtitle=WIDGET_LABEL(wbase,value=title,/frame)
      row=WIDGET_BASE(wbase,/row)
      i = -1 
      offset = 0
      FOR k=0,nx-1 DO BEGIN
         col=WIDGET_BASE(row,/column,/frame)
         IF (offset+ny) LT ntags THEN BEGIN
            tail = (offset+ny) 
            real_ny = ny
         ENDIF ELSE BEGIN
            tail = ntags
            real_ny = ntags-offset
         ENDELSE
         c_tags = tags(offset:tail-1)
         tag_len = STRLEN(c_tags)
         max_len = MAX(tag_len)
         FOR j=0, real_ny-1 DO BEGIN
            i=i+1
;            IF i GE ntags THEN GOTO,made
            temp=WIDGET_BASE(col,/row)
            c_tags(j) = c_tags(j)+':'+STRMID(blank,0,max_len-tag_len(j))
            label=WIDGET_LABEL(temp,value=c_tags(j),font='7x14bold')
            wtags(i)=WIDGET_TEXT(temp,value=' ',editable=editable)
         ENDFOR
         offset = offset+real_ny
      ENDFOR
   ENDIF

;-- populate text widgets

made:
   IF N_ELEMENTS(wtags) NE ntags THEN BEGIN
      MESSAGE,'incompatibility between input structure tags and '+$
         'wbase widget base',/CONTINUE
      RETURN
   ENDIF

   i=-1
   FOR k=0,nx-1 DO BEGIN
      FOR j=0,ny-1 DO BEGIN
         i=i+1
         IF i GE ntags THEN RETURN
         field=struct.(i)
         CASE 1 OF
            DATATYPE(field) eq 'STC': BEGIN
               widget_control,wtags(i),set_value='see over',sensitive=0
               xstruct,field,nx=nx,/just_reg,editable=editable
            END
            (SIZE(field))(0) GT 1 : BEGIN
               WIDGET_CONTROL,wtags(i),set_value='   >1D ARRAY  ',sensitive=0
            END
            ELSE: BEGIN
               IF datatype(field) EQ 'BYT' THEN field=FIX(field)
               WIDGET_CONTROL,wtags(i),set_value=STRING(field)
            END
         ENDCASE
      ENDFOR
   ENDFOR

RETURN & END
