;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Document name: xset_value.pro
; Created by:    Liyun Wang, GSFC/ARC, August 18, 1994
;
; Last Modified: Wed Mar 15 09:50:23 1995 (lwang@achilles.nascom.nasa.gov)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
PRO XSET_VALUE_EVENT, event
;
; Project     :	SOHO - CDS
;
; Pro Name    :	XSET_VALUE_EVENT
;
; Purpose     : Event handler of XSET_VALUE below
;
; Written     :	Liyun Wang, GSFC/ARC, August 19, 1994
;
;
   COMMON xset_value, value_sv

   ON_ERROR, 2

   wtype = WIDGET_INFO(event.id,/type)

;-- Get two widget IDs from the UVALUE of the top widget:
   WIDGET_CONTROL, event.top, get_uvalue = s_value

   WIDGET_CONTROL, event.id, get_uvalue = evt_num, get_value = cur_value
   CASE (evt_num) OF
      0: BEGIN                  ; Accepted
         WIDGET_CONTROL, s_value.sld_id, get_value = value
         IF (value NE  value_sv) THEN $
            value_sv = value
         WIDGET_CONTROL, event.top, /destroy
      END
      1: BEGIN                  ; Reset
         value = value_sv
         WIDGET_CONTROL, s_value.sld_id, set_value = value
         WIDGET_CONTROL, s_value.info_id, set_value = value
      END
      2: BEGIN                  ; Canceled, also restore original value
         value = value_sv
         WIDGET_CONTROL, event.top, /destroy
      END
      3: BEGIN                  ; Value obtained from the slider widget
         WIDGET_CONTROL, s_value.sld_id, get_value = value
         WIDGET_CONTROL, s_value.info_id, set_value = value
      END
      'KEYBOARD': BEGIN         ; Value entered from the keyboard
         WIDGET_CONTROL, s_value.info_id, set_value = cur_value(0)
         WIDGET_CONTROL, s_value.sld_id, set_value = cur_value(0)
      END
      ELSE: RETURN
   ENDCASE
END

PRO XSET_VALUE, value, min = min, max = max, title=title, group=group
;+
; Project     :	SOHO - CDS
;
; Name        :	XSET_VALUE
;
; Purpose     : Set the value of a variable interactively with a slider.
;
; Explanation :
;
; Use         : XSET_VALUE, value [, min = min] [, max = max]
;
; Inputs      : VALUE = Current default value
;
; Opt. Inputs : None.
;
; Outputs     : Also the input, but its value may have changed
;
; Opt. Outputs:
;
; Keywords    : MIN  -- Minimum value of the range (Default: 0)
;               MAX  -- Maximum value of the range (Default: 100)
;               TITLE -- Title of the widget; default: "Number Picker"
;               GROUP -- Group ID of an upper widget on which this one depends
;
; Calls       : BELL, DATATYPE
;
; Common      : XSET_VALUE
;
; Restrictions: Device must support widget
;
; Side effects: The input value is modified
;
; Category    : Utilities, Widget
;
; Prev. Hist. :
;
; Written     :	Liyun Wang, GSFC/ARC, August 18, 1994
;
; Modified    :
;
; Version     : 1.0
;-
;
   COMMON xset_value, value_sv
   ON_ERROR, 2
   IF (N_PARAMS() LT 1) THEN BEGIN
      PRINT, ' ' &  bell
      PRINT, 'XSET_VALUE -- You have to have a numerical number to change!'
      PRINT, ' '
      RETURN
   ENDIF

   dtype = datatype(value,2)
   IF ((dtype GE 6) OR (dtype LT 1)) THEN BEGIN
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  data not integer or (double precision) floating number
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      PRINT, ' ' &  bell
      PRINT, 'XSET_VALUE -- Wrong data type.'
      PRINT, ' '
      RETURN
   ENDIF

   value_sv = value
   IF N_ELEMENTS(title) EQ 0 THEN title = 'Number Picker'
;-- Create a top base
   base =  WIDGET_BASE (title = title, /column, space = 25)

   row1 = WIDGET_BASE(base, /row)
   temp1 = WIDGET_LABEL(row1,value = ' ')
   button_base = WIDGET_BASE (row1,space = 20, xpad = 30, /row, /frame)
   done        = WIDGET_BUTTON (button_base, value = 'Accept', uvalue = 0)
   resetting   = WIDGET_BUTTON (button_base, value = 'Reset', uvalue = 1)
   cancel      = WIDGET_BUTTON (button_base, value = 'Cancel', uvalue = 2)
   temp1 = WIDGET_LABEL(row1,value = ' ')

   base1  = WIDGET_BASE (base, /column); , ysize = 100)
   base2 = WIDGET_BASE(base1,/row)

   temp1 = WIDGET_LABEL(base2,value = ' ')
   info_id = cw_field(base2, title = 'Current Value', value = value, $
                      /row, xsize = 10,/RETURN, /frame,uvalue = 'KEYBOARD')
   temp1 = WIDGET_LABEL(base2,value = ' ')

   slider_y = 30
   IF (dtype LE 2) THEN BEGIN
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  value is a short integer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      IF (N_ELEMENTS(min) EQ 0) THEN min = 0
      IF (N_ELEMENTS(max) EQ 0) THEN max = 100
      sld_id =  WIDGET_SLIDER(base1, minimum = min, maximum = max, $
                              value = value, /frame, uvalue = 3, $
                              ysize = slider_y, /drag, /suppress)
   ENDIF ELSE IF (dtype EQ 3) THEN BEGIN
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  value is a long integer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      IF (N_ELEMENTS(min) EQ 0) THEN min = 0l
      IF (N_ELEMENTS(max) EQ 0) THEN max = 100l
      sld_id =  WIDGET_SLIDER(base1, minimum = min, maximum = max, $
                              value = value, /frame, uvalue = 3,$
                              ysize = slider_y, /drag, /suppress)
   ENDIF ELSE BEGIN
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  value is a floating point number (either single or double precsion)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
      IF (N_ELEMENTS(min) EQ 0) THEN min = 0.0
      IF (N_ELEMENTS(max) EQ 0) THEN max = 100.0
      cur_value = float(value)
      sld_id = CW_FSLIDER(base1, minimum = min, maximum = max, $
                          value = value, uvalue = 3,$
                          ysize = slider_y, /frame, /drag, /suppress)
   ENDELSE
   WIDGET_CONTROL, base, /realize

;-- Make a structure to pass some infos to the event handler:
   s_value = {sld_id:sld_id, info_id:info_id}

;-- Pass the structure s_value to the event handler via UVALUE of the top
;   base:
   WIDGET_CONTROL, base, set_uvalue = s_value
   
   XMANAGER, 'xset_value', base, group_leader = group, $
      modal = KEYWORD_SET(group)
   value = value_sv
END

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; End of 'xset_value.pro'.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
