;+
; Project     :	SOHO - CDS
;
; Name        :	XSTRUCT
;
; Purpose     : widget display of fields within an arbitrary structure
;
; Explanation :
;              Arranges structures in a matrix format with the tag
;              name in label widget and the tag value in a text widget.
;              If wbase and wtags exist, then widget is just updated
;              with input field values.
;
; Use         : xstruct,stc
;
; Inputs      : STC = structure name
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;              NX = # of columns by which to arrange widgets (def=2)
;              WTAGS = text widget id's for each tag
;              WBASE = parent widget base into which place structure
;              GROUP = event id of widget that calls XSTRUCT
;              JUST_REG = if set, then just register widget
;              TITLE = optional widget title
;              XOFF, YOFF = offset of structure base relative to calling widget
;              EDITABLE = make fields editable
;
; Calls       : xmatrix
;
; Common      : XSTRUCT
;
; Restrictions: 
;             Input must be a structure.
;             Cannot yet safely handle arrays of structures or nested structures
;
; Side effects: None.
;
; Category    : Widgets
;
; Prev. Hist. : None.
;
; Written     :	Zarro (ARC/GSFC) 20 August 1994
;
; Modified    :
;
; Version     : 1
;-

pro xstruct_event,  event                         ;event driver routine

on_error,1

common xstruct,caller,cwtags,wstruct

widget_control, event.id, get_uvalue = uservalue

if (n_elements(uservalue) eq 0) then uservalue=''
wtype=widget_info(event.id,/type)

;-- text events

if (wtype eq 3) and sexist(cwtags) then begin
 clook=where(cwtags eq event.id,cnt)
 sz=size(cwtags)
 if cnt gt 0 then begin
  two_d=0b
  if sz(0) eq 2 then begin
   two_d=1b
   ij=get_ij(clook(0),sz(1))
   i=ij(0) & j=ij(1)
   field=wstruct(j).(i)
  endif else begin
   index=clook(0)
   field=wstruct.(index)
  endelse
  widget_control,event.id,get_value=value
  value=strcompress(strtrim(value,2))
  if n_elements(field) eq 1 then value=value(0) else $
   value=str2arr(value(0),delim=' ')
  if datatype(field) ne 'STC' then begin
   if two_d then wstruct(j).(i)=value else wstruct.(index)=value
  endif
 endif
endif  

;-- button events

if wtype eq  1 then begin
 bname=strtrim(uservalue,2)

 case bname of 

  'close' : xkill,event.top,all=(1-sexist(caller))

  else:  return
 endcase
endif
return & end

;------------------------------------------------------------------------------

pro xstruct,struct,nx=nx,just_reg=just_reg,group=group,editable=editable,$
            wbase=wbase,wtags=wtags,title=title,xoff=xoff,yoff=yoff,modal=modal

on_error,1

common xstruct,caller,cwtags,wstruct

if not have_widgets() then message,'widgets unavailable'
if (datatype(struct) ne 'STC') then message,'input must be a structure'

editable=keyword_set(editable)
just_reg=keyword_set(just_reg)
modal=keyword_set(modal)

if (editable) and (not just_reg) then wstruct=struct
if (not just_reg) then delvarx,cwtags

if n_elements(xoff) ne 0 then xoff_sav=xoff
if n_elements(yoff) ne 0 then yoff_sav=yoff
just_reg_sav=just_reg

;-- trick to prevent XSTRUCT's XMANAGER from taking control

if sexist(group) then begin
 caller=group 
endif else delvarx,caller

if (sexist(caller) eq 0) and (not just_reg) and (not modal) then xkill,/all

nstruct=n_elements(struct)

for i=0,nstruct-1 do begin

;-- JUST_REG all bases (except last) when more that one structure

 if (nstruct gt 1) then begin
  just_reg=0

;-- get size and location of previous base

  if (i gt 0) then begin
   frac=.1
   widget_control,wbase,tlb_get_size=wsize,tlb_get_offset=woff
   xoff=woff(0)+frac*wsize(0)
   yoff=woff(1)+frac*wsize(1)
  endif
  update=0 & delvarx,wbase,wtags
  if i eq (nstruct-1) then begin
   if n_elements(just_reg_sav) ne 0 then just_reg=just_reg_sav else just_reg=0
  endif else just_reg=1
 endif

 if n_elements(wtags) eq 0 then wtags=0l
 wtags=long(wtags)
 update=xalive(wbase)

 if not update then begin  ;-- make top level base

  if nstruct gt 1 then begin
   wtitle='XSTRUCT_('+strtrim(string(i),2)+')' 
  endif else wtitle='XSTRUCT'

  wbase = widget_base(title =wtitle,/column)

;-- put CLOSE button in first row

  if (just_reg eq 0) then begin
   row=widget_base(wbase,/row,/frame)   
   cfont="-adobe-courier-bold-r-normal--25-180-100-100-m-150-iso8859-1"
   cfont=(get_dfont(cfont))(0)
   closeb=widget_button(row,value='Close',uvalue='close',/no_release,/frame,font=cfont)
  endif

 endif

;-- put matrix of tags in second row

 xmatrix,struct(i),wbase,nx=nx,wtags=wtags,title=title,editable=editable
 if editable then boost_array,cwtags,wtags

;-- realize and manage

 if (n_elements(xoff) eq 0) and (n_elements(yoff) eq 0) then begin
  if n_elements(group) ne 0 then begin
   widget_control,wbase,/realize,tlb_get_size=wsize,map=0,show=0
   widget_control,group,tlb_get_offset=goff,tlb_get_size=gsize
   xsize=((gsize(0)-wsize(0))/2) > 0.
   ysize=((gsize(1)-wsize(1))/2) > 0.
   xoff=(goff(0)+xsize)
   yoff=(goff(1)+ysize)
  endif
 endif

 if not update then begin
  if (n_elements(xoff) eq 0) and (n_elements(yoff) eq 0) then $
   widget_control,wbase,/realize else $
    widget_control,wbase,/realize,tlb_set_xoff=xoff,tlb_set_yoff=yoff
  xmanager,'xstruct',wbase,group=group,just_reg=just_reg,modal=modal
  if (not keyword_set(group)) and (not keyword_set(just_reg)) then xmanager
 endif else xshow,wbase
endfor

;-- restore initial parameters

if sexist(xoff_sav) then xoff=xoff_sav
if sexist(yoff_sav) then yoff=yoff_sav
if sexist(just_reg_sav) then just_reg=just_reg_sav
if sexist(caller) then group=caller

if sexist(wstruct) then struct=wstruct

return & end



