;+
;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;
;  widget_pfss_field.pro - a widget application for tracing and drawing the
;                          final field line image specified in the pfss data
;                          browser
;  usage:  widgetid=widget_pfss_field(parent)
;          where parent = widget id of top level base
;
;  M.DeRosa - 20 Feb 2003 - created
;             20 Mar 2003 - revamped a bit, added n_line field to widget
;             24 Mar 2003 - added notes button
;             25 Apr 2003 - added error checking for nlines field
;             25 Apr 2003 - added field line crossing toggle, draw open and/or
;                           closed line buttons
;              9 May 2003 - added ephemeris button, date label, save file
;                           dialog widget to allow the user to specify the 
;                           filename.  Also detects when new data has been
;                           loaded, so that DRAW_FIELD events now
;                           automatically retrace field lines when this
;                           happens 
;             12 May 2003 - changed pfss_field_start_coord call to agree with
;                           new calling sequence
;             30 May 2003 - changed some inaccuracies in the detailed notes
;                           section, as pointed out by Bart
;
;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
;-

pro widget_pfss_field_event,event

;  include common block
@pfss_data_block

;  trap window resize events: event.id is same as event.top since they are
;  events generated by the top-level-base widget
if event.id eq event.top then begin

  widget_control,event.top,get_uval=state,/no_copy
  xsiz=event.x<state.imxsiz
  ysiz=event.y<state.imysiz
  widget_control,state.wfield,xsiz=xsiz,ysiz=ysiz
  widget_control,event.top,set_uval=state,/no_copy

endif else begin

  ;  what happened?
  widget_control,event.id,get_uval=uval,/hourglass
  case uval of
    'CLOSE': widget_control,event.top,/destroy
    'EPHEM': begin
      widget_control,event.top,get_uval=state,/no_copy    
      widget_control,state.wmapbc,set_val=string(b0,f='(f7.3)')
      widget_control,state.wmaplc,set_val=string(l0,f='(f7.3)')
      widgetnow=state.pnow
      if widgetnow ne now then begin
        pfss_trace_field,/oneway
        state.pnow=now
      endif
      widget_control,event.top,set_uval=state,/no_copy    
      render_field,event.top
      end
    'DRAW_FIELD': begin
      widget_control,event.top,get_uval=state,/no_copy
      widgetnow=state.pnow
      if widgetnow ne now then begin
        pfss_trace_field,/oneway
        state.pnow=now
      endif
      widget_control,event.top,set_uval=state,/no_copy
      render_field,event.top
      end
    'TIFF': begin
      tvlct,re,gr,bl,/get
      filename=dialog_pickfile(group=event.top,/write)
      if filename ne '' then begin
        if strmid(filename,strlen(filename)-1,1) eq '/' then begin
          filename=filename+'pfss_image.tiff'
        endif else begin
          dotpos=strpos(filename,'.',/reverse_search)
          if dotpos eq -1 then begin
            filename=filename+'.tiff'
          endif else begin
            suffix=strmid(filename,dotpos,strlen(filename)-dotpos)
            if (suffix ne '.tif') and (suffix ne '.tiff') then $
              filename=filename+'.tiff'
          endelse
        endelse
        print,'  Writing '+filename
        write_tiff,filename,reverse(rimage,2),1,re=re,gr=gr,bl=bl
        result=dialog_message('Done saving TIFF file.',/info,$
          dialog_parent=event.top)
        endif else print,'  No image file written.'
      end
    'NEWMAG': begin
      widget_control,event.top,get_uval=state,/no_copy
      widget_control,state.wmapmag,get_val=val
      state.pmag=round(float(val(0)))
      widget_control,event.top,set_uval=state,/no_copy
      end
    'CROSS=ON': begin
      widget_control,event.top,get_uval=state,/no_copy
      state.pcross=uval
      widget_control,event.top,set_uval=state,/no_copy
      end
    'CROSS=OFF': begin
      widget_control,event.top,get_uval=state,/no_copy
      state.pcross=uval
      widget_control,event.top,set_uval=state,/no_copy
      end
    'DRAW_OPEN': begin
      widget_control,event.top,get_uval=state,/no_copy
      state.popen=1b xor state.popen
      widget_control,event.top,set_uval=state,/no_copy
      end
    'DRAW_CLOSED': begin
      widget_control,event.top,get_uval=state,/no_copy
      state.popen=2b xor state.popen
      widget_control,event.top,set_uval=state,/no_copy
      end

    'TRACE_FIELD': begin

      ;  get field line starting positions
      widget_control,event.top,get_uval=state,/no_copy
      widget_control,state.wparent,get_uval=pstate,/no_copy
      widget_control,state.wmapnlines,get_val=val
      val=round(float(val(0)))
      if val le 0 then begin
        result=dialog_message('Number of lines must be greater than 0',/err,$
          dialog_parent=event.top)
        widget_control,state.wmapnlines,set_val=string(state.pnlines,f='(i6)')
        widget_control,state.wparent,set_uval=pstate,/no_copy
        widget_control,event.top,set_uval=state,/no_copy
        return
      endif
      state.pnlines=val
      widget_control,state.wmapnlines,set_val=string(state.pnlines,f='(i6)')
      pfss_field_start_coord,3,state.pnlines,$
          bbox=[pstate.pl1,pstate.pb1,pstate.pl2,pstate.pb2]
      str(*)=rix(1)
      widget_control,state.wparent,set_uval=pstate,/no_copy
      widget_control,event.top,set_uval=state,/no_copy

      ;  do the field-line tracing and then render
      pfss_trace_field,/oneway
      render_field,event.top

      end
    'NOTES': begin
      print,'  Some detailed notes related to the field line rendering widget:'
      print,'  ---------------------------------------------------------------'
      print,'  Field line output variables:'
      print,'      The three arrays ptr,ptth,ptph contain the coordinates of'
      print,'      all N field lines that have been traced.  Since the field'
      print,'      lines can be of arbitrary length, the array nstep contains'
      print,'      the number of points needed to define the Nth field line.'
      print,'      Thus, field line i (where i is between 0 and N-1) is'
      print,'      represented by the points ptr(0:nstep(i)-1,i), and likewise'
      print,'      for ptth and ptph.'
      print,'  ---------------------------------------------------------------'
      print,'  (r,theta,phi) spherical/heliographic coordinate system:'
      print,'      r is the distance away from sun-center in units of solar'
      print,'      radii, such that valid values are between 1 (the nominal'
      print,'      photosphere and 2.5 (the radius of the source surface).'
      print,'      theta and phi are respectively the colatitude and'
      print,'      longitude in radians.'
      print,'  ---------------------------------------------------------------'
      print,'  Specifying field line starting points:'
      print,'      The widget is currently set up to pick field line starting'
      print,'      points at random within the box specified in the previewer'
      print,'      widget.  To have the tracer draw the field lines passing'
      print,'      through specific points anywhere within the coronal volume,'
      print,'      do the following: (a) if you haven''t done so already,'
      print,'      launch the rendering window, (b) on the command line,'
      print,'      define the str,stth,stph arrays to be coordinates of the'
      print,'      desired starting points, (c) type the command'
      print,'         IDL> pfss_trace_field'
      print,'      on the command line, and (d) hit the "redraw image" button'
      print,'      in the rendering window to display the field lines in the'
      print,'      widget.'
      print,'  ---------------------------------------------------------------'
      print,'  Coordinate transformations:'
      print,'      Once the lat/lon (B,L) of the subsolar point are known, it'
      print,'      is (reasonably) straightforward to convert from heliographic'
      print,'      coordinates (r,theta,phi) to normalized cartesian'
      print,'      coordinates (x,y,z):'
      print,'         x = r*sin(theta)*sin(phi-L)'
      print,'         y =   r*sin(theta)*cos(phi-L)*cos(B) + r*cos(theta)*sin(B)'
      print,'         z = - r*sin(theta)*cos(phi-L)*sin(B) + r*cos(theta)*cos(B)'
      print,'      Here,the (x,z)-plane defines the plane perpendicular to the'
      print,'      viewer (i.e. the image plane) with the x-axis positive'
      print,'      westward and the z-axis positive northward.  The y-axis is'
      print,'      along the line-of-sight and is positive inward (toward the'
      print,'      observer).  Note that (x,y,z) forms a *left*-handed'
      print,'      coordinate system.  The inverse transform is:'
      print,'         r     = sqrt(x^2+y^2+z^2)'
      print,'                         sqrt(x^2+(y*cos(B)+z*sin(B))^2)'
      print,'         theta = arccos( ------------------------------- )' 
      print,'                               -y*sin(B)+z*cos(B)'
      print,'                           sqrt(x^2+(y*cos(B)+z*sin(B))^2)'
      print,'         phi   = L+arcsin( -------------------------------- )'
      print,'                                 -y*sin(B)+z*cos(B)'
      print,'  ---------------------------------------------------------------'
      print,'  SSW Maps:'
      print,'      To generate a Dominic Zarro-style map for use within SSW,'
      print,'      issue the following commands:'
      print,'        IDL> pfss_draw_field,outim=outim,bcent=bcent,lcent=lcent,width=width,mag=mag'
      print,'        IDL> nax=size(outim,/dim)'
      print,'        IDL> map={data:outim,xc:0.0,yc:0.0,dx:2./nax(0),dy:2./nax(1),time:now,xunits:''normalized'',yunits:''normalized'',pfss_cent_l0:lcent,pfss_cent_b0:bcent,id:''PFSS field line map''}'
      print
      print,'      Before calling pfss_draw_field, the following keywords need'
      print,'      to be defined.  They correspond exactly to the fields in'
      print,'      the "PFSS field line renderer" widget:'
      print,'        bcent = map projection latitude in degrees'
      print,'        lcent = map projection Carrington longitude in degrees'
      print,'        width = width of map in units of solar diameters'
      print,'        mag   = magnification factor over nascent magnification'
      print,'                (currently 192x192 pixels for full sun)'
      print
      print,'      Note that the above map is in normalized coordinates, that' 
      print,'      is, disk center is at (0,0) and the limb has a radius of 1.'
      print,'      The image can be scaled to any measurement unit (such as'
      print,'      arcseconds) by multiplying dx and dy by the appropriate '
      print,'      conversion factor (e.g. rsun*width, where rsun is measured'
      print,'      in arcsec).'
      print,'  ---------------------------------------------------------------'
      end
    else: stop  ;  SHOULDN'T BE ABLE TO GET HERE !
  endcase
endelse

end

;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

pro render_field,top  ;  (re)draws the display

;  include common block
@pfss_data_block

;  get state
widget_control,top,get_uval=state,/no_copy

;  error checking on centroid
widget_control,state.wmaplc,get_val=curlon
widget_control,state.wmapbc,get_val=curlat
if ((curlon(0) gt 360) or (curlon(0) lt -360)) then begin
  result=dialog_message('Centroid L0 out of range.',/err,dialog_parent=top)
  widget_control,state.wmaplc,set_val=string(state.plc,f='(f7.3)')
  widget_control,top,set_uval=state,/no_copy
  return
endif
if ((curlat(0) gt 90) or (curlat(0) lt -90)) then begin
  result=dialog_message('Centroid B0 must be between -90 and +90.',$
    /err,dialog_parent=top)
  widget_control,state.wmapbc,set_val=string(state.pbc,f='(f7.3)')
  widget_control,top,set_uval=state,/no_copy
  return
endif

;  update state
state.plc=(curlon(0)+360.)mod 360
state.pbc=curlat(0)

;  get magnification
widget_control,state.wmapmag,get_val=val
state.pmag=round(float(val(0)))

;  get frame width
widget_control,state.wmapwidth,get_val=val
state.pwidth=float(val(0))>1.0  ;  renderer chokes if width<1

;  straighten out numbers in text boxes
widget_control,state.wmaplc,set_val=string(state.plc,f='(f7.3)')
widget_control,state.wmapbc,set_val=string(state.pbc,f='(f7.3)')
widget_control,state.wmapnlines,set_val=string(state.pnlines,f='(i6)')
widget_control,state.wmapmag,set_val=string(state.pmag,f='(i2)')
widget_control,state.wmapwidth,set_val=string(state.pwidth,f='(f5.2)')

;  do the rendering here, puts full image in rimage variable
wset,state.winfield
usedrawfield1=(state.pcross eq 'CROSS=OFF') or (state.popen eq 0b) or $
              (state.popen eq 2b)
if usedrawfield1 then begin
  pfss_draw_field,bcent=state.pbc,lcent=state.plc,$
    width=state.pwidth,imsc=400,mag=state.pmag,drawopen=state.popen and 1b, $
    drawclosed=state.popen and 2b
endif else begin
  pfss_draw_field2,bcent=state.pbc,lcent=state.plc,$
    width=state.pwidth,imsc=400,mag=state.pmag,drawopen=state.popen and 1b, $
    drawclosed=state.popen and 2b
endelse

;  resize if necessary, otherwise just redraw
nax=size(rimage,/dim)
if ((state.imxsiz ne nax(0)) or (state.imysiz ne nax(1))) then begin
  erase
  state.imxsiz=nax(0)
  state.imysiz=nax(1)
  xsiz=round(nlat*state.pmag*state.pwidth)<state.popensiz
  ysiz=round(nlat*state.pmag*state.pwidth)<state.popensiz
  widget_control,state.wfield,xsiz=xsiz,ysiz=ysiz, $
    draw_xsiz=state.imxsiz,draw_ysiz=state.imysiz
  widget_control,state.wfield, $
    set_draw_view=round([state.imxsiz-xsiz,state.imysiz-ysiz]/2)
endif

;  stick new image in widget
tv,rimage

;  update date label
widget_control,state.wdatelabel,set_val='Currently displaying data for '+now

;  get parent widget id (for wset command below)
widget_control,state.wparent,get_uval=pstate,/no_copy
win=state.winfield
wset,win
widget_control,state.wparent,set_uval=pstate,/no_copy

;  save state
widget_control,top,set_uval=state,/no_copy

end

;^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

function widget_pfss_field,wparent

;  include common block
@pfss_data_block

;  set defaults
mag=3l
width=1.5  ;  number of solar diameters
opensiz=640
nlines=100

;  construct widget
master=widget_base(/row,title='PFSS field line renderer',group=wparent,$
  /tlb_size_events)
optionbase=widget_base(master,/col)
cancelbut=widget_button(optionbase,val=' Close window ',uval='CLOSE')
savetiff=widget_button(optionbase,val=' Save as TIFF ',uval='TIFF')
redraw=widget_button(optionbase,val=' Redraw image ',uval='DRAW_FIELD')
notebut=widget_button(optionbase,val=' Detailed notes ',uval='NOTES')

;  projection centroid area
centroidbox=widget_base(optionbase,/col,frame=5)
centtitle=widget_label(centroidbox,val='Proj. cent. (deg):')
lbase=widget_base(centroidbox,/row,/align_center)
ltitle=widget_label(lbase,val='L0:')
ltext=widget_text(lbase,xsiz=7,/edit,uval='DRAW_FIELD',font='Courier')
bbase=widget_base(centroidbox,/row,/align_center)
btitle=widget_label(bbase,val='B0:')
btext=widget_text(bbase,xsiz=7,/edit,uval='DRAW_FIELD',font='Courier')
ephembutton=widget_button(centroidbox,val=' Use Earth L0,B0 ', $
  uval='EPHEM', /align_center)

;  magnification area
magbox=widget_base(optionbase,/col,frame=5)
magtitle=widget_label(magbox,val='Magnification:')
magbase=widget_base(magbox,/row,/align_center)
maglabel1=widget_label(magbase,val=' x')
magtext=widget_text(magbase,xsiz=2,/edit,uval='DRAW_FIELD', $
  val=string(mag,f='(i2)'),font='Courier')

;  width area
widthbox=widget_base(optionbase,/col,frame=5)
widthtitle1=widget_label(widthbox,val='Frame width:',/align_left)
widthbase=widget_base(widthbox,/row)
widthtext=widget_text(widthbase,xsiz=5,/edit,uval='DRAW_FIELD', $
  val=string(width,f='(f5.2)'),font='Courier')
widthlabel1=widget_label(widthbase,val='diameters')

;  nlines area
linebox=widget_base(optionbase,/col,frame=5)
linetitle=widget_label(linebox,val='Number of lines:')
linebase=widget_base(linebox,/row,/align_center)
linetext=widget_text(linebase,xsiz=6,/edit,uval='TRACE_FIELD', $
  val=string(nlines,f='(i6)'),font='Courier')
linebut=widget_button(linebox,val=' Retrace lines ',uval='TRACE_FIELD', $
  /align_center)

;  select which lines to draw
drawbox=widget_base(optionbase,/col,frame=5)
drawtitle1=widget_label(drawbox,val='Draw which lines?')
drawbase=widget_base(drawbox,/row,/nonexclusive)
drawopenbutton=widget_button(drawbase,val='Open ',uval='DRAW_OPEN')
drawclosedbutton=widget_button(drawbase,val='Closed',uval='DRAW_CLOSED')
widget_control,drawopenbutton,/set_button
widget_control,drawclosedbutton,/set_button

;  field line crossing toggle
crossbox=widget_base(optionbase,/col,frame=5)
crosstitle2=widget_label(crossbox,val='Display accurate')
crosstitle2=widget_label(crossbox,val='  line crossings?')
crossbase=widget_base(crossbox,/row,/exclusive)
docrossbut=widget_button(crossbase,val='Yes ',uval='CROSS=ON')
nocrossbut=widget_button(crossbase,val='No',uval='CROSS=OFF')
widget_control,nocrossbut,/set_button

;  drawing area
drawarea=widget_base(master,/col)
drawlabel=widget_label(drawarea,val='Currently displaying data for '+now)
draw=widget_draw(drawarea,retain=2,xsiz=nlat*mag*width,ysiz=nlat*mag*width,$
  x_scroll_size=round(nlat*mag*width)<opensiz,$
  y_scroll_size=round(nlat*mag*width)<opensiz)

;  realize
widget_control,master,/realize
widget_control,draw,get_val=windraw
xmanager,'widget_pfss_field',master,/no_block

;  grab projection centroid from previewer
widget_control,wparent,get_uval=pstate,/no_copy
lcent=pstate.plcent
bcent=pstate.pbcent
widget_control,ltext,set_val=string(pstate.plcent,f='(f7.3)')
widget_control,btext,set_val=string(pstate.pbcent,f='(f7.3)')
pstate.wfieldbase=master
widget_control,wparent,set_uval=pstate,/no_copy

;  save state structure (w=widget, win=window, p=parameter, im=image)
state={wparent:wparent, wfield:draw, wmaplc:ltext, wmapbc:btext, $
       wmapmag:magtext, wmapnlines:linetext, wmapwidth:widthtext, $
       wlinebut:linebut, wdrawbut:redraw, wdatelabel:drawlabel, $
       winfield:windraw, imxsiz:-1l, imysiz:-1l, $
       pmag:mag, pwidth:width, plc:lcent, pbc:bcent, pnlines:100l, $
       popensiz:opensiz,pcross:'CROSS=OFF',popen:3b,pnow:now}
widget_control,master,set_uval=state,/no_copy

;  generate 'TRACE_FIELD' event
widget_control,linetext,set_uval='TRACE_FIELD',send_event=$
  {ID:linebut,TOP:master,HANDLER:master,TYPE:4}

return,master

end

